/**
 * Copyright  2007 matsu@zuena.org.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.zuena.guiceex.jpa.transaction;


import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.zuena.guiceex.jpa.exception.PersistenceContextFieldNotFound;
import org.zuena.guiceex.transaction.TransactionAttributeType;
import org.zuena.guiceex.transaction.annotation.Transactional;

/**
 * \bhĂяoɃC^[ZvggUNVT[rX񋟂܂B<br/>
 * {@code @}{@link Transactional}֎w肳Ă{@link TransactionAttributeType}
 * ɉgUNVs܂B
 * ܂A{@link Object}X[p[NXNXTuNX֌p֌WHA
 * {@code @}{@link PersistenceContext}<b>ŏ̃tB[hɂ̂</b>{@link EntityManager}CX^X𒊓܂B
 * ׁ̈Apersistence-unit{@link EntityManager}CX^XAP{@link EntityManager}NCAg
 * ̃CX^Xň͂ł܂B<br/>
 * {@link EntityManager}́A{@code @}{@link PersistenceContext#unitName()}Ŏw肳ꂽjbgp̃CX^XA
 * Kvɉ{@code @}{@link PersistenceContext#properties()}gpč쐬AڃtB[hɑ΂Ē܂B
 * @auther matsu@zuena.org
 * @since 0.9.1
 */
public class TransactionInterceptor implements MethodInterceptor {

	private static final Log log = LogFactory.getLog(TransactionInterceptor.class);

	private final TransactionPropagator propagatorPrototype;
	
	public TransactionInterceptor(TransactionAttributeType type) {
		this.propagatorPrototype = TransactionPropagator.getTransactionPropagator(type);
	}
	/*
	 * (non-Javadoc)
	 * @see org.aopalliance.intercept.MethodInterceptor#invoke(org.aopalliance.intercept.MethodInvocation)
	 */
	public Object invoke(final MethodInvocation invocation) throws Throwable {
		
		TransactionPropagator propagator = propagatorPrototype.clone();
		Transactional transactional = invocation
										.getMethod()
										.getAnnotation(Transactional.class);
		if (log.isDebugEnabled()){
			log.debug("propargation=" + transactional.value() 
						+ "; rollbackFor=" + Arrays.toString(transactional.rollbackFor()));
		}
		final Object obj = invocation.getThis();
		final Field field = findPersistenceContextBindableField(obj);
		if(field == null){
			throw new PersistenceContextFieldNotFound();
		}
		PersistenceContext pc = field.getAnnotation(PersistenceContext.class);
		propagator.setPersistenceContext(pc);
		propagator.setRollbackFor(transactional.rollbackFor());
		return propagator.propagate(
				new TransactionPropagator.Process(){
					public Object execute(EntityManager manager) throws Throwable {
						if (field != null){
							field.setAccessible(true);
							field.set(obj, manager);
						}
						return invocation.proceed();
					}
				});
	}

	/**
	 * {@link Object}X[p[NXNXTuNX֌p֌WHA
	 * {@code @}{@link PersistenceContext}ŏ̃tB[h{@link EntityManager}
	 * 𒊓܂B
	 * 
	 */
	private Field findPersistenceContextBindableField(Object obj){

		List<Class> classes = new ArrayList<Class>();
		
		Class subclass = obj.getClass().getSuperclass(); 
		while(subclass != Object.class){
			classes.add(subclass);
			subclass = subclass.getSuperclass();
		}

		for(int i=classes.size()-1 ; i >= 0 ; i--){

			Field[] fields = classes.get(i).getDeclaredFields();
			
			for (Field field : fields) {
				if (log.isDebugEnabled()){
					log.debug(field.getName());
				}
				if (field.isAnnotationPresent(PersistenceContext.class)){
					return field;
				}
			}
		}
		return null;
	}
}
