﻿/*
Copyright (c) 2013, KAKUMOTO Masayuki
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of the outher KAKUMOTO Masayuki nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Reflection;
using System.Diagnostics;

namespace GustFront
{
    /// <summary>
    /// 例外クラス
    /// </summary>
    public class GustFrontException : Exception
    {
        public GustFrontException()
        {
        }

        public GustFrontException(string message)
            : base(message)
        {
        }

        public GustFrontException(string message, Exception innerException)
            : base(message, innerException)
        {
        }
    }

    /// <summary>
    /// Action.Render で使用するグラフィックスデバイス。
    /// 最上位のアクションが実際のオブジェクトを作成し、子アクションに伝播させる。
    /// 子アクションは渡されたものをキャストして描画する。
    /// </summary>
    public interface IGraphicsDevice
    {
    }

    /// <summary>
    /// 状態遷移と描画を行なうクラス。
    /// </summary>
    [DebuggerStepThrough()]
    public abstract class Action
    {
        /// <summary>
        /// アクションが登録解除される時に呼び出されるデリゲートの型。
        /// </summary>
        /// <param name="obj">解除されたアクション</param>
        public delegate void UnregisterActionCallback(Action obj);

        private UnregisterActionCallback unreg = null;
        private List<Action> registeredAction = new List<Action>(2);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="unreg">アクションが登録解除される時に呼び出されるデリゲート</param>
        protected Action(UnregisterActionCallback unreg)
        {
            this.unreg = unreg;
        }

        /// <summary>
        /// 子アクションが存在する場合 True を、それ以外の場合 False を返す。
        /// </summary>
        protected bool HasRegisteredAction
        {
            get { return registeredAction.Count > 0; }
        }

        /// <summary>
        /// 登録されている全ての子アクションを取得する。
        /// </summary>
        /// <returns>子アクションの配列</returns>
        protected Action[] GetRegisteredActions()
        {
            return registeredAction.ToArray();
        }

        /// <summary>
        /// 子アクションを登録する。
        /// </summary>
        /// <param name="obj">登録されるアクション</param>
        protected void RegisterAction(Action obj)
        {
            if (obj != null) {
                registeredAction.Add(obj);
            } else {
                throw new ArgumentNullException("obj");
            }
        }

        /// <summary>
        /// 子アクションを登録解除する。
        /// </summary>
        /// <param name="obj">登録解除されるアクション</param>
        protected void UnregisterAction(Action obj)
        {
            if (obj != null) {
                registeredAction.Remove(obj);
            } else {
                throw new ArgumentNullException("obj");
            }
        }

        /// <summary>
        /// このアクションを終了する。
        /// コンストラクタで渡されたデリゲートがあれば、このアクションを引数としてそれを呼び出す。
        /// </summary>
        public virtual void EndOfAction()
        {
            if (unreg != null) unreg(this);
        }

        /// <summary>
        /// 子アクションに描画を要求する。
        /// 描画は登録順に行なわれる。
        /// </summary>
        /// <param name="r">親アクションから伝播するグラフィックスデバイス</param>
        public virtual void Render(IGraphicsDevice r)
        {
            for (int i = 0; i < registeredAction.Count; i++)
                registeredAction[i].Render(r);
        }

        /// <summary>
        /// 子アクションに時間遷移を要求する。
        /// 途中でいずれかのアクションが登録解除されても影響しない。
        /// </summary>
        /// <param name="delta">前回の Tick の呼び出しからの経過時間</param>
        public virtual void Tick(TimeSpan delta)
        {
            if (HasRegisteredAction) {
                Action[] actions = GetRegisteredActions();
                for (int i = 0; i < actions.Length; i++)
                    actions[i].Tick(delta);
            }
        }
    }

    /// <summary>
    /// 独自のエレメントをサポートする。
    /// </summary>
    public interface IHasElement
    {
        /// <summary>
        /// 実装先でサポートされるエレメントを取得する。
        /// </summary>
        /// <returns>サポートされる全てのエレメントの型名の配列</returns>
        string[] GetTypeNames();
        /// <summary>
        /// 実際の型を取得する。
        /// </summary>
        /// <param name="typeName">型名</param>
        /// <returns>リフレクションに用いる Type のインスタンス</returns>
        Type GetType(string typeName);
        /// <summary>
        /// エレメントを型単位でストリームから読み取る。
        /// </summary>
        /// <param name="reader">ストリームから読み取るための TextReader のインスタンス</param>
        /// <param name="typeName">読み取るエレメントの型名</param>
        /// <returns>ストリームから読み取られた、対象型の全てのエレメント</returns>
        Element[] LoadElement(TextReader reader, string typeName);
        /// <summary>
        /// エレメントを型単位でストリームに書き込む。
        /// </summary>
        /// <param name="writer">ストリームに書き込むための TextWriter のインスタンス</param>
        /// <param name="typeName">書き込むエレメントの型名</param>
        void SaveElement(TextWriter writer, string typeName);
    }

    /// <summary>
    /// スクリプトの解析、プラグインの読み込みなどを行なう。
    /// スクリプトを使用するクラスは、このクラスを継承するべきである。
    /// </summary>
    public abstract class ScriptManager : Action
    {
        private Encoding myTextEncoding = Encoding.Default;
        private Dictionary<string, IHasElement> myPluginElement = new Dictionary<string, IHasElement>();
        private Dictionary<string, ModuleInfo> myModuleCache =
            CollectionUtil.CreateCaseInsensitiveDictionary<ModuleInfo>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="unreg">継承先から渡されるデリゲート</param>
        protected ScriptManager(UnregisterActionCallback unreg)
            : base(unreg)
        {
        }

        /// <summary>
        /// 設定ファイルやスクリプトのエンコーディング。
        /// ただし、必ずしも一貫している必要は無い。
        /// </summary>
        public Encoding TextEncoding { get { return myTextEncoding; } }

        /// <summary>
        /// 指定された型のエレメントを持つプラグインを取得する。
        /// </summary>
        /// <param name="typeName">型名</param>
        /// <returns>プラグイン内のクラスが実装する IHasElement のインスタンス</returns>
        public IHasElement GetPluginElementInterface(string typeName)
        {
            IHasElement result = null;
            myPluginElement.TryGetValue(typeName, out result);
            return result;
        }

        /// <summary>
        /// ファイルから読み取るためのストリームを取得する。
        /// ファイルは既に存在している必要がある。
        /// 特殊な読み込みをサポートするには派生クラスでオーバーライドすること。
        /// </summary>
        /// <param name="source">ファイルのパス</param>
        /// <returns>読み取り用の FileStream を返す</returns>
        public virtual Stream GetStreamToRead(string source)
        {
            return new FileStream(source, FileMode.Open, FileAccess.Read, FileShare.Read);
        }

        /// <summary>
        /// ファイルに書き込むためのストリームを取得する。
        /// ファイルは新規作成される。既に存在する場合は長さが 0 になる。
        /// 特殊な書き込みをサポートするには派生クラスでオーバーライドすること。
        /// </summary>
        /// <param name="destination">ファイルのパス</param>
        /// <returns>書き込み用の FileStream を返す</returns>
        public virtual Stream GetStreamToWrite(string destination)
        {
            return new FileStream(destination, FileMode.Create, FileAccess.Write, FileShare.None);
        }

        /// <summary>
        /// モジュールを解析する。
        /// </summary>
        /// <param name="source">モジュールのパス</param>
        /// <returns>解析されたモジュール。解析済みの場合は以前の結果が返る。</returns>
        public ModuleInfo ParseModule(string source)
        {
            ModuleInfo mi = null;
            if (!myModuleCache.TryGetValue(source, out mi)) {
                mi = ModuleInfo.Parse(GetStreamToRead, source, myTextEncoding);
                myModuleCache.Add(source, mi);
            }
            return mi;
        }

        /// <summary>
        /// 解析済みモジュールのキャッシュをクリアする。
        /// デバッグに用いる。
        /// </summary>
        public void ClearModuleCache()
        {
            myModuleCache.Clear();
        }

        /// <summary>
        /// 起動ルーチン。
        /// Environment.ini からエンコーディング情報を読み取り、プラグインの情報取得を行なう。
        /// デバイスの初期化などは派生クラスでオーバーライドして行なうこと。
        /// </summary>
        public virtual void Fire()
        {
            Element env = Element.Load(new StreamReader(GetStreamToRead("Environment.ini"), Encoding.ASCII),
                "Environment", typeof(UserElement))[0];
            myTextEncoding = Encoding.GetEncoding(env.GetData("Encoding").AsString());

            foreach (string pluginPath in Directory.GetFiles("Plugin", "*.dll")) {
                Assembly asm = Assembly.LoadFile(Path.GetFullPath(pluginPath));
                if (asm != null) {
                    List<string> command_sets = new List<string>();
                    foreach (Type t in asm.GetExportedTypes()) {
                        if (t.IsPublic && t.IsClass && !t.IsAbstract) {
                            if (t.GetInterface("IHasCommand") != null) {
                                command_sets.Add(t.FullName);
                            }
                            if (t.GetInterface("IHasElement") != null) {
                                IHasElement he = (IHasElement)asm.CreateInstance(t.FullName);
                                foreach (string typeName in he.GetTypeNames()) {
                                    if (!myPluginElement.ContainsKey(typeName)) {
                                        myPluginElement.Add(typeName, he);
                                    } else {
                                        throw new GustFrontException(typeName + " already exist in other plugin.");
                                    }
                                }
                            }
                        }
                    }
                    if (command_sets.Count > 0) ScriptEngine.AddCommandProcessor(asm, command_sets);
                }
            }
        }

        /// <summary>
        /// 終了ルーチン。
        /// 現在の実装では何もしない。
        /// 何かする場合は派生クラスでオーバーライドすること。
        /// </summary>
        public virtual void Extinguisher()
        {
        }
    }
}