﻿/*
Copyright (c) 2013, KAKUMOTO Masayuki
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of the outher KAKUMOTO Masayuki nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Diagnostics;

namespace GustFront
{
    /// <summary>
    /// アクションの結果の取得をサポートする。
    /// </summary>
    public interface IActionResultSender
    {
        /// <summary>
        /// アクションの結果を取得する。
        /// </summary>
        /// <returns>結果の値</returns>
        ScriptValue GetResult();
    }

    /// <summary>
    /// アクションの結果の設定をサポートする。
    /// 通常、スレッドの子アクションが IActionResultSender を実装し、
    /// スレッドの親アクションが IActionResultReceiver を実装する。
    /// </summary>
    public interface IActionResultReceiver
    {
        /// <summary>
        /// アクションの結果を設定する。
        /// </summary>
        /// <param name="state">結果の値</param>
        void SetResult(ScriptValue state);
    }

    /// <summary>
    /// コマンドのデリゲート
    /// </summary>
    /// <param name="params">コマンドに渡されるパラメータの配列</param>
    /// <returns>コマンドの結果</returns>
    public delegate ScriptValue CommandImpl(ScriptValue[] @params);

    /// <summary>
    /// 独自のコマンドをサポートする
    /// </summary>
    public interface IHasCommand
    {
        /// <summary>
        /// コマンドをどのスクリプトエンジンで実行するか指定する。
        /// </summary>
        /// <param name="obj">対象となるエンジン</param>
        void SetTarget(ScriptEngine obj);
        /// <summary>
        /// サポートされる全てのコマンドを取得する。
        /// </summary>
        /// <returns>コマンドの名前をキーとし、デリゲートを値とするディクショナリ</returns>
        IDictionary<string, CommandImpl> GetCommandMap();
    }

    /// <summary>
    /// メンバの値の取得および設定をサポートする。
    /// </summary>
    public interface IHasMember
    {
        /// <summary>
        /// メンバの値を取得する
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <param name="params">取得に必要なパラメータ</param>
        /// <param name="result">値を代入する変数</param>
        /// <returns>取得できた場合は True を、それ以外の場合は False を返す。</returns>
        bool GetMember(string name, ScriptValue[] @params, out ScriptValue result);
        /// <summary>
        /// メンバに値を設定する
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <param name="params">設定に必要なパラメータ、および設定する値</param>
        /// <returns>設定できた場合は True を、それ以外の場合は False を返す。</returns>
        bool SetMember(string name, ScriptValue[] @params);
    }

    /// <summary>
    /// スクリプトで使用する値。内部で保持されるオブジェクトの型は問わないが、
    /// 基本型の整数、実数、文字列との間では相互に変換可能である。
    /// 値が変数として使用される場合、その参照も保持する。
    /// </summary>
    public struct ScriptValue
    {
        private object v;
        internal object r;

        /// <summary>
        /// 他の ScriptValue の値を使って初期化する。
        /// </summary>
        /// <param name="value">値のコピー元の ScriptValue</param>
        public ScriptValue(ScriptValue value)
        {
            v = value.v;
            r = null;
        }
        /// <summary>
        /// Object を使って初期化する。
        /// </summary>
        /// <param name="value">初期値</param>
        public ScriptValue(object value)
        {
            v = value;
            r = null;
        }

        /// <summary>
        /// 値を取得する。設定はできない。
        /// </summary>
        public object Instance
        {
            get { return v; }
            internal set { v = value; }
        }
        /// <summary>
        /// 値の型。null の場合は null を返す。
        /// </summary>
        public Type Type
        {
            get { return (v != null) ? v.GetType() : null; }
        }

        public static implicit operator ScriptValue(double v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(float v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(decimal v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(long v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(int v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(short v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(byte v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(bool v)
        {
            return new ScriptValue((object)v);
        }
        public static implicit operator ScriptValue(string v)
        {
            return new ScriptValue((object)v);
        }

        public static explicit operator double(ScriptValue value)
        {
            return value.AsDouble();
        }
        public static explicit operator float(ScriptValue value)
        {
            return value.AsSingle();
        }
        public static explicit operator decimal(ScriptValue value)
        {
            return value.AsDecimal();
        }
        public static explicit operator long(ScriptValue value)
        {
            return value.AsInt64();
        }
        public static explicit operator int(ScriptValue value)
        {
            return value.AsInt32();
        }
        public static explicit operator short(ScriptValue value)
        {
            return value.AsInt16();
        }
        public static explicit operator byte(ScriptValue value)
        {
            return value.AsByte();
        }
        public static explicit operator bool(ScriptValue value)
        {
            return value.AsBoolean();
        }
        public static explicit operator string(ScriptValue value)
        {
            return value.AsString();
        }

        public double AsDouble()
        {
            return Convert.ToDouble(v);
        }
        public float AsSingle()
        {
            return Convert.ToSingle(v);
        }
        public decimal AsDecimal()
        {
            return Convert.ToDecimal(v);
        }
        public long AsInt64()
        {
            return Convert.ToInt64(v);
        }
        public int AsInt32()
        {
            return Convert.ToInt32(v);
        }
        public short AsInt16()
        {
            return Convert.ToInt16(v);
        }
        public byte AsByte()
        {
            return Convert.ToByte(v);
        }
        public bool AsBoolean()
        {
            return Convert.ToBoolean(v);
        }
        public string AsString()
        {
            return Convert.ToString(v);
        }

        /// <summary>
        /// 内部の値のハッシュ値を返す。つまり Instance.GetHashCode() と等価。
        /// </summary>
        /// <returns>この ScriptValue のハッシュ値</returns>
        public override int GetHashCode()
        {
            return v.GetHashCode();
        }

        /// <summary>
        /// obj を ScriptValue とみなして、その Instance と自らの Instance の等価性を調べる。
        /// </summary>
        /// <param name="obj">比較される ScriptValue</param>
        /// <returns>等価の場合 True を、それ以外の場合 False を返す。</returns>
        public override bool Equals(object obj)
        {
            return v.Equals(((ScriptValue)obj).v);
        }

        /// <summary>
        /// obj の Instance と自らの Instance の等価性を調べる。
        /// </summary>
        /// <param name="sv">比較される ScriptValue</param>
        /// <returns>等価の場合 True を、それ以外の場合 False を返す。</returns>
        public bool Equals(ScriptValue sv)
        {
            return v.Equals(sv.v);
        }

        /// <summary>
        /// 内部の値の文字列表現を返す。つまり Instance.ToString() と等価。
        /// </summary>
        /// <returns>この ScriptValue を表す文字列</returns>
        public override string ToString()
        {
            return v.ToString();
        }
    }

    /// <summary>
    /// スクリプトエンジン
    /// </summary>
    public abstract class ScriptEngine : Action
    {
        internal class VarRef
        {
            public ScriptValue[] ptr;
            public int index;
        }

        /// <summary>
        /// 値が変数かどうか調べる。
        /// </summary>
        /// <param name="sv">対象となる ScriptValue</param>
        /// <returns>変数の場合 True を、それ以外の場合 False を返す。</returns>
        public static bool IsVarRef(ScriptValue sv)
        {
            return sv.r is VarRef;
        }
        /// <summary>
        /// 変数に値を設定する。
        /// </summary>
        /// <param name="varRef">IsVarRef(varRef) が True となる値。</param>
        /// <param name="value">設定する値</param>
        public static void SetToVariable(ScriptValue varRef, object value)
        {
            if (IsVarRef(varRef)) {
                VarRef vr = (VarRef)varRef.r;
                vr.ptr[vr.index].Instance = value;
            } else {
                throw new ArgumentException("変数の参照ではありません。");
            }
        }

        /// <summary>
        /// スレッドの状態を管理する
        /// </summary>
        protected internal sealed class ActionThreadInfo
        {
            [DebuggerStepThrough()]
            private sealed class ProcedureState : Action, IActionResultReceiver, IActionResultSender
            {
                private ActionThreadInfo myThread;
                private RuntimeProcedureInfo myRPI;

                public ProcedureState(ActionThreadInfo ati, RuntimeProcedureInfo rpi)
                    : base(ati.UnregisterAction)
                {
                    myThread = ati;
                    myRPI = rpi;
                }

                public RuntimeProcedureInfo RPI { get { return myRPI; } }

                public NativeCommandInfo GetInstruction(int index)
                {
                    return myRPI.info[index];
                }

                public override void Tick(TimeSpan delta)
                {
                    myThread.Process.Engine.ProcessScript(myThread);
                }

                public void SetResult(ScriptValue state)
                {
                    myThread.Push(state);
                }

                public ScriptValue GetResult()
                {
                    return myThread.Pop();
                }
            }

            private class CallStackInfo
            {
                public string Name;
                public string File;
                public int Line;
            }

            private ActionProcessInfo myProcess;
            private ScriptValue? myExitCode;
            private bool myRunning = true;
            private ScriptValue[] myMemory = new ScriptValue[65536];
            private Stack<Action> myAction = new Stack<Action>();
            private Queue<ScriptValue[]> myRequestedEvent = new Queue<ScriptValue[]>();
            private int myInstructionPointer;
            private int myStackPointer;
            private int myBasePointer;
            private Stack<CallStackInfo> myCallStack = new Stack<CallStackInfo>();
            private ScriptValue myOnException;

            internal void SetCurrentLineNumber(int number)
            {
                myCallStack.Peek().Line = number;
            }

            internal ActionThreadInfo(ActionProcessInfo process, RuntimeProcedureInfo proc, params ScriptValue[] args)
            {
                myProcess = process;

                for (int i = 0; i < @args.Length; i++) {
                    @args[i].r = null;
                }

                EnterProcedure(proc, args);
            }

            internal ActionThreadInfo(ActionProcessInfo process, string name, Action obj)
            {
                myProcess = process;
                PushCallee(name);
                RegisterAction(obj);
            }

            /// <summary>
            /// スレッドが所属するプロセス
            /// </summary>
            public ActionProcessInfo Process
            {
                [DebuggerStepThrough()]
                get { return myProcess; }
            }

            /// <summary>
            /// スレッドの終了コード
            /// </summary>
            public ScriptValue? ExitCode
            {
                [DebuggerStepThrough()]
                get { return myExitCode; }
            }

            internal bool Running
            {
                [DebuggerStepThrough()]
                get { return myRunning; }
                [DebuggerStepThrough()]
                set { myRunning = value; }
            }

            internal void Push(ScriptValue value)
            {
                if (myStackPointer < myMemory.Length) {
                    myMemory[myStackPointer++] = value;
                } else {
                    throw new GustFrontException("内部スタックがオーバーフローしました。");
                }
            }

            internal ScriptValue Pop()
            {
                return myMemory[--myStackPointer];
            }

            internal void PushCallee(string name)
            {
                CallStackInfo csi = new CallStackInfo();
                int sep_index = name.IndexOf("@");
                if (sep_index != -1) {
                    csi.Name = name.Substring(0, sep_index);
                    csi.File = name.Substring(sep_index + 1);
                    csi.Line = 0;
                } else {
                    csi.Name = name;
                    csi.File = null;
                    csi.Line = 0;
                }
                myCallStack.Push(csi);
            }

            internal void PopCallee()
            {
                myCallStack.Pop();
            }

            internal ScriptValue SetExceptionHandler(ScriptValue name)
            {
                if (myProcess.Engine.IsValidProcedure(name)) {
                    ScriptValue prev = myOnException;
                    myOnException = name;
                    return prev;
                } else {
                    throw new ArgumentException("プロシージャが必要です。");
                }
            }

            internal void Terminate(Exception ex)
            {
                foreach (Action a in myAction) {
                    if (a is IDisposable) ((IDisposable)a).Dispose();
                }
                myAction.Clear();
                foreach (ScriptValue sv in myMemory) {
                    if (sv.Instance is IDisposable) ((IDisposable)sv.Instance).Dispose();
                }
                Array.Clear(myMemory, 0, myMemory.Length);
                myRequestedEvent.Clear();
                myBasePointer = 0;
                myStackPointer = 0;
                myInstructionPointer = 0;

                if (ex != null && myOnException.Instance != null) {
                    List<string> cs = new List<string>(myCallStack.Count);
                    foreach (CallStackInfo csi in myCallStack) {
                        if (csi.File != null) {
                            cs.Add(String.Format("{0}({1}) in {2}",
                                csi.Name, csi.Line, csi.File));
                        } else {
                            cs.Add(csi.Name);
                        }
                    }
                    Stack<string> msg = new Stack<string>();
                    msg.Push(ex.Message);
                    Exception innerEx = ex.InnerException;
                    while (innerEx != null) {
                        msg.Push(innerEx.Message);
                        innerEx = innerEx.InnerException;
                    }
                    EnterProcedure((RuntimeProcedureInfo)myOnException.Instance, new ScriptValue[] {
					new ScriptValue(CollectionUtil.CreateStdList(msg)),
					new ScriptValue(CollectionUtil.CreateStdList(cs))
				});
                } else {
                    myCallStack.Clear();
                    myRunning = false;
                }
            }

            internal Action CurrentAction
            {
                [DebuggerStepThrough()]
                get { return myAction.Peek(); }
            }

            internal RuntimeProcedureInfo CurrentProcedureInfo
            {
                [DebuggerStepThrough()]
                get { return CurrentProcedure.RPI; }
            }

            private ProcedureState CurrentProcedure
            {
                [DebuggerStepThrough()]
                get { return (ProcedureState)CurrentAction; }
            }

            internal void Render(IGraphicsDevice r)
            {
                Action[] actions = myAction.ToArray();
                Array.Reverse(actions);
                foreach (Action a in actions) {
                    a.Render(r);
                }
            }

            [DebuggerStepThrough()]
            internal void Tick(TimeSpan delta)
            {
                do {
                    CurrentAction.Tick(delta);
                } while (myAction.Count > 0 && myAction.Peek() is ProcedureState);
            }

            [DebuggerStepThrough()]
            internal void GoNext(int jumpTo)
            {
                myInstructionPointer = jumpTo;
            }

            internal NativeCommandInfo[] GetNextCommand()
            {
                if (myAction.Count > 0 && myAction.Peek() is ActionEventProc) {
                    ((ActionEventProc)CurrentAction).Tick(TimeSpan.Zero);
                }

                if (myAction.Count > 0 && myAction.Peek() is ProcedureState) {
                    List<NativeCommandInfo> ncis = new List<NativeCommandInfo>(16);
                    NativeCommandInfo inst = null;
                    do {
                        inst = CurrentProcedure.GetInstruction(myInstructionPointer++);
                        ncis.Add(inst);
                    } while (!inst.CP && ncis.Count <= 16);
                    return ncis.ToArray();
                } else {
                    return null;
                }
            }

            internal ScriptValue FastGetLocalValue(int index)
            {
                ScriptValue mem = myMemory[myBasePointer + index];
                VarRef vr = (VarRef)mem.r;
                if (vr != null){
                    return vr.ptr[vr.index];
                } else {
                    return mem;
                }
            }

            internal void FastSetLocalValue(int index, ScriptValue sv)
            {
                ScriptValue mem = myMemory[myBasePointer + index];
                VarRef vr = (VarRef)mem.r;
                if (vr != null) {
                    vr.ptr[vr.index].Instance = sv.Instance;
                } else {
                    mem.Instance = sv.Instance;
                }
            }

            internal void RegisterAction(Action obj)
            {
                myAction.Push(obj);
            }

            internal void UnregisterAction(Action obj)
            {
                if (obj != myAction.Pop()) {
                    throw new ArgumentException("obj is not last action.", "obj");
                }
                PopCallee();

                ScriptValue retval;
                if (obj is IActionResultSender) {
                    retval = ((IActionResultSender)obj).GetResult();
                } else {
                    retval = String.Empty;
                }

                if (myAction.Count > 0) {
                    if (myAction.Peek() is IActionResultReceiver) {
                        ((IActionResultReceiver)myAction.Peek()).SetResult(retval);
                    }
                } else {
                    if (obj is ProcedureState) {
                        Push(retval);
                    } else {
                        myExitCode = null;
                        myProcess.RemoveThread(this);
                    }
                }
            }

            internal ScriptValue[] GetArguments()
            {
                int argc = myMemory[myBasePointer - 3].AsInt32();
                ScriptValue[] argv = new ScriptValue[argc];
                for (int i = 0; i < argc; i++) {
                    argv[i] = myMemory[myBasePointer - 4 - i];
                }
                return argv;
            }

            private void EnterProcedure(RuntimeProcedureInfo rpi, int argc)
            {
                for (int i = 0; i < argc; i++) {
                    VarRef vr = new VarRef();
                    vr.ptr = myMemory;
                    vr.index = myStackPointer - i;
                    if (myMemory[vr.index].r == null) {
                        myMemory[vr.index].r = vr;
                    }
                }
                Push(argc);
                Push(myInstructionPointer);
                Push(myBasePointer);
                myBasePointer = myStackPointer;
                myInstructionPointer = 0;
                for (int i = 0; i < rpi.local; i++) {
                    VarRef vr = new VarRef();
                    vr.ptr = myMemory;
                    vr.index = myStackPointer + i;
                    myMemory[vr.index].r = vr;
                }

                myStackPointer += rpi.local;

                PushCallee(String.Concat(rpi.code.Name, "@", rpi.code.Module.FilePath));
                RegisterAction(new ProcedureState(this, rpi));
            }

            internal void EnterProcedure(RuntimeProcedureInfo rpi, ScriptValue[] args)
            {
                for (int i = args.Length - 1; i >= 0; i--) {
                    Push(args[i]);
                }
                EnterProcedure(rpi, args.Length);
            }

            internal void EnterProcedureFromScript(RuntimeProcedureInfo rpi, int argc)
            {
                EnterProcedure(rpi, argc);
            }

            internal bool ExitProcedure(ScriptValue retval)
            {
                myStackPointer = myBasePointer;
                myBasePointer = Pop().AsInt32();
                myInstructionPointer = Pop().AsInt32();
                int argc = Pop().AsInt32();
                myStackPointer -= argc;

                Push(retval);

                ProcedureState proc = CurrentProcedure;
                proc.EndOfAction();
                if (myAction.Count > 0) {
                    return true;
                } else {
                    myExitCode = proc.GetResult();
                    myProcess.RemoveThread(this);
                    return false;
                }
            }

            internal void Raise(RuntimeProcedureInfo eventProc, params ScriptValue[] args)
            {
                ScriptValue[] evp = new ScriptValue[args.Length + 1];
                evp[0] = new ScriptValue(eventProc);
                for (int i = 0; i < args.Length; i++) {
                    evp[i + 1].Instance = args[i].Instance;
                }
                myRequestedEvent.Enqueue(evp);
            }

            internal void Idle(ScriptValue onIdle)
            {
                if (onIdle.Instance == null || myProcess.Engine.IsValidProcedure(onIdle)) {
                    RegisterAction(new ActionEventProc(this.UnregisterAction, this, onIdle));
                } else {
                    throw new GustFrontException("プロシージャが必要です。");
                }
            }

            internal void Quit()
            {
                myRequestedEvent.Enqueue(null);
            }

            private class ActionEventProc : Action
            {
                private ActionThreadInfo myThread;
                private ScriptValue myOnIdle;

                public ActionEventProc(UnregisterActionCallback unreg, ActionThreadInfo ati, ScriptValue onIdle)
                    : base(unreg)
                {
                    myThread = ati;
                    myOnIdle = onIdle;
                }

                public override void Tick(TimeSpan delta)
                {
                    if (myThread.myRequestedEvent.Count > 0) {
                        ScriptValue[] evp = myThread.myRequestedEvent.Dequeue();
                        if (evp != null) {
                            myThread.EnterProcedure((RuntimeProcedureInfo)evp[0].Instance, CollectionUtil.ToArray(evp, 1));
                            if (delta != TimeSpan.Zero) {
                                myThread.Tick(delta);
                            }
                        } else {
                            EndOfAction();
                        }
                    } else {
                        if (myOnIdle.Instance != null) {
                            myThread.EnterProcedure((RuntimeProcedureInfo)myOnIdle.Instance, new ScriptValue[0]);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// プロセスの状態を管理する
        /// </summary>
        protected internal sealed class ActionProcessInfo
        {
            private ScriptEngine myEngine;
            private List<ActionThreadInfo> myThreads = new List<ActionThreadInfo>(4);
            private Dictionary<string, List<RuntimeProcedureInfo>> myEventHandlers = CollectionUtil.CreateCaseInsensitiveDictionary<List<RuntimeProcedureInfo>>(64);
            private ScriptValue? myExitCode;
            private ScriptValue[] myStorage;

            internal ScriptValue FastGetValue(int index)
            {
                return myStorage[index];
            }

            internal void FastSetValue(int index, ScriptValue value)
            {
                myStorage[index].Instance = value.Instance;
            }

            private Dictionary<string, RuntimeProcedureInfo> myRPIMap;
            internal RuntimeProcedureInfo GetRPI(string name)
            {
                RuntimeProcedureInfo rpi = null;
                myRPIMap.TryGetValue(name, out rpi);
                return rpi;
            }

            internal ActionProcessInfo(ScriptEngine eng, ModuleInfo script, params ScriptValue[] args)
            {
                myEngine = eng;
                int storage_size = 0;
                myRPIMap = eng.MapModuleToRuntimeSpace(script, out storage_size);
                myStorage = new ScriptValue[storage_size];
                for (int i = 0; i < myStorage.Length; i++) {
                    VarRef vr = new VarRef();
                    vr.ptr = myStorage;
                    vr.index = i;
                    myStorage[vr.index].r = vr;
                }

                ActionThreadInfo ati = AddThread(new ScriptValue(myRPIMap[
                    String.Concat(script.GetHashCode(), ".", ModuleInfo.MainRoutine)]), args);
                foreach (KeyValuePair<string, RuntimeProcedureInfo> kv in myRPIMap) {
                    if (kv.Key.StartsWith("!")) {
                        ati.EnterProcedure(kv.Value, new ScriptValue[0]);
                    }
                }
            }

            /// <summary>
            /// プロセスが所属するスクリプトエンジン
            /// </summary>
            public ScriptEngine Engine
            {
                get { return myEngine; }
            }

            internal ActionThreadInfo[] GetThreads()
            {
                return myThreads.ToArray();
            }

            internal bool HasThread
            {
                get { return myThreads.Count > 0; }
            }

            private ActionThreadInfo PrimaryThread
            {
                get { return myThreads[0]; }
            }

            /// <summary>
            /// プロセスの終了コード
            /// </summary>
            public ScriptValue? ExitCode
            {
                get { return myExitCode; }
            }

            internal ActionThreadInfo AddThread(ScriptValue proc, params ScriptValue[] args)
            {
                if (myEngine.IsValidProcedure(proc)) {
                    ActionThreadInfo obj = new ActionThreadInfo(this, (RuntimeProcedureInfo)proc.Instance, args);
                    myThreads.Add(obj);
                    return obj;
                } else {
                    throw new GustFrontException("プロシージャが必要です。");
                }
            }

            internal void RemoveThread(ActionThreadInfo ati)
            {
                if (ati == PrimaryThread) {
                    foreach (ActionThreadInfo ti in myThreads) {
                        ti.Terminate(null);
                    }
                    myThreads.Clear();
                    myExitCode = (ati.ExitCode.HasValue) ? ati.ExitCode : null;
                    myEngine.EndProcess(this);
                } else {
                    ati.Terminate(null);
                    myThreads.Remove(ati);
                }
            }

            internal void RegisterEventHandler(string eventName, ScriptValue proc)
            {
                if (myEngine.IsValidProcedure(proc)) {
                    List<RuntimeProcedureInfo> theList;
                    if (!myEventHandlers.TryGetValue(eventName, out theList)) {
                        theList = new List<RuntimeProcedureInfo>(2);
                        myEventHandlers.Add(eventName, theList);
                    }
                    theList.Add((RuntimeProcedureInfo)proc.Instance);
                } else {
                    throw new GustFrontException("プロシージャが必要です。");
                }
            }

            internal void UnregisterEventHandler(string eventName, ScriptValue proc)
            {
                if (myEngine.IsValidProcedure(proc)) {
                    List<RuntimeProcedureInfo> handlerList;
                    if (myEventHandlers.TryGetValue(eventName, out handlerList)) {
                        handlerList.Remove((RuntimeProcedureInfo)proc.Instance);
                    }
                } else {
                    throw new GustFrontException("プロシージャが必要です。");
                }
            }

            internal bool CallEventHandler(string eventName, params ScriptValue[] args)
            {
                if (eventName != null) {
                    List<RuntimeProcedureInfo> handlers = null;
                    if (myEventHandlers.TryGetValue(eventName, out handlers)) {
                        foreach (RuntimeProcedureInfo handler in handlers) {
                            PrimaryThread.Raise(handler, args);
                        }
                        return true;
                    } else {
                        return false;
                    }
                } else {
                    PrimaryThread.Quit();
                    return true;
                }
            }

            internal void RegisterAction(Action obj)
            {
                myThreads.Add(new ActionThreadInfo(this, String.Empty, obj));
            }
        }

        private ScriptManager myScriptManager;
        private List<ActionProcessInfo> myProcesses = new List<ActionProcessInfo>(4);
        private List<int> mySignals = new List<int>(64);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="unreg">スクリプトエンジンが終了する時に呼び出されるデリゲート</param>
        /// <param name="script_manager">スクリプトエンジンが所属するスクリプトマネージャ</param>
        protected ScriptEngine(UnregisterActionCallback unreg, ScriptManager script_manager)
            : base(unreg)
        {
            myScriptManager = script_manager;
        }

        /// <summary>
        /// プロセスを開始する。
        /// </summary>
        /// <param name="script">Main プロシージャを含むモジュール</param>
        /// <param name="args">Main に渡すパラメータ</param>
        /// <returns>開始されたプロセス</returns>
        protected ActionProcessInfo BeginProcess(ModuleInfo script, ScriptValue[] args)
        {
            ActionProcessInfo api = new ActionProcessInfo(this, script, args);
            myProcesses.Add(api);

            return api;
        }

        /// <summary>
        /// プロセスを終了する。
        /// 終了時に特殊な処理を行なう場合、派生クラスでオーバーライドすること。
        /// </summary>
        /// <param name="api">終了されるプロセス</param>
        protected virtual void EndProcess(ActionProcessInfo api)
        {
            myProcesses.Remove(api);
            if (myProcesses.Count == 0) {
                EndOfAction();
            }
        }

        /// <summary>
        /// プロセスが存在する場合 True を、存在しない場合 False を返す。
        /// </summary>
        protected bool HasProcess
        {
            get { return myProcesses.Count > 0; }
        }

        private class ActionWaiter : Action, IActionResultSender
        {
            private ScriptEngine myEngine;
            private Nullable<int> mySignal;
            private TimeSpan myDuration;
            private TimeSpan myTimeout;
            private bool mySignaled = true;

            public ActionWaiter(UnregisterActionCallback unreg, ScriptEngine eng, Nullable<int> signal, TimeSpan timeout)
                : base(unreg)
            {
                myEngine = eng;
                mySignal = signal;
                myTimeout = timeout;
            }

            public override void Tick(TimeSpan delta)
            {
                myDuration += delta;

                if (myDuration < myTimeout) {
                    if (mySignal.HasValue && myEngine.mySignals.Contains(mySignal.Value)) {
                        EndOfAction();
                    }
                } else {
                    mySignaled = !mySignal.HasValue;
                    EndOfAction();
                }
            }

            public ScriptValue GetResult()
            {
                return mySignaled;
            }
        }

        /// <summary>
        /// イベントに関連付けられたプロシージャ（イベントハンドラ）を呼び出す。
        /// </summary>
        /// <param name="eventName">イベントの名前</param>
        /// <param name="args">プロシージャに渡すパラメータ</param>
        /// <returns>ハンドラが 1 つ以上呼び出された場合は True を、それ以外の場合 False を返す。</returns>
        protected bool CallEventHandler(string eventName, params ScriptValue[] args)
        {
            bool called = false;
            foreach (ActionProcessInfo process in myProcesses) {
                called = called | process.CallEventHandler(eventName, args);
            }
            return called;
        }

        /// <summary>
        /// 非同期コマンドの結果を取得する。
        /// </summary>
        /// <param name="async">非同期コマンド</param>
        /// <param name="result">結果を設定する変数</param>
        /// <returns>コマンドが終了していれば Treu を、それ以外の場合 False を返す。</returns>
        protected static bool GetAsyncCalledCommandResult(ScriptValue async, out ScriptValue result)
        {
            KeyValuePair<IAsyncResult, CommandImpl> arci =
                (KeyValuePair<IAsyncResult, CommandImpl>)async.Instance;
            if (arci.Key.IsCompleted) {
                result = arci.Value.EndInvoke(arci.Key);
                return true;
            } else {
                result = null;
                return false;
            }
        }

        #region "cmd impl"

        private ScriptValue cmdGetProcedure(ScriptValue[] @params)
        {
            ModuleInfo current_mod = CurrentThread.CurrentProcedureInfo.code.Module;
            string proc_name = @params[0].AsString();
            int dot = proc_name.IndexOf('.');
            if (dot != -1) {
                RuntimeProcedureInfo rpi = CurrentThread.Process.GetRPI(String.Concat(
                    current_mod.ModRootNs[proc_name.Substring(0, dot)].GetHashCode(),
                    ".", proc_name.Substring(dot + 1)));
                if (rpi != null) {
                    return new ScriptValue(rpi);
                } else {
                    throw new GustFrontException("プロシージャを特定できません。");
                }
            } else {
                RuntimeProcedureInfo rpi = CurrentThread.Process.GetRPI(
                    String.Concat(current_mod.GetHashCode(), ".", proc_name));
                if (rpi == null) {
                    foreach (ModuleInfo mod in current_mod.GetReferences()) {
                        RuntimeProcedureInfo _rpi = CurrentThread.Process.GetRPI(
                            String.Concat(mod.GetHashCode(), ".", proc_name));
                        if (_rpi != null) {
                            if (rpi == null) {
                                rpi = _rpi;
                            } else {
                                throw new GustFrontException("プロシージャを特定できません。");
                            }
                        }
                    }
                    if (rpi == null) {
                        throw new GustFrontException("プロシージャを特定できません。");
                    }
                }
                return new ScriptValue(rpi);
            }
        }

        private ScriptValue cmdGetArguments(ScriptValue[] @params)
        {
            return cmdCreateList(CurrentThread.GetArguments());
        }

        private ScriptValue cmdCall(ScriptValue[] @params)
        {
            CallScript(@params[0], CollectionUtil.ToArray(@params, 1));
            CurrentThread.PopCallee();
            CurrentThread.PopCallee();
            CurrentThread.PushCallee(((RuntimeProcedureInfo)@params[0].Instance).code.Name);
            return null;
        }

        private ScriptValue cmdPreloadScript(ScriptValue[] @params)
        {
            try {
                myScriptManager.ParseModule(@params[0].AsString());
                return null;
            } catch (Exception ex) {
                IList msgs = CollectionUtil.CreateStdList(ex.Message.Split(new char[] { }, StringSplitOptions.RemoveEmptyEntries));
                return new ScriptValue(msgs);
            }
        }

        private ScriptValue cmdClearScriptCache(ScriptValue[] @params)
        {
            myScriptManager.ClearModuleCache();
            return null;
        }

        private ScriptValue cmdBeginProcess(ScriptValue[] @params)
        {
            return new ScriptValue(myTargetThread.Process.Engine.BeginProcess(
                myScriptManager.ParseModule(@params[0].AsString()), CollectionUtil.ToArray(@params, 1)));
        }

        private ScriptValue cmdRunThread(ScriptValue[] @params)
        {
            return new ScriptValue(myTargetThread.Process.AddThread(
                @params[0], CollectionUtil.ToArray(@params, 1)));
        }

        private ScriptValue cmdSuspendThread(ScriptValue[] @params)
        {
            ((ActionThreadInfo)@params[0].Instance).Running = false;
            return null;
        }

        private ScriptValue cmdResumeThread(ScriptValue[] @params)
        {
            ((ActionThreadInfo)@params[0].Instance).Running = true;
            return null;
        }

        private ScriptValue cmdSleep(ScriptValue[] @params)
        {
            RegisterThreadAction(new ActionWaiter(myTargetThread.UnregisterAction, this, null, TimeSpan.FromMilliseconds(@params[0].AsInt32())));
            return null;
        }

        private ScriptValue cmdIdle(ScriptValue[] @params)
        {
            if (@params.Length > 0) {
                myTargetThread.Idle(@params[0]);
            } else {
                myTargetThread.Idle(null);
            }
            return null;
        }

        private ScriptValue cmdQuit(ScriptValue[] @params)
        {
            myTargetThread.Process.CallEventHandler(null);
            return null;
        }

        private ScriptValue cmdRegisterEventHandler(ScriptValue[] @params)
        {
            myTargetThread.Process.RegisterEventHandler(@params[0].AsString(), @params[1]);
            return null;
        }

        private ScriptValue cmdUnregisterEventHandler(ScriptValue[] @params)
        {
            myTargetThread.Process.UnregisterEventHandler(@params[0].AsString(), @params[1]);
            return null;
        }

        private ScriptValue cmdRaiseEvent(ScriptValue[] @params)
        {
            return CallEventHandler(@params[0].AsString(), CollectionUtil.ToArray(@params, 1));
        }

        private ScriptValue cmdSendSignal(ScriptValue[] @params)
        {
            int hash = @params[0].AsString().GetHashCode();
            if (!mySignals.Contains(hash)) {
                mySignals.Add(hash);
            }
            return null;
        }

        private ScriptValue cmdResetSignal(ScriptValue[] @params)
        {
            int hash = @params[0].AsString().GetHashCode();
            if (mySignals.Contains(hash)) {
                mySignals.Remove(hash);
            }
            return null;
        }

        private ScriptValue cmdWaitForSignal(ScriptValue[] @params)
        {
            int hash = @params[0].AsString().GetHashCode();
            if (!mySignals.Contains(hash)) {
                TimeSpan timeout = TimeSpan.MaxValue;
                if (@params.Length > 1) {
                    timeout = TimeSpan.FromMilliseconds(@params[1].AsInt32());
                }
                RegisterThreadAction(new ActionWaiter(myTargetThread.UnregisterAction, this, hash, timeout));
                return null;
            } else {
                return true;
            }
        }

        private ScriptValue cmdGetMember(ScriptValue[] @params)
        {
            ScriptValue result;
            ScriptValue obj = @params[0];
            string name = @params[1].AsString();
            if (obj.Instance is IHasMember) {
                if (((IHasMember)obj.Instance).GetMember(name, CollectionUtil.ToArray(@params, 2), out result)) {
                    return result;
                } else {
                    throw new GustFrontException("メソッドまたは取得メンバ '" + name + "' は、このオブジェクトには存在しません。");
                }
            } else {
                object[] args = new object[@params.Length - 2];
                for (int i = 2; i < @params.Length; i++) {
                    args[i - 2] = @params[i].Instance;
                }
                if (!(obj.Instance is Type)) {
                    if (name == "Default") name = String.Empty;
                    result = new ScriptValue(obj.Instance.GetType().InvokeMember(name,
                        BindingFlags.InvokeMethod | BindingFlags.GetProperty | BindingFlags.GetField,
                        null, obj.Instance, args));
                } else {
                    result = new ScriptValue(((Type)obj.Instance).InvokeMember(name,
                        BindingFlags.InvokeMethod | BindingFlags.GetProperty | BindingFlags.GetField,
                        null, null, args));
                }
                for (int i = 2; i < @params.Length; i++) {
                    if (IsVarRef(@params[i])) SetToVariable(@params[i], args[i - 2]);
                }
                return result;
            }
        }

        private ScriptValue cmdSetMember(ScriptValue[] @params)
        {
            ScriptValue obj = @params[0];
            string name = @params[1].AsString();
            if (obj.Instance is IHasMember) {
                if (((IHasMember)obj.Instance).SetMember(name, CollectionUtil.ToArray(@params, 2))) {
                    return null;
                } else {
                    throw new GustFrontException("設定メンバ '" + name + "' は、このオブジェクトには存在しません。");
                }
            } else {
                object[] args = new object[@params.Length - 2];
                for (int i = 2; i < @params.Length; i++) {
                    args[i - 2] = @params[i].Instance;
                }
                if (!(obj.Instance is Type)) {
                    if (name == "Default") name = String.Empty;
                    obj.Instance.GetType().InvokeMember(name,
                        BindingFlags.SetProperty | BindingFlags.SetField, null, obj.Instance, args);
                } else {
                    ((Type)obj.Instance).InvokeMember(name,
                        BindingFlags.SetProperty | BindingFlags.SetField, null, null, args);
                }
                for (int i = 2; i < @params.Length; i++) {
                    if (IsVarRef(@params[i])) SetToVariable(@params[i], args[i - 2]);
                }
                return null;
            }
        }

        private ScriptValue cmdCreateInstance(ScriptValue[] @params)
        {
            object[] args = new object[@params.Length - 1];
            for (int i = 1; i < @params.Length; i++) {
                args[i - 1] = @params[i].Instance;
            }
            ScriptValue result = new ScriptValue(((Type)@params[0].Instance).InvokeMember(
                null, BindingFlags.CreateInstance, null, null, args));
            for (int i = 1; i < @params.Length; i++) {
                if (IsVarRef(@params[i])) SetToVariable(@params[i], args[i - 1]);
            }
            return result;
        }

        private ScriptValue cmdCreateList(ScriptValue[] @params)
        {
            int count = @params.Length;
            if (count > 0) {
                IList obj = CollectionUtil.CreateStdList(count);
                for (int i = 0; i < count; i++) {
                    obj.Add(@params[i].Instance);
                }
                return new ScriptValue(obj);
            } else {
                return new ScriptValue(CollectionUtil.CreateStdList());
            }
        }

        private ScriptValue cmdCreateArrayList(ScriptValue[] @params)
        {
            int length = @params[0].AsInt32();
            IList obj = CollectionUtil.CreateStdList(length);
            for (int i = 0; i < length; i++) {
                obj.Add(null);
            }
            return new ScriptValue(obj);
        }

        private ScriptValue cmdCombineList(ScriptValue[] @params)
        {
            IList result = CollectionUtil.CreateStdList();
            for (int i = 0; i < @params.Length; i++) {
                IList elm = (IList)@params[i].Instance;
                for (int j = 0; j < elm.Count; j++) {
                    result.Add(elm[j]);
                }
            }
            return new ScriptValue(result);
        }

        private class StringEqualityComparer : IEqualityComparer
        {
            private StringComparer sc = null;

            public StringEqualityComparer(bool ignoreCase)
            {
                if (ignoreCase) {
                    sc = StringComparer.InvariantCultureIgnoreCase;
                } else {
                    sc = StringComparer.InvariantCulture;
                }
            }

            bool IEqualityComparer.Equals(object x, object y)
            {
                return sc.Compare(Convert.ToString(x), Convert.ToString(y)) == 0;
            }

            int IEqualityComparer.GetHashCode(object obj)
            {
                return sc.GetHashCode(Convert.ToString(obj));
            }
        }

        private ScriptValue cmdCreateDictionary(ScriptValue[] @params)
        {
            if (@params.Length > 0) {
                return new ScriptValue(CollectionUtil.CreateStdDictionary(
                    new StringEqualityComparer(@params[0].AsBoolean())));
            } else {
                return new ScriptValue(CollectionUtil.CreateStdDictionary());
            }
        }

        private ScriptValue cmdDuplicateDictionary(ScriptValue[] @params)
        {
            if (@params.Length > 1) {
                return new ScriptValue(CollectionUtil.CreateStdDictionary(
                    (IDictionary)@params[0].Instance, new StringEqualityComparer(@params[1].AsBoolean())));
            } else {
                return new ScriptValue(CollectionUtil.CreateStdDictionary(
                    (IDictionary)@params[0].Instance));
            }
        }

        private ScriptValue cmdFormat(ScriptValue[] @params)
        {
            object[] args = new object[@params.Length];
            args[0] = String.Empty;
            for (int i = 1; i < @params.Length; i++) {
                args[i] = @params[i].Instance;
            }
            return String.Format(@params[0].AsString(), args);
        }

        private ScriptValue cmdTrace(ScriptValue[] @params)
        {
            string[] text = new string[@params.Length];
            for (int i = 0; i < text.Length; i++) {
                text[i] = @params[i].AsString();
            }
            Util.AppendToLog(text);
            return null;
        }

        private ScriptValue cmdGetClock(ScriptValue[] @params)
        {
            return Stopwatch.GetTimestamp() * 10000000L /
                Stopwatch.Frequency;
        }

        private ScriptValue cmdGetElement(ScriptValue[] @params)
        {
            return new ScriptValue(Element.GetElement(@params[0].AsString()));
        }

        private ScriptValue cmdGetElements(ScriptValue[] @params)
        {
            return new ScriptValue(CollectionUtil.CreateStdList(Element.GetElements(@params[0].AsString())));
        }

        private ScriptValue cmdGetType(ScriptValue[] @params)
        {
            object obj = @params[0].Instance;
            if (obj is Element) {
                return ((Element)obj).Type;
            } else if (obj != null) {
                return obj.GetType().ToString();
            } else {
                return null;
            }
        }

        private ScriptValue cmdDuplicateElement(ScriptValue[] @params)
        {
            object obj = @params[0].Instance;
            if (obj is Element) {
                Element elm = (Element)obj;
                string new_tag = (@params.Length == 2) ? @params[1].AsString() : elm.Tag;
                return new ScriptValue(elm.Duplicate(new_tag));
            } else if (obj is ICloneable) {
                return new ScriptValue(((ICloneable)obj).Clone());
            } else {
                return null;
            }
        }

        private ScriptValue cmdWaitForCompletion(ScriptValue[] @params)
        {
            object target = @params[0].Instance;
            if (target is ActionThreadInfo) {
                ActionThreadInfo ati = (ActionThreadInfo)target;
                if (ati.ExitCode.HasValue) {
                    if (@params.Length == 2 && IsVarRef(@params[1])) {
                        SetToVariable(@params[1], ati.ExitCode.Value.Instance);
                    }
                    return true;
                } else {
                    return false;
                }
            } else if (target is ActionProcessInfo) {
                ActionProcessInfo api = (ActionProcessInfo)target;
                if (api.ExitCode.HasValue) {
                    if (@params.Length == 2 && IsVarRef(@params[1])) {
                        SetToVariable(@params[1], api.ExitCode.Value.Instance);
                    }
                    return true;
                } else {
                    return false;
                }
            } else {
                ScriptValue result;
                bool completed = GetAsyncCalledCommandResult(@params[0], out result);
                if (@params.Length == 2 && IsVarRef(@params[1])) {
                    SetToVariable(@params[1], result.Instance);
                }
                return completed;
            }
        }

        private ScriptValue cmdSetExceptionHandler(ScriptValue[] @params)
        {
            CurrentThread.SetExceptionHandler(@params[0]);
            return null;
        }

        private ScriptValue cmdCDbl(ScriptValue[] @params) { return @params[0].AsDouble(); }
        private ScriptValue cmdCSng(ScriptValue[] @params) { return @params[0].AsSingle(); }
        private ScriptValue cmdCDec(ScriptValue[] @params) { return @params[0].AsDecimal(); }
        private ScriptValue cmdCLng(ScriptValue[] @params) { return @params[0].AsInt64(); }
        private ScriptValue cmdCInt(ScriptValue[] @params) { return @params[0].AsInt32(); }
        private ScriptValue cmdCShort(ScriptValue[] @params) { return @params[0].AsInt16(); }
        private ScriptValue cmdCByte(ScriptValue[] @params) { return @params[0].AsByte(); }
        private ScriptValue cmdCBool(ScriptValue[] @params) { return @params[0].AsBoolean(); }
        private ScriptValue cmdCStr(ScriptValue[] @params) { return @params[0].AsString(); }

        private static Dictionary<Assembly, IList<string>> myProcessors = new Dictionary<Assembly, IList<string>>();
        static internal void AddCommandProcessor(Assembly asm, IList<string> command_set_list)
        {
            myProcessors.Add(asm, command_set_list);
        }

        /// <summary>
        /// このスクリプトエンジンおよびプラグインでサポートされる全てのコマンドを取得する。
        /// 派生クラスでは、base.GetCommandMap() の戻り値に、自らのコマンドを追加すること。
        /// </summary>
        /// <returns>コマンドの名前をキーとし、そのデリゲートを値とするディクショナリ</returns>
        protected virtual IDictionary<string, CommandImpl> GetCommandMap()
        {
            Dictionary<string, CommandImpl> dic = new Dictionary<string, CommandImpl>();

            foreach (KeyValuePair<Assembly, IList<string>> plugin in myProcessors) {
                foreach (string command_set in plugin.Value) {
                    IHasCommand obj = (IHasCommand)plugin.Key.CreateInstance(command_set);
                    obj.SetTarget(this);
                    foreach (KeyValuePair<string, CommandImpl> kv in obj.GetCommandMap()) {
                        dic.Add(String.Concat(plugin.Key.GetName().Name, ".", ((object)obj).GetType().Name, ".", kv.Key), kv.Value);
                    }
                }
            }

            dic.Add("GetProcedure", cmdGetProcedure);
            dic.Add("GetArguments", cmdGetArguments);
            dic.Add("Call", cmdCall);
            dic.Add("PreloadScript", cmdPreloadScript);
            dic.Add("ClearScriptCache", cmdClearScriptCache);
            dic.Add("BeginProcess", cmdBeginProcess);
            dic.Add("RunThread", cmdRunThread);
            dic.Add("SuspendThread", cmdSuspendThread);
            dic.Add("ResumeThread", cmdResumeThread);
            dic.Add("Sleep", cmdSleep);
            dic.Add("Idle", cmdIdle);
            dic.Add("Quit", cmdQuit);
            dic.Add("RegisterEventHandler", cmdRegisterEventHandler);
            dic.Add("UnregisterEventHandler", cmdUnregisterEventHandler);
            dic.Add("RaiseEvent", cmdRaiseEvent);
            dic.Add("SendSignal", cmdSendSignal);
            dic.Add("ResetSignal", cmdResetSignal);
            dic.Add("WaitForSignal", cmdWaitForSignal);
            dic.Add("GetMember", cmdGetMember);
            dic.Add("SetMember", cmdSetMember);
            dic.Add("CreateInstance", cmdCreateInstance);
            dic.Add("CreateList", cmdCreateList);
            dic.Add("CreateArrayList", cmdCreateArrayList);
            dic.Add("CombineList", cmdCombineList);
            dic.Add("CreateDictionary", cmdCreateDictionary);
            dic.Add("DuplicateDictionary", cmdDuplicateDictionary);
            dic.Add("Format", cmdFormat);
            dic.Add("Trace", cmdTrace);
            dic.Add("GetClock", cmdGetClock);
            dic.Add("GetElement", cmdGetElement);
            dic.Add("GetElements", cmdGetElements);
            dic.Add("GetType", cmdGetType);
            dic.Add("DuplicateElement", cmdDuplicateElement);
            dic.Add("WaitForCompletion", cmdWaitForCompletion);
            dic.Add("SetExceptionHandler", cmdSetExceptionHandler);
            dic.Add("CDbl", cmdCDbl);
            dic.Add("CSng", cmdCSng);
            dic.Add("CDec", cmdCDec);
            dic.Add("CLng", cmdCLng);
            dic.Add("CInt", cmdCInt);
            dic.Add("CShort", cmdCShort);
            dic.Add("CByte", cmdCByte);
            dic.Add("CBool", cmdCBool);
            dic.Add("CStr", cmdCStr);

            return dic;
        }

        #endregion

        internal delegate void NativeCommandImpl(ActionThreadInfo ati, int arg1, object arg2);

        internal class NativeCommandInfo
        {
            public NativeCommandImpl IMPL;
            public int ARG1;
            public object ARG2;
            public bool CP;
        }

        internal class RuntimeProcedureInfo
        {
            public ProcedureInfo code;
            public int local;
            //public int argc;
            //I don't use caller-side argc. 
            //Stack pointer is incremented or decremented by the count of arguments passed actually.
            public List<NativeCommandInfo> info;

            public RuntimeProcedureInfo(ProcedureInfo proc)
            {
                code = proc;
                local = proc.GetLocalDataNames().Length;
                //argc = proc.GetArgumentNames().Length;
                info = new List<NativeCommandInfo>(proc.Lines.Count);
            }

            public override string ToString()
            {
                return code.Name;
            }
        }

        private void ThrowLinkError(RuntimeProcedureInfo rpi, string name, int cause)
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine("リンクに失敗しました。");
            sb.Append("モジュール : ");
            sb.AppendLine(rpi.code.Module.FilePath);
            sb.Append("プロシージャ : ");
            if (!rpi.code.Name.StartsWith("!")) {
                sb.AppendLine(rpi.code.Name);
            } else {
                sb.AppendLine("モジュール イニシャライザ");
            }
            sb.Append("対象 : ");
            sb.AppendLine(name);
            sb.Append("報告 : ");
            switch (cause) {
                case 0:
                    sb.Append("コマンドが定義されていません。");
                    break;
                case 1:
                    sb.Append("プロシージャが定義されていません。");
                    break;
                case 2:
                    sb.Append("コマンドまたはプロシージャが定義されていません。");
                    break;
                case 3:
                    sb.Append("プロシージャを特定できません。");
                    break;
            }
            throw new GustFrontException(sb.ToString());
        }

        private RuntimeProcedureInfo CompileProcedure(RuntimeProcedureInfo rpi, int globalOffset,
            IDictionary<string, CommandImpl> cmdmap, IDictionary<string, RuntimeProcedureInfo> rpis)
        {
            ProcedureInfo info = rpi.code;

            foreach (ScriptCommand line in info.Lines) {

                NativeCommandImpl nci = null;
                int arg1 = (int)line.Argument1;
                object arg2 = line.Argument2;

                switch (line.Opcode) {
                    case Opcodes.DBG:
                        nci = ncDBG;
                        break;
                    case Opcodes.CALL:
                        string callee = (string)arg2;
                        int dot = callee.IndexOf('.');
                        if (dot != -1) { //Engine.Command or Module.Procedure
                            string mod = callee.Substring(0, dot);
                            string name = callee.Substring(dot + 1);
                            if (Util.CaseInsensitiveEquals(mod, "Engine")) {
                                CommandImpl ci;
                                if (cmdmap.TryGetValue(name, out ci)) {
                                    nci = ncCALLCmd;
                                    arg2 = new object[] { name, ci, false };
                                } else {
                                    ThrowLinkError(rpi, name, 0);
                                }
                            } else if (Util.CaseInsensitiveEquals(mod, "Async")) {
                                CommandImpl ci;
                                if (cmdmap.TryGetValue(name, out ci)) {
                                    nci = ncCALLCmd;
                                    arg2 = new object[] { name, ci, true };
                                } else {
                                    ThrowLinkError(rpi, name, 0);
                                }
                            } else {
                                ModuleInfo ref_mod = null;
                                if (info.Module.ModRootNs.TryGetValue(mod, out ref_mod)) {
                                    RuntimeProcedureInfo _rpi;
                                    if (rpis.TryGetValue(String.Concat(ref_mod.GetHashCode(), ".", name), out _rpi)) {
                                        nci = ncCALLProc;
                                        arg2 = new object[] { name, _rpi };
                                    } else {
                                        ThrowLinkError(rpi, name, 1);
                                    }
                                }
                                if (nci == null) {
                                    CommandImpl ci;
                                    if (cmdmap.TryGetValue(callee, out ci)) {
                                        nci = ncCALLCmd;
                                        arg2 = new object[] { callee, ci, false };
                                    } else {
                                        ThrowLinkError(rpi, callee, 2);
                                    }
                                }
                            }
                        } else { //Command or Procedure
                            RuntimeProcedureInfo _rpi;
                            if (rpis.TryGetValue(String.Concat(info.Module.GetHashCode(), ".", callee), out _rpi)) {
                                nci = ncCALLProc;
                                arg2 = new object[] { callee, _rpi };
                            } else {
                                foreach (ModuleInfo ref_mod in info.Module.GetReferences()) {
                                    if (rpis.TryGetValue(String.Concat(ref_mod.GetHashCode(), ".", callee), out _rpi)) {
                                        if (nci == null) {
                                            nci = ncCALLProc;
                                            arg2 = new object[] { callee, _rpi };
                                        } else {
                                            ThrowLinkError(rpi, callee, 3);
                                        }
                                    }
                                }
                                if (nci == null) {
                                    CommandImpl ci;
                                    if (cmdmap.TryGetValue(callee, out ci)) {
                                        nci = ncCALLCmd;
                                        arg2 = new object[] { callee, ci, false };
                                    } else {
                                        ThrowLinkError(rpi, callee, 2);
                                    }
                                }
                            }
                        }
                        break;
                    case Opcodes.RET:
                        if (arg1 != 0) {
                            nci = ncRET;
                        } else {
                            nci = ncRETNull;
                        }
                        break;
                    case Opcodes.JMP:
                        nci = ncJMP;
                        break;
                    case Opcodes.JZ:
                        nci = ncJZ;
                        break;
                    case Opcodes.JNZ:
                        nci = ncJNZ;
                        break;
                    case Opcodes.POP:
                        nci = ncPOP;
                        break;
                    case Opcodes.LDL:
                        nci = ncLDL;
                        break;
                    case Opcodes.LDA:
                        nci = ncLDL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.LDG:
                        nci = ncLDG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.LDI:
                        nci = ncLDI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.STL:
                        nci = ncSTL;
                        break;
                    case Opcodes.STA:
                        nci = ncSTL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.STG:
                        nci = ncSTG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.NEG:
                        nci = ncNEG;
                        break;
                    case Opcodes.ADD:
                        nci = ncADD;
                        break;
                    case Opcodes.ADDL:
                        nci = ncADDL;
                        break;
                    case Opcodes.ADDA:
                        nci = ncADDL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.ADDG:
                        nci = ncADDG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.ADDI:
                        nci = ncADDI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.SUB:
                        nci = ncSUB;
                        break;
                    case Opcodes.SUBL:
                        nci = ncSUBL;
                        break;
                    case Opcodes.SUBA:
                        nci = ncSUBL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.SUBG:
                        nci = ncSUBG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.SUBI:
                        nci = ncSUBI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.MUL:
                        nci = ncMUL;
                        break;
                    case Opcodes.MULL:
                        nci = ncMULL;
                        break;
                    case Opcodes.MULA:
                        nci = ncMULL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.MULG:
                        nci = ncMULG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.MULI:
                        nci = ncMULI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.DIV:
                        nci = ncDIV;
                        break;
                    case Opcodes.DIVL:
                        nci = ncDIVL;
                        break;
                    case Opcodes.DIVA:
                        nci = ncDIVL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.DIVG:
                        nci = ncDIVG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.DIVI:
                        nci = ncDIVI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.REM:
                        nci = ncREM;
                        break;
                    case Opcodes.REML:
                        nci = ncREML;
                        break;
                    case Opcodes.REMA:
                        nci = ncREML;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.REMG:
                        nci = ncREMG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.REMI:
                        nci = ncREMI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.SAL:
                        nci = ncSAL;
                        break;
                    case Opcodes.SALL:
                        nci = ncSALL;
                        break;
                    case Opcodes.SALA:
                        nci = ncSALL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.SALG:
                        nci = ncSALG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.SALI:
                        nci = ncSALI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.SAR:
                        nci = ncSAR;
                        break;
                    case Opcodes.SARL:
                        nci = ncSARL;
                        break;
                    case Opcodes.SARA:
                        nci = ncSARL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.SARG:
                        nci = ncSARG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.SARI:
                        nci = ncSARI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.NOT:
                        nci = ncNOT;
                        break;
                    case Opcodes.AND:
                        nci = ncAND;
                        break;
                    case Opcodes.ANDL:
                        nci = ncANDL;
                        break;
                    case Opcodes.ANDA:
                        nci = ncANDL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.ANDG:
                        nci = ncANDG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.ANDI:
                        nci = ncANDI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.OR:
                        nci = ncOR;
                        break;
                    case Opcodes.ORL:
                        nci = ncORL;
                        break;
                    case Opcodes.ORA:
                        nci = ncORL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.ORG:
                        nci = ncORG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.ORI:
                        nci = ncORI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.XOR:
                        nci = ncXOR;
                        break;
                    case Opcodes.XORL:
                        nci = ncXORL;
                        break;
                    case Opcodes.XORA:
                        nci = ncXORL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.XORG:
                        nci = ncXORG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.XORI:
                        nci = ncXORI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.EQ:
                        nci = ncEQ;
                        break;
                    case Opcodes.EQL:
                        nci = ncEQL;
                        break;
                    case Opcodes.EQA:
                        nci = ncEQL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.EQG:
                        nci = ncEQG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.EQI:
                        nci = ncEQI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.NE:
                        nci = ncNE;
                        break;
                    case Opcodes.NEL:
                        nci = ncNEL;
                        break;
                    case Opcodes.NEA:
                        nci = ncNEL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.NEGC:
                        nci = ncNEGC;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.NEI:
                        nci = ncNEI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.LT:
                        nci = ncLT;
                        break;
                    case Opcodes.LTL:
                        nci = ncLTL;
                        break;
                    case Opcodes.LTA:
                        nci = ncLTL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.LTG:
                        nci = ncLTG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.LTI:
                        nci = ncLTI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.GT:
                        nci = ncGT;
                        break;
                    case Opcodes.GTL:
                        nci = ncGTL;
                        break;
                    case Opcodes.GTA:
                        nci = ncGTL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.GTG:
                        nci = ncGTG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.GTI:
                        nci = ncGTI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.LE:
                        nci = ncLE;
                        break;
                    case Opcodes.LEL:
                        nci = ncLEL;
                        break;
                    case Opcodes.LEA:
                        nci = ncLEL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.LEG:
                        nci = ncLEG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.LEI:
                        nci = ncLEI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    case Opcodes.GE:
                        nci = ncGE;
                        break;
                    case Opcodes.GEL:
                        nci = ncGEL;
                        break;
                    case Opcodes.GEA:
                        nci = ncGEL;
                        arg1 = -(arg1 + 4);
                        break;
                    case Opcodes.GEG:
                        nci = ncGEG;
                        arg1 += globalOffset;
                        break;
                    case Opcodes.GEI:
                        nci = ncGEI;
                        arg2 = new ScriptValue(arg2);
                        break;
                    default:
                        nci = ncNOP;
                        break;
                }

                NativeCommandInfo op = new NativeCommandInfo();
                op.IMPL = nci;
                op.ARG1 = arg1;
                op.ARG2 = arg2;
                op.CP = Opcodes.DBG < line.Opcode && line.Opcode < Opcodes.POP;
                rpi.info.Add(op);
            }

            return rpi;
        }

        private Dictionary<string, RuntimeProcedureInfo> MapModuleToRuntimeSpace(ModuleInfo mod, out int globalAreaSize)
        {
            globalAreaSize = 0;

            Dictionary<string, RuntimeProcedureInfo> rpis = new Dictionary<string, RuntimeProcedureInfo>();
            MapModuleToRuntimeSpace(mod, rpis, GetCommandMap(), ref globalAreaSize);
            return rpis;
        }

        private void MapModuleToRuntimeSpace(ModuleInfo mod,
            Dictionary<string, RuntimeProcedureInfo> rpis, IDictionary<string, CommandImpl> cmdmap, ref int globalAreaSize)
        {
            foreach (ModuleInfo ref_mod in mod.GetReferences()) {
                if (!rpis.ContainsKey(String.Concat("!", ref_mod.GetHashCode()))) {
                    MapModuleToRuntimeSpace(ref_mod, rpis, cmdmap, ref globalAreaSize);
                }
            }

            RuntimeProcedureInfo rpi = new RuntimeProcedureInfo(mod.GetGlobalInitializer());
            rpis.Add(String.Concat("!", rpi.code.Module.GetHashCode()), rpi);
            foreach (KeyValuePair<string, ProcedureInfo> kv in mod.GetProcedures()) {
                rpi = new RuntimeProcedureInfo(kv.Value);
                rpis.Add(String.Concat(rpi.code.Module.GetHashCode(), ".", rpi.code.Name), rpi);
            }

            foreach (KeyValuePair<string, RuntimeProcedureInfo> kv in rpis) {
                CompileProcedure(kv.Value, globalAreaSize, cmdmap, rpis);
            }
            globalAreaSize += mod.GetGlobalVariableCount();
        }

        private static void ncDBG(ActionThreadInfo ati, int arg1, object arg2)
        {
            object[] info = (object[])arg2;
            switch (arg1) {
                case 0:
                    ati.SetCurrentLineNumber((int)info[0]);
                    break;
            }
        }

        private static void ncCALLCmd(ActionThreadInfo ati, int arg1, object arg2)
        {
            object[] cmdInfo = (object[])arg2;
            string name = (string)cmdInfo[0];
            CommandImpl cmd = (CommandImpl)cmdInfo[1];
            bool callAsAsync = (bool)cmdInfo[2];

            ScriptValue[] @params = new ScriptValue[arg1];
            for (int i = 0; i < arg1; i++) {
                @params[i] = ati.Pop();
            }

            if (!callAsAsync) {
                ati.PushCallee(name);
                Action prev = ati.CurrentAction;
                ScriptValue result = cmd(@params);
                if (prev == ati.CurrentAction) {
                    ati.PopCallee();
                    ati.Push(result);
                }
            } else {
                ati.Push(new ScriptValue(new KeyValuePair<IAsyncResult, CommandImpl>(
                    cmd.BeginInvoke(@params, null, null), cmd)));
            }
        }

        private static void ncCALLProc(ActionThreadInfo ati, int arg1, object arg2)
        {
            object[] procInfo = (object[])arg2;
            string name = (string)procInfo[0];
            RuntimeProcedureInfo rpi = (RuntimeProcedureInfo)procInfo[1];

            ati.EnterProcedureFromScript(rpi, arg1);
        }

        private static void ncRET(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.ExitProcedure(new ScriptValue(ati.Pop().Instance));
        }

        private static void ncRETNull(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.ExitProcedure(ati.Pop());
            ati.Pop();
        }

        private static void ncJMP(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.GoNext(arg1);
        }

        private static void ncJZ(ActionThreadInfo ati, int arg1, object arg2)
        {
            if (ati.Pop().AsInt32() == 0)
                ati.GoNext(arg1);
        }

        private static void ncJNZ(ActionThreadInfo ati, int arg1, object arg2)
        {
            if (ati.Pop().AsInt32() != 0)
                ati.GoNext(arg1);
        }

        private static void ncPOP(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.Pop();
        }

        private static void ncLDL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.Push(ati.FastGetLocalValue(arg1));
        }
        private static void ncLDG(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.Push(ati.Process.FastGetValue(arg1));
        }
        private static void ncLDI(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.Push((ScriptValue)arg2);
        }

        private static void ncSTL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.FastSetLocalValue(arg1, ati.Pop());
        }
        private static void ncSTG(ActionThreadInfo ati, int arg1, object arg2)
        {
            ati.Process.FastSetValue(arg1, ati.Pop());
        }

        private static void ncNEG(ActionThreadInfo ati, int arg1, object arg2)
        {
            object o1 = ati.Pop().Instance;
            if (o1 is double) {
                ati.Push(-Convert.ToDouble(o1));
            } else if (o1 is float) {
                ati.Push(-Convert.ToSingle(o1));
            } else if (o1 is decimal) {
                ati.Push(-Convert.ToDecimal(o1));
            } else if (o1 is long) {
                ati.Push(-Convert.ToInt64(o1));
            } else if (o1 is int) {
                ati.Push(-Convert.ToInt32(o1));
            } else if (o1 is short) {
                ati.Push(-Convert.ToInt16(o1));
            } else if (o1 is byte) {
                ati.Push(-Convert.ToByte(o1)); // type of result is short
            } else {
                throw new GustFrontException("符号反転できない値です。");
            }
        }

        private static void ADDImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 is string || o2 is string) {
                ati.Push(Convert.ToString(o1) + Convert.ToString(o2));
            } else if (o1 is double || o2 is double) {
                ati.Push(Convert.ToDouble(o1) + Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                ati.Push(Convert.ToSingle(o1) + Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                ati.Push(Convert.ToDecimal(o1) + Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                ati.Push(Convert.ToInt64(o1) + Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                ati.Push(Convert.ToInt32(o1) + Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                ati.Push(Convert.ToInt16(o1) + Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                ati.Push(Convert.ToByte(o1) + Convert.ToByte(o2));
            } else {
                throw new GustFrontException("加算できない値です。");
            }
        }
        private static void ncADD(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            ADDImpl(ati, v1, v2);
        }
        private static void ncADDL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ADDImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncADDG(ActionThreadInfo ati, int arg1, object arg2)
        {
            ADDImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncADDI(ActionThreadInfo ati, int arg1, object arg2)
        {
            ADDImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void SUBImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 is double || o2 is double) {
                ati.Push(Convert.ToDouble(o1) - Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                ati.Push(Convert.ToSingle(o1) - Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                ati.Push(Convert.ToDecimal(o1) - Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                ati.Push(Convert.ToInt64(o1) - Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                ati.Push(Convert.ToInt32(o1) - Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                ati.Push(Convert.ToInt16(o1) - Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                ati.Push(Convert.ToByte(o1) - Convert.ToByte(o2));
            } else {
                throw new GustFrontException("減算できない値です。");
            }
        }
        private static void ncSUB(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            SUBImpl(ati, v1, v2);
        }
        private static void ncSUBL(ActionThreadInfo ati, int arg1, object arg2)
        {
            SUBImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncSUBG(ActionThreadInfo ati, int arg1, object arg2)
        {
            SUBImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncSUBI(ActionThreadInfo ati, int arg1, object arg2)
        {
            SUBImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void MULImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 is double || o2 is double) {
                ati.Push(Convert.ToDouble(o1) * Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                ati.Push(Convert.ToSingle(o1) * Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                ati.Push(Convert.ToDecimal(o1) * Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                ati.Push(Convert.ToInt64(o1) * Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                ati.Push(Convert.ToInt32(o1) * Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                ati.Push(Convert.ToInt16(o1) * Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                ati.Push(Convert.ToByte(o1) * Convert.ToByte(o2));
            } else {
                throw new GustFrontException("乗算できない値です。");
            }
        }
        private static void ncMUL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            MULImpl(ati, v1, v2);
        }
        private static void ncMULL(ActionThreadInfo ati, int arg1, object arg2)
        {
            MULImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncMULG(ActionThreadInfo ati, int arg1, object arg2)
        {
            MULImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncMULI(ActionThreadInfo ati, int arg1, object arg2)
        {
            MULImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void DIVImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 is double || o2 is double) {
                ati.Push(Convert.ToDouble(o1) / Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                ati.Push(Convert.ToSingle(o1) / Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                ati.Push(Convert.ToDecimal(o1) / Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                ati.Push(Convert.ToInt64(o1) / Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                ati.Push(Convert.ToInt32(o1) / Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                ati.Push(Convert.ToInt16(o1) / Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                ati.Push(Convert.ToByte(o1) / Convert.ToByte(o2));
            } else {
                throw new GustFrontException("除算できない値です。");
            }
        }
        private static void ncDIV(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            DIVImpl(ati, v1, v2);
        }
        private static void ncDIVL(ActionThreadInfo ati, int arg1, object arg2)
        {
            DIVImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncDIVG(ActionThreadInfo ati, int arg1, object arg2)
        {
            DIVImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncDIVI(ActionThreadInfo ati, int arg1, object arg2)
        {
            DIVImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void REMImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 is double || o2 is double) {
                ati.Push(Convert.ToDouble(o1) % Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                ati.Push(Convert.ToSingle(o1) % Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                ati.Push(Convert.ToDecimal(o1) % Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                ati.Push(Convert.ToInt64(o1) % Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                ati.Push(Convert.ToInt32(o1) % Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                ati.Push(Convert.ToInt16(o1) % Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                ati.Push(Convert.ToByte(o1) % Convert.ToByte(o2));
            } else {
                throw new GustFrontException("剰余算できない値です。");
            }
        }
        private static void ncREM(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            REMImpl(ati, v1, v2);
        }
        private static void ncREML(ActionThreadInfo ati, int arg1, object arg2)
        {
            REMImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncREMG(ActionThreadInfo ati, int arg1, object arg2)
        {
            REMImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncREMI(ActionThreadInfo ati, int arg1, object arg2)
        {
            REMImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void SALImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            int o2 = v2.AsInt32();
            if (o1 is long) {
                ati.Push(Convert.ToInt64(o1) << o2);
            } else if (o1 is int) {
                ati.Push(Convert.ToInt32(o1) << o2);
            } else if (o1 is short) {
                ati.Push(Convert.ToInt16(o1) << o2); // type of result is int
            } else if (o1 is byte) {
                ati.Push(Convert.ToByte(o1) << o2); // type of result is int
            } else {
                throw new GustFrontException("左シフトできない値です。");
            }
        }
        private static void ncSAL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            SALImpl(ati, v1, v2);
        }
        private static void ncSALL(ActionThreadInfo ati, int arg1, object arg2)
        {
            SALImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncSALG(ActionThreadInfo ati, int arg1, object arg2)
        {
            SALImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncSALI(ActionThreadInfo ati, int arg1, object arg2)
        {
            SALImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void SARImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            int o2 = v2.AsInt32();
            if (o1 is long) {
                ati.Push(Convert.ToInt64(o1) >> o2);
            } else if (o1 is int) {
                ati.Push(Convert.ToInt32(o1) >> o2);
            } else if (o1 is short) {
                ati.Push(Convert.ToInt16(o1) >> o2); // type of result is int
            } else if (o1 is byte) {
                ati.Push(Convert.ToByte(o1) >> o2); // type of result is int
            } else {
                throw new GustFrontException("右シフトできない値です。");
            }
        }
        private static void ncSAR(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            SARImpl(ati, v1, v2);
        }
        private static void ncSARL(ActionThreadInfo ati, int arg1, object arg2)
        {
            SARImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncSARG(ActionThreadInfo ati, int arg1, object arg2)
        {
            SARImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncSARI(ActionThreadInfo ati, int arg1, object arg2)
        {
            SARImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static bool IsInteger(object o)
        {
            return o is int || o is long || o is short || o is byte;
        }

        private static void ncNOT(ActionThreadInfo ati, int arg1, object arg2)
        {
            object o1 = ati.Pop().Instance;
            if (o1 is long) {
                ati.Push(~(long)o1);
            } else if (o1 is int) {
                ati.Push(~(int)o1);
            } else if (o1 is short) {
                ati.Push(~(short)o1); // type of result is int
            } else if (o1 is byte) {
                ati.Push(~(byte)o1); // type of result is int
            } else if (o1 is bool) {
                ati.Push(!(bool)o1);
            } else {
                throw new GustFrontException("補数を計算できません。");
            }
        }

        private static void ANDImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (IsInteger(o1) && IsInteger(o2)) {
                if (o1 is long || o2 is long) {
                    ati.Push(Convert.ToInt64(o1) & Convert.ToInt64(o2));
                } else if (o1 is int || o2 is int) {
                    ati.Push(Convert.ToInt32(o1) & Convert.ToInt32(o2));
                } else if (o1 is short || o2 is short) {
                    ati.Push(Convert.ToInt16(o1) & Convert.ToInt16(o2));
                } else if (o1 is byte || o2 is byte) {
                    ati.Push(Convert.ToByte(o1) & Convert.ToByte(o2));
                }
            } else if (o1 is bool && o2 is bool) {
                ati.Push((bool)o1 & (bool)o2);
            } else {
                throw new GustFrontException("論理積を計算できません。");
            }
        }
        private static void ncAND(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            ANDImpl(ati, v1, v2);
        }
        private static void ncANDL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ANDImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncANDG(ActionThreadInfo ati, int arg1, object arg2)
        {
            ANDImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncANDI(ActionThreadInfo ati, int arg1, object arg2)
        {
            ANDImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void ORImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (IsInteger(o1) && IsInteger(o2)) {
                if (o1 is long || o2 is long) {
                    ati.Push(Convert.ToInt64(o1) | Convert.ToInt64(o2));
                } else if (o1 is int || o2 is int) {
                    ati.Push(Convert.ToInt32(o1) | Convert.ToInt32(o2));
                } else if (o1 is short || o2 is short) {
                    ati.Push(Convert.ToInt16(o1) | Convert.ToInt16(o2));
                } else if (o1 is byte || o2 is byte) {
                    ati.Push(Convert.ToByte(o1) | Convert.ToByte(o2));
                }
            } else if (o1 is bool && o2 is bool) {
                ati.Push((bool)o1 | (bool)o2);
            } else {
                throw new GustFrontException("論理和を計算できません。");
            }
        }
        private static void ncOR(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            ORImpl(ati, v1, v2);
        }
        private static void ncORL(ActionThreadInfo ati, int arg1, object arg2)
        {
            ORImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncORG(ActionThreadInfo ati, int arg1, object arg2)
        {
            ORImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncORI(ActionThreadInfo ati, int arg1, object arg2)
        {
            ORImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void XORImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (IsInteger(o1) && IsInteger(o2)) {
                if (o1 is long || o2 is long) {
                    ati.Push(Convert.ToInt64(o1) ^ Convert.ToInt64(o2));
                } else if (o1 is int || o2 is int) {
                    ati.Push(Convert.ToInt32(o1) ^ Convert.ToInt32(o2));
                } else if (o1 is short || o2 is short) {
                    ati.Push(Convert.ToInt16(o1) ^ Convert.ToInt16(o2));
                } else if (o1 is byte || o2 is byte) {
                    ati.Push(Convert.ToByte(o1) ^ Convert.ToByte(o2));
                }
            } else if (o1 is bool && o2 is bool) {
                ati.Push((bool)o1 ^ (bool)o2);
            } else {
                throw new GustFrontException("排他的論理和を計算できません。");
            }
        }
        private static void ncXOR(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            XORImpl(ati, v1, v2);
        }
        private static void ncXORL(ActionThreadInfo ati, int arg1, object arg2)
        {
            XORImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncXORG(ActionThreadInfo ati, int arg1, object arg2)
        {
            XORImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncXORI(ActionThreadInfo ati, int arg1, object arg2)
        {
            XORImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static int CompareScriptValue(ScriptValue v1, ScriptValue v2)
        {
            object o1 = v1.Instance;
            object o2 = v2.Instance;
            if (o1 == null) {
                return (o2 != null) ? -1 : 0;
            } else if (o2 == null) {
                return 1;
            } else if (o1 is double || o2 is double) {
                return Convert.ToDouble(o1).CompareTo(Convert.ToDouble(o2));
            } else if (o1 is float || o2 is float) {
                return Convert.ToSingle(o1).CompareTo(Convert.ToSingle(o2));
            } else if (o1 is decimal || o2 is decimal) {
                return Convert.ToDecimal(o1).CompareTo(Convert.ToDecimal(o2));
            } else if (o1 is long || o2 is long) {
                return Convert.ToInt64(o1).CompareTo(Convert.ToInt64(o2));
            } else if (o1 is int || o2 is int) {
                return Convert.ToInt32(o1).CompareTo(Convert.ToInt32(o2));
            } else if (o1 is short || o2 is short) {
                return Convert.ToInt16(o1).CompareTo(Convert.ToInt16(o2));
            } else if (o1 is byte || o2 is byte) {
                return Convert.ToByte(o1).CompareTo(Convert.ToByte(o2));
            } else if (o1 is IComparable) {
                return ((IComparable)o1).CompareTo(o2);
            } else if (o2 is IComparable) {
                return -((IComparable)o2).CompareTo(o1);
            } else {
                return (o1.Equals(o2)) ? 0 : -1;
            }
        }

        private static void EQImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) == 0);
        }
        private static void ncEQ(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            EQImpl(ati, v1, v2);
        }
        private static void ncEQL(ActionThreadInfo ati, int arg1, object arg2)
        {
            EQImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncEQG(ActionThreadInfo ati, int arg1, object arg2)
        {
            EQImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncEQI(ActionThreadInfo ati, int arg1, object arg2)
        {
            EQImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void NEImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) != 0);
        }
        private static void ncNE(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            NEImpl(ati, v1, v2);
        }
        private static void ncNEL(ActionThreadInfo ati, int arg1, object arg2)
        {
            NEImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncNEGC(ActionThreadInfo ati, int arg1, object arg2)
        {
            NEImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncNEI(ActionThreadInfo ati, int arg1, object arg2)
        {
            NEImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void LTImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) < 0);
        }
        private static void ncLT(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            LTImpl(ati, v1, v2);
        }
        private static void ncLTL(ActionThreadInfo ati, int arg1, object arg2)
        {
            LTImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncLTG(ActionThreadInfo ati, int arg1, object arg2)
        {
            LTImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncLTI(ActionThreadInfo ati, int arg1, object arg2)
        {
            LTImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void GTImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) > 0);
        }
        private static void ncGT(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            GTImpl(ati, v1, v2);
        }
        private static void ncGTL(ActionThreadInfo ati, int arg1, object arg2)
        {
            GTImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncGTG(ActionThreadInfo ati, int arg1, object arg2)
        {
            GTImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncGTI(ActionThreadInfo ati, int arg1, object arg2)
        {
            GTImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void LEImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) <= 0);
        }
        private static void ncLE(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            LEImpl(ati, v1, v2);
        }
        private static void ncLEL(ActionThreadInfo ati, int arg1, object arg2)
        {
            LEImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncLEG(ActionThreadInfo ati, int arg1, object arg2)
        {
            LEImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncLEI(ActionThreadInfo ati, int arg1, object arg2)
        {
            LEImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void GEImpl(ActionThreadInfo ati, ScriptValue v1, ScriptValue v2)
        {
            ati.Push(CompareScriptValue(v1, v2) >= 0);
        }
        private static void ncGE(ActionThreadInfo ati, int arg1, object arg2)
        {
            ScriptValue v2 = ati.Pop();
            ScriptValue v1 = ati.Pop();
            GEImpl(ati, v1, v2);
        }
        private static void ncGEL(ActionThreadInfo ati, int arg1, object arg2)
        {
            GEImpl(ati, ati.Pop(), ati.FastGetLocalValue(arg1));
        }
        private static void ncGEG(ActionThreadInfo ati, int arg1, object arg2)
        {
            GEImpl(ati, ati.Pop(), ati.Process.FastGetValue(arg1));
        }
        private static void ncGEI(ActionThreadInfo ati, int arg1, object arg2)
        {
            GEImpl(ati, ati.Pop(), (ScriptValue)arg2);
        }

        private static void ncNOP(ActionThreadInfo ati, int arg1, object arg2)
        {
            //No OPeration
        }

        private void ProcessScript(ActionThreadInfo ati)
        {
            int count = 0;
            for (; ; ) {
                NativeCommandInfo[] ncis = ati.GetNextCommand();
                if (ncis != null) {
                    foreach (NativeCommandInfo nci in ncis) {
                        nci.IMPL(ati, nci.ARG1, nci.ARG2);
                        if (count++ >= 1024 * 1024) {
                            throw new GustFrontException("スクリプトが無限ループしているようです。");
                        }
                    }
                } else {
                    break;
                }
            }
        }

        /// <summary>
        /// 値がプロシージャかどうか調べる。
        /// </summary>
        /// <param name="value">対象となる値</param>
        /// <returns>プロシージャの場合 True を、それ以外の場合 False を返す。</returns>
        public bool IsValidProcedure(ScriptValue value)
        {
            return value.Instance is RuntimeProcedureInfo;
        }

        /// <summary>
        /// 現在処理されているスレッド上で、プロシージャを呼び出す。
        /// </summary>
        /// <param name="theProcedure">IsValidProcedure(theProcedure) が True となる値</param>
        /// <param name="args">プロシージャのパラメータ</param>
        public void CallScript(ScriptValue theProcedure, params ScriptValue[] args)
        {
            if (myTargetThread != null) {
                myTargetThread.EnterProcedure((RuntimeProcedureInfo)theProcedure.Instance, args);
            } else {
                throw new InvalidOperationException("Must call ScriptEngine.CallScript on a thread.");
            }
        }

        /// <summary>
        /// 現在処理されているスレッドに、アクションを登録する。
        /// </summary>
        /// <param name="obj">登録するアクション</param>
        public void RegisterThreadAction(Action obj)
        {
            if (myTargetThread != null) {
                myTargetThread.RegisterAction(obj);
            } else {
                throw new InvalidOperationException("Must call ScriptEngine.RegisterThreadAction on a thread.");
            }
        }

        /// <summary>
        /// 現在処理されているスレッドから、アクションを登録解除する。
        /// </summary>
        /// <param name="obj">登録解除されるアクション</param>
        public void UnregisterThreadAction(Action obj)
        {
            if (myTargetThread != null) {
                myTargetThread.UnregisterAction(obj);
            } else {
                throw new InvalidOperationException("Must call ScriptEngine.UnregisterThreadAction on a thread.");
            }
        }

        /// <summary>
        /// 現在処理されているプロセスにスレッドを追加し、そのスレッド上でアクションを実行する。
        /// これにより、開始プロシージャを用意することなく、アクションを非同期的に実行できる。
        /// </summary>
        /// <param name="obj">実行されるアクション</param>
        public void RunProcessAction(Action obj) // RunThreadWithAction
        {
            if (myTargetThread != null) {
                myTargetThread.Process.RegisterAction(obj);
            } else {
                throw new InvalidOperationException("Must call ScriptEngine.RunProcessAction on a thread.");
            }
        }

        /// <summary>
        /// 現在処理されているスレッド。
        /// </summary>
        protected ActionThreadInfo CurrentThread
        {
            get { return myTargetThread; }
        }

        private ActionThreadInfo[] GetAllThreads()
        {
            List<ActionThreadInfo> threads = new List<ActionThreadInfo>();
            foreach (ActionProcessInfo api in myProcesses) {
                threads.AddRange(api.GetThreads());
            }
            return threads.ToArray();
        }

        private ActionThreadInfo myTargetThread = null;

        /// <summary>
        /// このスクリプトエンジンにある全スレッドに対し、
        /// 時間遷移を要求する。
        /// </summary>
        /// <param name="delta">前回の呼び出しからの経過時間</param>
        protected void TickAllProcess(TimeSpan delta)
        {
            foreach (ActionThreadInfo ati in GetAllThreads()) {
                if (ati.Running) {
                    myTargetThread = ati;
                    try {
                        ati.Tick(delta);
                    } catch (Exception ex) {
                        Util.AppendToLog("エラーが発生しました。");
                        ati.Terminate(ex);
                        if (!ati.Running) {
                            ati.Process.RemoveThread(ati);
                        }
                    } finally {
                        myTargetThread = null;
                    }
                }
            }
        }

        /// <summary>
        /// このスクリプトエンジンにある全スレッドに対し、
        /// 描画を要求する。
        /// </summary>
        /// <param name="r">グラフィックスデバイスのインスタンス</param>
        protected void RenderAllProcess(IGraphicsDevice r)
        {
            foreach (ActionThreadInfo ati in GetAllThreads()) {
                ati.Render(r);
            }
        }
    }
}