﻿/*
Copyright (c) 2013, KAKUMOTO Masayuki
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of the outher KAKUMOTO Masayuki nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Globalization;
using System.Reflection;
using System.Diagnostics;

namespace GustFront
{
    internal enum Opcodes
    {
        NOP = 1,
        DBG,
        CALL,
        RET,
        JMP, JZ, JNZ,
        POP,
        LDL, LDA, LDG, LDI, // load
        STL, STA, STG, // store
        NEG, // -x
        ADD, ADDL, ADDA, ADDG, ADDI, // +
        SUB, SUBL, SUBA, SUBG, SUBI, // -
        MUL, MULL, MULA, MULG, MULI, // *
        DIV, DIVL, DIVA, DIVG, DIVI, // /
        REM, REML, REMA, REMG, REMI, // %
        SAL, SALL, SALA, SALG, SALI, // <<
        SAR, SARL, SARA, SARG, SARI, // >>
        NOT, // Not
        AND, ANDL, ANDA, ANDG, ANDI, // And
        OR, ORL, ORA, ORG, ORI, // Or
        XOR, XORL, XORA, XORG, XORI, // Xor
        EQ, EQL, EQA, EQG, EQI, // =
        NE, NEL, NEA, NEGC, NEI, // <>
        LT, LTL, LTA, LTG, LTI, // <
        GT, GTL, GTA, GTG, GTI, // >
        LE, LEL, LEA, LEG, LEI, // <=
        GE, GEL, GEA, GEG, GEI // >=
    }

    internal class ScriptCommand
    {
        private Opcodes myOpcode;
        private object myArgument1;
        private object myArgument2;

        public ScriptCommand()
            : this(Opcodes.NOP, null, null)
        {
        }

        public ScriptCommand(Opcodes opcode)
            : this(opcode, null, null)
        {
        }

        public ScriptCommand(Opcodes opcode, object arg1)
            : this(opcode, arg1, null)
        {
        }

        public ScriptCommand(Opcodes opcode, object arg1, object arg2)
        {
            myOpcode = opcode;
            myArgument1 = arg1;
            myArgument2 = arg2;
        }

        public Opcodes Opcode
        {
            get { return myOpcode; }
            set { myOpcode = value; }
        }

        public object Argument1
        {
            get { return myArgument1; }
            set { myArgument1 = value; }
        }

        public object Argument2
        {
            get { return myArgument2; }
            set { myArgument2 = value; }
        }
    }

    internal class ProcedureInfo
    {
        private class SpecialString
        {
            string v;
            public SpecialString(string value)
            {
                v = value;
            }
            public override string ToString()
            {
                return v;
            }
        }

        private ModuleInfo myModule = null;
        private string myName = null;
        private string[] myArgumentNames = null;
        private Dictionary<string, int> myLabels = CollectionUtil.CreateCaseInsensitiveDictionary<int>(16);
        private List<string> myLocalDataNames = new List<string>();
        private List<ScriptCommand> myLines = new List<ScriptCommand>(128);

        public List<ScriptCommand> Lines
        {
            get { return myLines; }
        }

        public ProcedureInfo(ModuleInfo space, string name, string[] args)
        {
            myModule = space;
            myName = name;
            myArgumentNames = args;
        }

        public void RestoreStrings()
        {
            Assembly[] asms = AppDomain.CurrentDomain.GetAssemblies();

            int[] indexMap = new int[myLines.Count * 2];
            for (int i = 0; i < indexMap.Length; i++) {
                indexMap[i] = i;
            }

            int index = 0;
            while (index < myLines.Count) {
                ScriptCommand line = myLines[index];
                string arg = Convert.ToString(line.Argument1);
                if (arg != null && arg.Length > 0) {
                    if (arg.EndsWith(".")) arg = arg.Substring(0, arg.Length - 1);
                    if (line.Opcode == Opcodes.LDI) {
                        if (arg.Length >= 2 && arg[0] == '0') {
                            byte[] ba = new byte[arg.Length / 2 - 1];
                            for (int i = 2; i < arg.Length; i += 2) {
                                ba[i / 2 - 1] = Byte.Parse(arg.Substring(i, 2),
                                    NumberStyles.HexNumber, CultureInfo.InvariantCulture);
                            }
                            switch (arg[1]) {
                                case 'L':
                                    line.Argument1 = BitConverter.ToInt64(ba, 0);
                                    break;
                                case 'I':
                                    line.Argument1 = BitConverter.ToInt32(ba, 0);
                                    break;
                                case 'S':
                                    line.Argument1 = BitConverter.ToInt16(ba, 0);
                                    break;
                                case 'B':
                                    line.Argument1 = ba[0];
                                    break;
                                case 'R':
                                    line.Argument1 = BitConverter.ToDouble(ba, 0);
                                    break;
                                case 'F':
                                    line.Argument1 = BitConverter.ToSingle(ba, 0);
                                    break;
                                case 'D':
                                    int[] dp = new int[4];
                                    for (int i = 0; i < dp.Length; i++) {
                                        dp[i] = BitConverter.ToInt32(ba, i * 4);
                                    }
                                    line.Argument1 = new Decimal(dp);
                                    break;
                                case 'T':
                                    StringBuilder sb = new StringBuilder(arg.Length / 2 - 1);
                                    for (int i = 0; i < ba.Length; i += 2) {
                                        sb.Append(BitConverter.ToChar(ba, i));
                                    }
                                    line.Argument1 = sb.ToString();
                                    break;
                            }
                        } else if (Util.CaseInsensitiveEquals(arg, "True")) {
                            line.Argument1 = true;
                        } else if (Util.CaseInsensitiveEquals(arg, "False")) {
                            line.Argument1 = false;
                        } else if (Util.CaseInsensitiveEquals(arg, "Nothing")) {
                            line.Argument1 = null;
                        } else if (Util.CaseInsensitiveEquals(arg, "Infinity")) {
                            if (myLines[index + 1].Opcode != Opcodes.NEG) {
                                line.Argument1 = Double.PositiveInfinity;
                            } else {
                                myLines[index] = new ScriptCommand();
                                myLines[index + 1] = new ScriptCommand(Opcodes.LDI, Double.NegativeInfinity);
                                index++;
                            }
                        } else if (Util.CaseInsensitiveEquals(arg, "NaN")) {
                            line.Argument1 = Double.NaN;
                        } else {
                            bool is_proc = false;
                            int dot = arg.IndexOf('.');
                            if (dot != -1) {
                                string head = arg.Substring(0, dot);
                                string name = arg.Substring(dot + 1);
                                if (myModule.ModRootNs.ContainsKey(head)) {
                                    if (myModule.ModRootNs[head].GetProcedure(name) != null) {
                                        is_proc = true;
                                    }
                                }
                            } else {
                                if (myModule.HasProcedure(arg)) {
                                    is_proc = true;
                                } else {
                                    foreach (ModuleInfo mod in myModule.GetReferences()) {
                                        if (mod.HasProcedure(arg)) {
                                            is_proc = true;
                                            break;
                                        }
                                    }
                                }
                            }
                            if (!is_proc) {
                                if (line.Argument1.ToString().IndexOf('.') == -1) {
                                    if (!myModule.HasImported(arg)) {
                                        throw new GustFrontException(String.Format("識別子 {0} は不明です。", arg));
                                    }
                                }
                                line.Argument1 = new SpecialString(arg);
                            } else {
                                myLines.Insert(index, new ScriptCommand(Opcodes.LDI, new SpecialString(arg)));
                                myLines[index + 1] = new ScriptCommand(Opcodes.CALL, "GetProcedure", 1);
                                for (int i = 0; i < indexMap.Length; i++) {
                                    if (indexMap[i] > index) {
                                        for (int j = i; j < indexMap.Length; j++) {
                                            indexMap[j]++;
                                        }
                                        break;
                                    }
                                }
                                index++;
                            }
                        }
                    } else if (line.Opcode == Opcodes.CALL && (string)line.Argument1 == "GetMember") {
                        ScriptCommand gm_arg0 = myLines[index - 1];
                        if (gm_arg0.Opcode == Opcodes.LDI && gm_arg0.Argument1 is SpecialString) {
                            ScriptCommand gm_arg1 = myLines[index - 2];
                            if (gm_arg1.Opcode == Opcodes.LDI && gm_arg1.Argument1 is SpecialString) {
                                Type t = null;
                                string type_name = gm_arg0.Argument1.ToString();
                                for (int i = 0; i < asms.Length; i++) {
                                    t = asms[i].GetType(type_name, false, true);
                                    if (t != null) {
                                        if (typeof(IHasCommand).IsAssignableFrom(t)) {
                                            myLines[index] = new ScriptCommand(Opcodes.CALL,
                                                String.Concat(asms[i].GetName().Name, ".", gm_arg0.Argument1, ".", gm_arg1.Argument1),
                                                (int)myLines[index].Argument2 - 2);
                                            myLines[index - 1] = new ScriptCommand();
                                            myLines[index - 2] = new ScriptCommand();
                                        } else {
                                            myLines[index - 1] = new ScriptCommand(Opcodes.LDI, t);
                                        }
                                        break;
                                    }
                                }
                                if (t == null) {
                                    throw new GustFrontException("識別子 '" + type_name + "' は不明です。");
                                }
                            }
                        }
                    }
                }
                index++;
            }

            index = 1;
            while (index < myLines.Count) {
                ScriptCommand prev = myLines[index - 1];
                ScriptCommand line = myLines[index];
                int baseop = (int)line.Opcode;
                switch (line.Opcode) {
                    case Opcodes.ADD:
                    case Opcodes.SUB:
                    case Opcodes.MUL:
                    case Opcodes.DIV:
                    case Opcodes.REM:
                    case Opcodes.SAL:
                    case Opcodes.SAR:
                    case Opcodes.EQ:
                    case Opcodes.NE:
                    case Opcodes.LT:
                    case Opcodes.GT:
                    case Opcodes.LE:
                    case Opcodes.GE:
                        switch (prev.Opcode) {
                            case Opcodes.LDL:
                                myLines[index] = new ScriptCommand((Opcodes)baseop + 1, prev.Argument1);
                                myLines[index - 1] = new ScriptCommand();
                                break;
                            case Opcodes.LDA:
                                myLines[index] = new ScriptCommand((Opcodes)baseop + 2, prev.Argument1);
                                myLines[index - 1] = new ScriptCommand();
                                break;
                            case Opcodes.LDG:
                                myLines[index] = new ScriptCommand((Opcodes)baseop + 3, prev.Argument1);
                                myLines[index - 1] = new ScriptCommand();
                                break;
                            case Opcodes.LDI:
                                myLines[index] = new ScriptCommand((Opcodes)baseop + 4, null, prev.Argument1);
                                myLines[index - 1] = new ScriptCommand();
                                break;
                        }
                        break;
                }
                index += 1;
            }

            index = 0;
            while (index < myLines.Count) {
                if (myLines[index].Opcode == Opcodes.NOP) {
                    for (int i = 0; i < indexMap.Length; i++) {
                        if (indexMap[i] > index) {
                            for (int j = i; j < indexMap.Length; j++) {
                                indexMap[j]--;
                            }
                            break;
                        }
                    }
                    myLines.RemoveAt(index);
                } else {
                    index++;
                }
            }

            for (int i = 0; i < myLines.Count; i++) {
                switch (myLines[i].Opcode) {
                    case Opcodes.JMP:
                    case Opcodes.JZ:
                    case Opcodes.JNZ:
                        myLines[i].Argument1 = indexMap[GetLabelIndex(myLines[i].Argument1.ToString())];
                        break;
                    case Opcodes.LDI:
                        myLines[i].Argument2 = myLines[i].Argument1;
                        myLines[i].Argument1 = 0;
                        break;
                    case Opcodes.CALL: {
                            object temp = myLines[i].Argument1;
                            myLines[i].Argument1 = myLines[i].Argument2;
                            myLines[i].Argument2 = temp;
                        }
                        break;
                    default:
                        if (!(myLines[i].Argument1 is int)) {
                            myLines[i].Argument1 = 0;
                        }
                        break;
                }

                if (myLines[i].Argument2 is SpecialString) {
                    myLines[i].Argument2 = ((SpecialString)myLines[i].Argument2).ToString();
                    string type_name = myLines[i].Argument2.ToString();
                    for (int j = 0; j < asms.Length; j++) {
                        Type t = asms[j].GetType(type_name, false, true);
                        if (t != null) {
                            myLines[i].Argument2 = t;
                            break;
                        }
                    }
                }
            }
        }

        public ModuleInfo Module
        {
            get { return myModule; }
        }

        public string Name
        {
            get { return myName; }
        }

        public string[] GetArgumentNames()
        {
            return myArgumentNames;
        }

        private int myInnerLabelCount = 0;

        public string DefineLabel(bool mark)
        {
            string name = String.Concat("@", myInnerLabelCount.ToString());
            myInnerLabelCount++;
            myLabels.Add(name, (mark) ? myLines.Count : -1);
            return name;
        }

        public void DefineLabel(string name, bool mark)
        {
            if (!myLabels.ContainsKey(name)) {
                myLabels.Add(name, (mark) ? myLines.Count : -1);
            }
        }

        public void MarkLabel(string name)
        {
            myLabels[name] = myLines.Count;
        }

        private List<string> myCurrentLocalDataNames = new List<string>();
        private Stack<int> myVarBlockIndexes = new Stack<int>();

        public void EnterVarBlock()
        {
            myVarBlockIndexes.Push(myCurrentLocalDataNames.Count);
        }

        public void LeaveVarBlock()
        {
            int index = myVarBlockIndexes.Pop();
            myCurrentLocalDataNames.RemoveRange(index, myCurrentLocalDataNames.Count - index);
        }

        public void DeclareLocal(string name)
        {
            if (Util.CaseInsensitiveContains(myCurrentLocalDataNames, name)) {
                throw new GustFrontException(String.Format("ローカル変数 {0} は既に宣言されています。", name));
            }
            myCurrentLocalDataNames.Add(name);
            myLocalDataNames.Add(name);
        }

        public string[] GetLocalDataNames()
        {
            return myLocalDataNames.ToArray();
        }

        public bool IsValue(string name)
        {
            return Char.IsDigit(name[0]) ||
                Util.CaseInsensitiveContains(myCurrentLocalDataNames, name) ||
                Util.CaseInsensitiveContains(myArgumentNames, name) ||
                Util.CaseInsensitiveContains(myModule.GetGlobalVariableNames(), name);
        }

        public void EmitLoad(string argument)
        {
            if (Char.IsDigit(argument[0])) {
                myLines.Add(new ScriptCommand(Opcodes.LDI, argument));
            } else {
                int var_index = -1;
                if (Util.CaseInsensitiveContains(myCurrentLocalDataNames, argument)) {
                    if ((var_index = Util.CaseInsensitiveLastIndexOf(myLocalDataNames, argument)) != -1) {
                        myLines.Add(new ScriptCommand(Opcodes.LDL, var_index));
                    } else {
                        throw new GustFrontException("内部コンパイラ エラー 0x03 が発生しました。");
                    }
                } else if ((var_index = Util.CaseInsensitiveIndexOf(myArgumentNames, argument)) != -1) {
                    myLines.Add(new ScriptCommand(Opcodes.LDA, var_index));
                } else if ((var_index = Util.CaseInsensitiveIndexOf(myModule.GetGlobalVariableNames(), argument)) != -1) {
                    myLines.Add(new ScriptCommand(Opcodes.LDG, var_index));
                } else {
                    myLines.Add(new ScriptCommand(Opcodes.LDI, argument));
                }
            }
        }

        public void EmitStore(string argument)
        {
            int var_index = -1;
            if (Util.CaseInsensitiveContains(myCurrentLocalDataNames, argument)) {
                if ((var_index = Util.CaseInsensitiveLastIndexOf(myLocalDataNames, argument)) != -1) {
                    myLines.Add(new ScriptCommand(Opcodes.STL, var_index));
                } else {
                    throw new GustFrontException("内部コンパイラ エラー 0x03 が発生しました。");
                }
            } else if ((var_index = Util.CaseInsensitiveIndexOf(myArgumentNames, argument)) != -1) {
                myLines.Add(new ScriptCommand(Opcodes.STA, var_index));
            } else if ((var_index = Util.CaseInsensitiveIndexOf(myModule.GetGlobalVariableNames(), argument)) != -1) {
                myLines.Add(new ScriptCommand(Opcodes.STG, var_index));
            } else {
                throw new GustFrontException(String.Format("{0} には代入できません。", argument));
            }
        }

        public void Emit(Opcodes opcode, object arg1, object arg2)
        {
            myLines.Add(new ScriptCommand(opcode, arg1, arg2));
        }

        public void Emit(Opcodes opcode, object argument)
        {
            myLines.Add(new ScriptCommand(opcode, argument));
        }

        public void Emit(Opcodes opcode)
        {
            myLines.Add(new ScriptCommand(opcode));
        }

        public int GetLabelIndex(string name)
        {
            int index = -1;
            myLabels.TryGetValue(name, out index);
            return index;
        }

        public ScriptCommand GetCommand(int index)
        {
            return myLines[index];
        }

        public void ParseFunc(List<string> tokens)
        {
            Stack<string> aa = new Stack<string>();
            Stack<string> oe = new Stack<string>();

            if (tokens.Count > 0) {
                while (tokens[0] == "(") {
                    if (tokens[tokens.Count - 1] == ")") {
                        tokens.RemoveAt(tokens.Count - 1);
                        tokens.RemoveAt(0);
                    } else {
                        throw new GustFrontException("内部コンパイラ エラー 0x02 が発生しました。");
                    }
                }
            }

            int i = tokens.Count - 1;
            while (i >= 0) {
                if (tokens[i] == "(") {
                    if (tokens[i + 1] != null && tokens[i + 1] != ")") {
                        EmitLoad(tokens[i + 1]);
                        tokens[i + 1] = null;
                    }
                    int start = i - 1;
                    while (i < tokens.Count) {
                        if (tokens[i] == ")") {
                            List<string> part = tokens.GetRange(start, i - start + 1);
                            tokens.RemoveRange(start, part.Count);
                            tokens.Insert(start, null);
                            if (Util.CaseInsensitiveEquals(part[0], "ST")) {
                                int var = Convert.ToInt32(myLines[myLines.Count - 1].Argument1);
                                switch (myLines[myLines.Count - 1].Opcode) {
                                    case Opcodes.LDL:
                                        myLines[myLines.Count - 1] = new ScriptCommand(Opcodes.STL, var);
                                        break;
                                    case Opcodes.LDA:
                                        myLines[myLines.Count - 1] = new ScriptCommand(Opcodes.STA, var);
                                        break;
                                    case Opcodes.LDG:
                                        myLines[myLines.Count - 1] = new ScriptCommand(Opcodes.STG, var);
                                        break;
                                    default:
                                        throw new GustFrontException("変数以外への代入はできません。");
                                }
                            } else if (part[0] == "&2") {
                                string notZero = DefineLabel(false);
                                aa.Push(DefineLabel(false));
                                Emit(Opcodes.JNZ, notZero);
                                Emit(Opcodes.LDI, "0I00000000");
                                Emit(Opcodes.JMP, aa.Peek());
                                MarkLabel(notZero);
                            } else if (part[0] == "&1") {
                                MarkLabel(aa.Pop());
                            } else if (part[0] == "&&") {
                            } else if (part[0] == "|2") {
                                string zero = DefineLabel(false);
                                oe.Push(DefineLabel(false));
                                Emit(Opcodes.JZ, zero);
                                Emit(Opcodes.LDI, "0IFFFFFFFF");
                                Emit(Opcodes.JMP, oe.Peek());
                                MarkLabel(zero);
                            } else if (part[0] == "|1") {
                                MarkLabel(oe.Pop());
                            } else if (part[0] == "||") {
                            } else {
                                if (Enum.IsDefined(typeof(Opcodes), part[0])) {
                                    Emit((Opcodes)Enum.Parse(typeof(Opcodes), part[0]));
                                } else {
                                    Emit(Opcodes.CALL, part[0], (part.Count / 2) - 1);
                                }
                            }
                            i = start;
                            break;
                        } else {
                            i += 1;
                            if (i == tokens.Count) {
                                throw new GustFrontException("内部コンパイラ エラー 0x01 が発生しました。");
                            }
                        }
                    }
                } else if (tokens[i] == "," && tokens[i + 1] != null) {
                    EmitLoad(tokens[i + 1]);
                    tokens[i + 1] = null;
                }
                i -= 1;
            }
            if (tokens.Count > 0 && tokens[0] != null) {
                EmitLoad(tokens[0]);
                tokens[0] = null;
            }
        }

        private static List<Dictionary<string, string>> operators = null;

        static ProcedureInfo()
        {
            operators = new List<Dictionary<string, string>>();

            Dictionary<string, string> op = null;

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add(".", "MA");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, " ");
            op.Add("-", "NEG");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("*", "MUL");
            op.Add("/", "DIV");
            op.Add("%", "REM");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("+", "ADD");
            op.Add("-", "SUB");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("<<", "SAL");
            op.Add(">>", "SAR");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("=", "EQ");
            op.Add("<>", "NE");
            op.Add("<", "LT");
            op.Add(">", "GT");
            op.Add("<=", "LE");
            op.Add(">=", "GE");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, " ");
            op.Add("Not", "NOT");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("And", "AND");
            op.Add("AndAlso", "AND");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("Or", "OR");
            op.Add("OrElse", "OR");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("Xor", "XOR");
            operators.Add(op);

            op = new Dictionary<string, string>();
            op.Add(String.Empty, "  ");
            op.Add("=", "ST");
            op.Add("*=", "ST");
            op.Add("/=", "ST");
            op.Add("%=", "ST");
            op.Add("+=", "ST");
            op.Add("-=", "ST");
            op.Add(">>=", "ST");
            op.Add("<<=", "ST");
            operators.Add(op);
        }

        public static bool IsOperator(string token)
        {
            foreach (Dictionary<string, string> dic in operators) {
                if (dic.ContainsKey(token)) return true;
            }
            return false;
        }

        private static List<string> GetRightOperand(List<string> tokens, int opIndex)
        {
            int right = opIndex + 1;
            if (right <= tokens.Count - 1) {
                int nest = 0;
                if (tokens[right] == "(") {
                    right += 1;
                    nest = 1;
                } else if (right < tokens.Count - 1 && tokens[right + 1] == "(") {
                    right += 2;
                    nest = 1;
                }
                if (nest > 0) {
                    while (right < tokens.Count) {
                        if (tokens[right] == ")") {
                            nest -= 1;
                            if (nest == 0) break;
                        } else if (tokens[right] == "(") {
                            nest += 1;
                        }
                        right += 1;
                    }
                }
            } else {
                throw new GustFrontException("右オペランドがありません。");
            }

            return tokens.GetRange(opIndex + 1, right - opIndex);
        }

        private static List<string> GetLeftOperand(List<string> tokens, int opIndex)
        {
            int left = opIndex - 1;
            if (left >= 0) {
                if (tokens[left] == ")") {
                    int nest = 1;
                    left -= 1;
                    while (left >= 0) {
                        if (tokens[left] == "(") {
                            nest -= 1;
                            if (nest == 0) break;
                        } else if (tokens[left] == ")") {
                            nest += 1;
                        }
                        left -= 1;
                    }
                    if (left > 0 && !(tokens[left - 1] == "(" || tokens[left - 1] == "," || IsOperator(tokens[left - 1]))) {
                        left -= 1;
                    }
                }
            } else {
                throw new GustFrontException("左オペランドがありません。");
            }

            return tokens.GetRange(left, opIndex - left);
        }

        private static void Op1ToFunc(List<string> tokens, Dictionary<string, string> op)
        {
            int i = 0;
            while (i < tokens.Count) {
                if (op.ContainsKey(tokens[i]) && (i == 0 || tokens[i - 1] == "," || tokens[i - 1] == "(" || IsOperator(tokens[i - 1]))) {
                    List<string> rv = GetRightOperand(tokens, i);
                    int rlen = rv.Count;
                    if (rv[0] == "(") {
                        rv.RemoveAt(rv.Count - 1);
                        rv.RemoveAt(0);
                    }
                    List<string> func = new List<string>(rv.Count + 3);
                    func.Add(op[tokens[i]]);
                    func.Add("(");
                    func.AddRange(rv);
                    func.Add(")");
                    tokens.RemoveRange(i, rlen + 1);
                    tokens.InsertRange(i, func);
                    i = 0;
                }
                i += 1;
            }
        }

        private static void Op2ToFunc(bool cond, List<string> tokens, Dictionary<string, string> op)
        {
            if (!cond && op.ContainsKey("<>")) {
                int depth = 0;
                for (int index = 0; index < tokens.Count; index++) {
                    if (tokens[index] == "(") {
                        depth += 1;
                    } else if (tokens[index] == ")") {
                        depth -= 1;
                    } else if (tokens[index] == "=" && depth == 0) {
                        tokens[index] = ":=";
                        break;
                    }
                }
            }

            int i = 0;
            while (i < tokens.Count) {

                if (op.ContainsKey(tokens[i])) {
                    List<string> lv = GetLeftOperand(tokens, i);
                    int llen = lv.Count;
                    if (lv[0] == "(") {
                        lv.RemoveAt(lv.Count - 1);
                        lv.RemoveAt(0);
                    }
                    List<string> rv = GetRightOperand(tokens, i);
                    int rlen = rv.Count;
                    if (rv[0] == "(") {
                        rv.RemoveAt(rv.Count - 1);
                        rv.RemoveAt(0);
                    }

                    List<string> func = new List<string>(lv.Count + rv.Count + 4);
                    if (Util.CaseInsensitiveEquals(tokens[i], "AndAlso")) {
                        func.AddRange(new string[] { "&&", "(", "&1", "(" });
                        func.AddRange(rv);
                        func.AddRange(new string[] { ")", ",", "&2", "(" });
                        func.AddRange(lv);
                        func.AddRange(new string[] { ")", ")" });
                    } else if (Util.CaseInsensitiveEquals(tokens[i], "OrElse")) {
                        func.AddRange(new string[] { "||", "(", "|1", "(" });
                        func.AddRange(rv);
                        func.AddRange(new string[] { ")", ",", "|2", "(" });
                        func.AddRange(lv);
                        func.AddRange(new string[] { ")", ")" });
                    } else if (op.ContainsKey("+=")) {
                        func.Add(op[tokens[i]]);
                        func.Add("(");
                        func.AddRange(lv);
                        func.Add(",");
                        func.AddRange(rv);
                        func.Add(")");
                    } else if (op.ContainsKey(".")) {
                        func.Add("GetMember");
                        func.Add("(");
                        func.AddRange(lv);
                        func.Add(",");
                        func.AddRange(rv);
                        func[func.Count - rv.Count] += ".";
                        if (rv.Count > 3) {
                            func[func.Count - rv.Count + 1] = ",";
                        } else if (rv.Count == 3) {
                            func.RemoveAt(func.Count - 2);
                        } else {
                            func.Add(")");
                        }
                    } else {
                        func.Add(op[tokens[i]]);
                        func.Add("(");
                        func.AddRange(rv);
                        func.Add(",");
                        func.AddRange(lv);
                        func.Add(")");
                    }
                    tokens.RemoveRange(i - llen, rlen + llen + 1);
                    tokens.InsertRange(i - llen, func);
                    i = 0;
                }
                i += 1;
            }

            for (int index = 0; index < tokens.Count; index++) {
                if (tokens[index] == ":=") {
                    tokens[index] = "=";
                    break;
                }
            }
        }

        private static void OpToFunc(bool mustReturnValue, List<string> tokens)
        {
            for (int i = 0; i < operators.Count - 1; i++) {
                if (operators[i][String.Empty].Length == 2) {
                    Op2ToFunc(mustReturnValue, tokens, operators[i]);
                } else {
                    Op1ToFunc(tokens, operators[i]);
                }
            }

            if (!mustReturnValue) {
                Dictionary<string, string> letOps = operators[operators.Count - 1];
                for (int i = 0; i < tokens.Count; i++) {
                    if (tokens[i] != String.Empty && letOps.ContainsKey(tokens[i])) {
                        List<string> lv = tokens.GetRange(i + 1, tokens.Count - i - 1);
                        List<string> rv = tokens.GetRange(0, i);

                        List<string> func = new List<string>(tokens.Count * 2);
                        if (tokens[0] != "GetMember") {
                            func.Add("ST");
                            func.Add("(");
                            func.AddRange(rv);
                            func.Add(",");
                            if (tokens[i].Length == 1) {
                                func.AddRange(lv);
                            } else {
                                switch (tokens[i][0]) {
                                    case '+':
                                        func.Add("ADD");
                                        break;
                                    case '-':
                                        func.Add("SUB");
                                        break;
                                    case '*':
                                        func.Add("MUL");
                                        break;
                                    case '/':
                                        func.Add("DIV");
                                        break;
                                    case '%':
                                        func.Add("REM");
                                        break;
                                    case '<':
                                        func.Add("SAL");
                                        break;
                                    case '>':
                                        func.Add("SAR");
                                        break;
                                }
                                func.Add("(");
                                func.AddRange(lv);
                                func.Add(",");
                                func.AddRange(rv);
                                func.Add(")");
                            }
                            func.Add(")");

                            tokens.Clear();
                            tokens.AddRange(func);
                        } else {
                            func.Add("SetMember");
                            func.Add("(");
                            func.AddRange(rv.GetRange(2, rv.Count - 3));
                            func.Add(",");
                            if (tokens[i].Length == 1) {
                                func.AddRange(lv);
                            } else {
                                switch (tokens[i][0]) {
                                    case '+':
                                        func.Add("ADD");
                                        break;
                                    case '-':
                                        func.Add("SUB");
                                        break;
                                    case '*':
                                        func.Add("MUL");
                                        break;
                                    case '/':
                                        func.Add("DIV");
                                        break;
                                    case '%':
                                        func.Add("REM");
                                        break;
                                    case '<':
                                        func.Add("SAL");
                                        break;
                                    case '>':
                                        func.Add("SAR");
                                        break;
                                }
                                func.Add("(");
                                func.AddRange(lv);
                                func.Add(",");
                                func.AddRange(rv);
                                func.Add(")");
                            }
                            func.Add(")");
                            tokens.Clear();
                            tokens.AddRange(func);
                        }
                        break;
                    }
                }
                for (int i = 0; i < tokens.Count; i++) {
                    if (tokens[i] != String.Empty && letOps.ContainsKey(tokens[i])) {
                        throw new GustFrontException("不正な代入です。");
                    }
                }
            }
        }

        internal void ParseExpression(List<string> tokens, List<string> lblDo, List<string> lblFor)
        {
            if (Util.CaseInsensitiveEquals(tokens[0], "Return")) {
                tokens.RemoveAt(0);
                if (tokens.Count > 0) {
                    Module.ExpandAlias(this, tokens);
                    OpToFunc(true, tokens);
                    ParseFunc(tokens);
                    Emit(Opcodes.RET, 1);
                } else {
                    Emit(Opcodes.LDI, null);
                    Emit(Opcodes.RET, 1);
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Exit")) {
                if (tokens.Count > 1) {
                    if (Util.CaseInsensitiveEquals(tokens[1], "Do")) {
                        Emit(Opcodes.JMP, lblDo[lblDo.Count - 1]);
                    } else if (Util.CaseInsensitiveEquals(tokens[1], "For")) {
                        Emit(Opcodes.JMP, lblFor[lblFor.Count - 1]);
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Continue")) {
                if (tokens.Count > 1) {
                    if (Util.CaseInsensitiveEquals(tokens[1], "Do")) {
                        Emit(Opcodes.JMP, lblDo[lblDo.Count - 2]);
                    } else if (Util.CaseInsensitiveEquals(tokens[1], "For")) {
                        Emit(Opcodes.JMP, lblFor[lblFor.Count - 2]);
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "GoTo")) {
                Emit(Opcodes.JMP, tokens[1]);
            } else {
                Module.ExpandAlias(this, tokens);
                OpToFunc(false, tokens);
                ParseFunc(tokens);

                if (myLines[myLines.Count - 1].Opcode < Opcodes.STL || Opcodes.STG < myLines[myLines.Count - 1].Opcode) {
                    Emit(Opcodes.POP);
                }
            }
        }

        internal void ParseCondition(List<string> tokens)
        {
            Module.ExpandAlias(this, tokens);
            OpToFunc(true, tokens);
            ParseFunc(tokens);
        }

        public void ParseStatement(List<string> tokens, Stack<ModuleInfo.BlockMode> mode, Stack<string> lblIf, List<string> lblDo, List<string> lblFor, List<string> varFor)
        {
            if (Util.CaseInsensitiveEquals(tokens[0], "If")) {
                int[] indexes = ModuleInfo.GetIndexes(tokens, "Then", "Else");
                if (indexes[0] != -1) {
                    if (indexes[1] != -1) {
                        string at3 = DefineLabel(false);
                        string at4 = DefineLabel(false);
                        ParseCondition(tokens.GetRange(1, indexes[0] - 1));
                        Emit(Opcodes.JZ, at3);
                        ParseExpression(tokens.GetRange(indexes[0] + 1, indexes[1] - indexes[0]), lblDo, lblFor);
                        Emit(Opcodes.JMP, at4);
                        MarkLabel(at3);
                        ParseExpression(tokens.GetRange(indexes[1] + 1, tokens.Count - indexes[1] - 1), lblDo, lblFor);
                        MarkLabel(at4);
                    } else if (indexes[0] < tokens.Count - 1) {
                        string at3 = DefineLabel(false);
                        ParseCondition(tokens.GetRange(1, indexes[0] - 1));
                        Emit(Opcodes.JZ, at3);
                        ParseExpression(tokens.GetRange(indexes[0] + 1, tokens.Count - indexes[0] - 1), lblDo, lblFor);
                        MarkLabel(at3);
                    } else {
                        EnterVarBlock();
                        string atx = DefineLabel(false);
                        string at2 = DefineLabel(false);
                        ParseCondition(tokens.GetRange(1, indexes[0] - 1));
                        Emit(Opcodes.JZ, at2);
                        lblIf.Push(null);
                        lblIf.Push(atx);
                        lblIf.Push(at2);
                        mode.Push(ModuleInfo.BlockMode.If);
                    }
                } else {
                    throw new GustFrontException("構文エラー");
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "ElseIf")) {
                LeaveVarBlock();
                if (mode.Peek() == ModuleInfo.BlockMode.If) {
                    int[] indexes = ModuleInfo.GetIndexes(tokens, "Then");
                    if (indexes[0] != -1) {
                        string at0 = lblIf.Pop();
                        string at2 = DefineLabel(false);
                        Emit(Opcodes.JMP, lblIf.Peek());
                        MarkLabel(at0);
                        ParseCondition(tokens.GetRange(1, indexes[0] - 1));
                        Emit(Opcodes.JZ, at2);
                        lblIf.Push(at2);
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                } else {
                    throw new GustFrontException("構文エラー");
                }
                EnterVarBlock();
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Else")) {
                LeaveVarBlock();
                if (mode.Peek() == ModuleInfo.BlockMode.If) {
                    string at0 = lblIf.Pop();
                    Emit(Opcodes.JMP, lblIf.Peek());
                    MarkLabel(at0);
                } else {
                    throw new GustFrontException("構文エラー");
                }
                EnterVarBlock();
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Do")) {
                EnterVarBlock();
                if (tokens.Count > 1) {
                    string at0 = DefineLabel(true);
                    string at2 = DefineLabel(false);
                    if (Util.CaseInsensitiveEquals(tokens[1], "While")) {
                        ParseCondition(tokens.GetRange(2, tokens.Count - 2));
                        Emit(Opcodes.JZ, at2);
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                    lblDo.Add(at0);
                    lblDo.Add(at2);
                    mode.Push(ModuleInfo.BlockMode.DoWhile);
                } else {
                    lblDo.Add(DefineLabel(true));
                    lblDo.Add(DefineLabel(false));
                    lblDo.Add(DefineLabel(false));
                    mode.Push(ModuleInfo.BlockMode.Do);
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Loop")) {
                if (mode.Peek() == ModuleInfo.BlockMode.DoWhile) {
                    Emit(Opcodes.JMP, lblDo[lblDo.Count - 2]);
                    MarkLabel(lblDo[lblDo.Count - 1]);
                    lblDo.RemoveRange(lblDo.Count - 2, 2);
                    mode.Pop();
                } else if (mode.Peek() == ModuleInfo.BlockMode.Do) {
                    MarkLabel(lblDo[lblDo.Count - 2]);
                    if (tokens.Count > 1) {
                        if (Util.CaseInsensitiveEquals(tokens[1], "While")) {
                            ParseCondition(tokens.GetRange(2, tokens.Count - 2));
                            Emit(Opcodes.JNZ, lblDo[lblDo.Count - 3]);
                        } else {
                            throw new GustFrontException("構文エラー");
                        }
                    } else {
                        Emit(Opcodes.JMP, lblDo[lblDo.Count - 3]);
                    }
                    MarkLabel(lblDo[lblDo.Count - 1]);
                    lblDo.RemoveRange(lblDo.Count - 3, 3);
                    mode.Pop();
                } else {
                    throw new GustFrontException("構文エラー");
                }
                LeaveVarBlock();
            } else if (Util.CaseInsensitiveEquals(tokens[0], "For")) {
                EnterVarBlock();
                mode.Push(ModuleInfo.BlockMode.For);
                if (tokens.Count > 5) {
                    varFor.Add(tokens[1]);
                    if (!IsValue(tokens[1])) DeclareLocal(tokens[1]);
                    if (Util.CaseInsensitiveEquals(tokens[2], "=")) {
                        int[] indexes = ModuleInfo.GetIndexes(tokens, "To", "Step");
                        if (indexes[0] >= 4) {
                            varFor.Add(String.Concat("@", "F", varFor.Count.ToString()));
                            DeclareLocal(varFor[varFor.Count - 1]);
                            varFor.Add(String.Concat("@", "F", varFor.Count.ToString()));
                            DeclareLocal(varFor[varFor.Count - 1]);
                            if (indexes[1] != -1) {
                                ParseCondition(tokens.GetRange(indexes[0] + 1, indexes[1] - indexes[0]));
                                EmitStore(varFor[varFor.Count - 2]);
                                ParseCondition(tokens.GetRange(indexes[1] + 1, tokens.Count - indexes[1] - 1));
                                EmitStore(varFor[varFor.Count - 1]);
                            } else {
                                ParseCondition(tokens.GetRange(indexes[0] + 1, tokens.Count - indexes[0] - 1));
                                EmitStore(varFor[varFor.Count - 2]);
                                Emit(Opcodes.LDI, "0I01000000");
                                EmitStore(varFor[varFor.Count - 1]);
                            }
                            ParseCondition(tokens.GetRange(3, indexes[0] - 3));
                            EmitStore(varFor[varFor.Count - 3]);
                            lblFor.Add(DefineLabel(true));
                            lblFor.Add(DefineLabel(false));
                            lblFor.Add(DefineLabel(false));
                            if (indexes[1] != -1) {
                                EmitLoad(varFor[varFor.Count - 1]);
                                Emit(Opcodes.LDI, "0I1F000000");
                                Emit(Opcodes.SAR);
                                EmitLoad(varFor[varFor.Count - 3]);
                                Emit(Opcodes.XOR);
                                EmitLoad(varFor[varFor.Count - 1]);
                                Emit(Opcodes.LDI, "0I1F000000");
                                Emit(Opcodes.SAR);
                                EmitLoad(varFor[varFor.Count - 2]);
                                Emit(Opcodes.XOR);
                            } else {
                                EmitLoad(varFor[varFor.Count - 3]);
                                EmitLoad(varFor[varFor.Count - 2]);
                            }
                            Emit(Opcodes.LE);
                            Emit(Opcodes.JZ, lblFor[lblFor.Count - 1]);
                        } else {
                            throw new GustFrontException("構文エラー");
                        }
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                } else {
                    throw new GustFrontException("構文エラー");
                }
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Next")) {
                if (mode.Pop() == ModuleInfo.BlockMode.For) {
                    MarkLabel(lblFor[lblFor.Count - 2]);
                    EmitLoad(varFor[varFor.Count - 3]);
                    EmitLoad(varFor[varFor.Count - 1]);
                    Emit(Opcodes.ADD);
                    EmitStore(varFor[varFor.Count - 3]);
                    Emit(Opcodes.JMP, lblFor[lblFor.Count - 3]);
                    MarkLabel(lblFor[lblFor.Count - 1]);
                    lblFor.RemoveRange(lblFor.Count - 3, 3);
                    varFor.RemoveRange(varFor.Count - 3, 3);
                } else {
                    throw new GustFrontException("構文エラー");
                }
                LeaveVarBlock();
            } else if (Util.CaseInsensitiveEquals(tokens[0], "Dim")) {
                if (tokens.Count > 1) {
                    if (tokens.Count > 2 && Util.CaseInsensitiveEquals(tokens[2], "=")) {
                        if (tokens.Count > 3) {
                            ParseCondition(tokens.GetRange(3, tokens.Count - 3));
                            DeclareLocal(tokens[1]);
                            EmitStore(tokens[1]);
                        } else {
                            throw new GustFrontException("構文エラー");
                        }
                    } else {
                        DeclareLocal(tokens[1]);
                    }
                } else {
                    throw new GustFrontException("構文エラー");
                }
            } else {
                ParseExpression(tokens.GetRange(0, tokens.Count), lblDo, lblFor);
            }
        }
    }

    /// <summary>
    /// モジュールを解析し、その結果を保持する。
    /// </summary>
    public class ModuleInfo
    {
        internal const string MainRoutine = "Main";

        internal enum BlockMode
        {
            Reference = 0,
            Constant,
            Import,
            Procedure,
            If,
            DoWhile,
            Do,
            For
        }

        private string myFilePath = null;
        private Dictionary<string, Dictionary<string, string>> myConstants = CollectionUtil.CreateCaseInsensitiveDictionary<Dictionary<string, string>>();
        private Dictionary<string, string[]> myImports = CollectionUtil.CreateCaseInsensitiveDictionary<string[]>();
        private List<string> myUsingNamespaces = new List<string>();
        private List<ModuleInfo> myRefMods = new List<ModuleInfo>();
        private Dictionary<string, ModuleInfo> myModRootNs = new Dictionary<string, ModuleInfo>();
        private Dictionary<string, ProcedureInfo> myProcedures = CollectionUtil.CreateCaseInsensitiveDictionary<ProcedureInfo>();
        private List<string> myGlobalVariables = new List<string>();
        private ProcedureInfo myGlobalInitializer = null;
        private Assembly[] asms = AppDomain.CurrentDomain.GetAssemblies();

        private ModuleInfo()
        {
        }

        internal string FilePath
        {
            get { return myFilePath; }
            set { myFilePath = value; }
        }

        internal IEnumerable<string> UsingNamespaces
        {
            get { return myUsingNamespaces; }
        }

        internal IDictionary<string, ModuleInfo> ModRootNs
        {
            get { return myModRootNs; }
        }

        internal bool HasImported(string name)
        {
            foreach (string[] imps in myImports.Values) {
                if (Util.CaseInsensitiveContains(imps, name)) return true;
            }
            return false;
        }

        internal string[] GetGlobalVariableNames()
        {
            return myGlobalVariables.ToArray();
        }
        internal int GetGlobalVariableCount()
        {
            return myGlobalVariables.Count;
        }

        internal ProcedureInfo GetGlobalInitializer()
        {
            return myGlobalInitializer;
        }

        internal bool HasProcedure(string routineName)
        {
            return myProcedures.ContainsKey(routineName);
        }

        internal ProcedureInfo GetProcedure(string routineName)
        {
            ProcedureInfo info = null;
            if (myProcedures.TryGetValue(routineName, out info)) {
                return info;
            } else {
                throw new GustFrontException(String.Format("プロシージャ {0} が見つかりませんでした。", routineName));
            }
        }

        internal IDictionary<string, ProcedureInfo> GetProcedures()
        {
            return myProcedures;
        }

        static internal int[] GetIndexes(List<string> tokens, params string[] keywords)
        {
            int[] result = new int[keywords.Length];
            for (int i = 0; i < result.Length; i++) {
                result[i] = -1;
            }

            for (int ki = 0; ki < keywords.Length; ki++) {
                for (int i = 0; i < tokens.Count; i++) {
                    if (Util.CaseInsensitiveEquals(tokens[i], keywords[ki])) {
                        if (result[ki] == -1) {
                            result[ki] = i;
                        } else {
                            throw new GustFrontException("構文エラー");
                        }
                    }
                }
            }

            int last = -1;
            foreach (int index in result) {
                if (index != -1) {
                    if (last < index) {
                        last = index;
                    } else {
                        throw new GustFrontException("構文エラー");
                    }
                }
            }

            return result;
        }

        private static int EncodeNumeric(string line, int start, List<string> tokens)
        {
            bool is_minus = false;
            if (tokens.Count >= 1 && tokens[tokens.Count - 1] == "-") {
                if (tokens.Count >= 2) {
                    string last_token = tokens[tokens.Count - 2];
                    is_minus = last_token == "(" || last_token == "," || ProcedureInfo.IsOperator(last_token);
                } else {
                    is_minus = true;
                }
            }

            char c = Char.ToUpperInvariant(line[start]);
            char nc = (start < line.Length - 1) ? Char.ToUpperInvariant(line[start + 1]) : '\0';
            char suffix = '\0';

            if (c == '0' && nc == 'X') { // is a hex?
                int num_end = -1;
                for (int i = start + 2; i < line.Length; i++) {
                    c = Char.ToUpperInvariant(line[i]);
                    nc = (i < line.Length - 1) ? Char.ToUpperInvariant(line[i + 1]) : '\0';
                    if (('0' <= c && c <= '9') || ('A' <= c && c <= 'F')) {
                        num_end = i;
                    } else {
                        if (c == 'L' || c == 'I' || c == 'S' || c == 'B') {
                            if (nc != '.' && IsSeparator(nc)) {
                                num_end = i;
                            } else {
                                num_end = -1;
                            }
                            suffix = c;
                            break;
                        } else {
                            if (c == '.' || !IsSeparator(c)) {
                                num_end = -1;
                            }
                            break;
                        }
                    }
                }
                if (num_end != -1) {
                    UInt64 temp = UInt64.Parse(
                        line.Substring(start + 2, num_end - start - ((suffix == '\0') ? 1 : 2)),
                        NumberStyles.HexNumber, CultureInfo.InvariantCulture);
                    if (is_minus && temp != 0) {
                        temp = ~temp + 1;
                        tokens.RemoveAt(tokens.Count - 1);
                    }
                    byte[] ba = BitConverter.GetBytes(temp);
                    StringBuilder isb = new StringBuilder(16);
                    isb.Append(new char[] { '0', (suffix == '\0') ? 'I' : suffix });
                    for (int i = 0; i < ba.Length; i++) {
                        isb.Append(ba[i].ToString("X2"));
                    }
                    tokens.Add(isb.ToString());
                    return num_end;
                } else {
                    throw new GustFrontException("不正な形式の 16 進数です。");
                }
            } else {
                bool dot_is = false;
                bool exp_is = false;
                int num_end = -1;
                for (int i = start; i < line.Length; i++) {
                    c = Char.ToUpperInvariant(line[i]);
                    nc = (i < line.Length - 1) ? Char.ToUpperInvariant(line[i + 1]) : '\0';
                    if (num_end == -1) {
                        if ('0' <= c && c <= '9') {
                            num_end = i;
                        } else {
                            break;
                        }
                    } else if ('0' <= c && c <= '9') {
                        num_end = i;
                    } else if (c == '.') {
                        if (!dot_is) {
                            dot_is = true;
                            num_end = -1;
                        } else {
                            throw new GustFrontException("不正な形式の小数です。");
                        }
                    } else if (c == 'E') {
                        if (!exp_is) {
                            if (nc == '+' || nc == '-') {
                                i++;
                            }
                            dot_is = true;
                            exp_is = true;
                            num_end = -1;
                        } else {
                            throw new GustFrontException("不正な形式の小数です。");
                        }
                    } else {
                        if (c == 'L' || c == 'I' || c == 'S' || c == 'B') {
                            if (!dot_is) {
                                if (nc != '.' && IsSeparator(nc)) {
                                    num_end = i;
                                } else {
                                    num_end = -1;
                                }
                                suffix = c;
                                break;
                            } else {
                                throw new GustFrontException("不正な形式の整数です。");
                            }
                        } else if (c == 'R' || c == 'F' || c == 'D') {
                            if (nc != '.' && IsSeparator(nc)) {
                                dot_is = true;
                                num_end = i;
                            } else {
                                num_end = -1;
                            }
                            suffix = c;
                            break;
                        } else {
                            if (c == '.' || !IsSeparator(c)) {
                                num_end = -1;
                            }
                            break;
                        }
                    }
                }
                if (num_end != -1) {
                    if (!dot_is) {
                        UInt64 temp = UInt64.Parse(
                            line.Substring(start, num_end - start + ((suffix == '\0') ? 1 : 0)),
                            NumberStyles.Integer, CultureInfo.InvariantCulture);
                        if (is_minus && temp != 0) {
                            temp = ~temp + 1;
                            tokens.RemoveAt(tokens.Count - 1);
                        }
                        byte[] ba = BitConverter.GetBytes(temp);
                        StringBuilder isb = new StringBuilder(16);
                        isb.Append(new char[] { '0', (suffix == '\0') ? 'I' : suffix });
                        for (int i = 0; i < ba.Length; i++) {
                            isb.Append(ba[i].ToString("X2"));
                        }
                        tokens.Add(isb.ToString());
                    } else {
                        byte[] ba = null;
                        StringBuilder isb = new StringBuilder(32);
                        switch (suffix) {
                            case 'D':
                                decimal tempd = Decimal.Parse(
                                    line.Substring(start, num_end - start + ((suffix == '\0') ? 1 : 0)),
                                    NumberStyles.Float, CultureInfo.InvariantCulture);
                                if (is_minus) tempd = -tempd;
                                int[] dp = Decimal.GetBits(tempd);
                                ba = new byte[4 * dp.Length];
                                for (int i = 0; i < dp.Length; i++) {
                                    Array.Copy(BitConverter.GetBytes(dp[i]), 0, ba, i * 4, 4);
                                }
                                break;
                            case 'F':
                                float tempf = Single.Parse(
                                    line.Substring(start, num_end - start + ((suffix == '\0') ? 1 : 0)),
                                    NumberStyles.Float, CultureInfo.InvariantCulture);
                                ba = BitConverter.GetBytes((!is_minus) ? tempf : -tempf);
                                break;
                            default:
                                double temp = Double.Parse(
                                    line.Substring(start, num_end - start + ((suffix == '\0') ? 1 : 0)),
                                    NumberStyles.Float, CultureInfo.InvariantCulture);
                                ba = BitConverter.GetBytes((!is_minus) ? temp : -temp);
                                break;
                        }
                        isb.Append(new char[] { '0', (suffix == '\0') ? 'R' : suffix });
                        for (int i = 0; i < ba.Length; i++) {
                            isb.Append(ba[i].ToString("X2"));
                        }
                        if (is_minus) tokens.RemoveAt(tokens.Count - 1);
                        tokens.Add(isb.ToString());
                    }
                    return num_end;
                } else {
                    if (!dot_is) {
                        throw new GustFrontException("不正な形式の整数です。");
                    } else {
                        throw new GustFrontException("不正な形式の小数です。");
                    }
                }
            }
        }

        private static int EncodeString(string line, int start, List<string> tokens)
        {
            int endOfString = start;

            StringBuilder str = new StringBuilder(line.Length);

            for (int i = start + 1; i < line.Length; i++) {
                if (line[i] == '"') {
                    if (i == line.Length - 1 || line[i + 1] != '"') {
                        endOfString = i;
                        break;
                    } else {
                        i += 1;
                    }
                }
                if (i < line.Length - 1) {
                    str.Append(line[i]);
                } else {
                    throw new GustFrontException("文字列が閉じていません。");
                }
            }

            StringBuilder isb = new StringBuilder();
            isb.Append(new char[] { '0', 'T' });
            char[] ca = str.ToString().ToCharArray();
            for (int i = 0; i < ca.Length; i++) {
                byte[] ba = BitConverter.GetBytes(ca[i]);
                isb.Append(ba[0].ToString("X2"));
                isb.Append(ba[1].ToString("X2"));
            }
            tokens.Add(isb.ToString());

            return endOfString;
        }

        private static string DecodeString(string token)
        {
            byte[] ba = new byte[token.Length / 2 - 1];
            for (int i = 2; i < token.Length; i += 2) {
                ba[i / 2 - 1] = Byte.Parse(token.Substring(i, 2),
                    NumberStyles.HexNumber, CultureInfo.InvariantCulture);
            }
            StringBuilder sb = new StringBuilder(token.Length / 2 - 1);
            for (int i = 0; i < ba.Length; i += 2) {
                sb.Append(BitConverter.ToChar(ba, i));
            }
            return sb.ToString();
        }

        private static char[] symbolic_chars = 
        {
            '+',
            '-',
            '*',
            '/',
            '%',
            '=',
            '<',
            '>',
            '(',
            ')',
            '[',
            ']',
            '{',
            '}',
            ',',
            '.'
        };

        private static bool IsSymbol(char c)
        {
            return Array.IndexOf(symbolic_chars, c) != -1;
        }
        internal ModuleInfo[] GetReferences()
        {
            return myRefMods.ToArray();
        }

        private static bool IsSeparator(char c)
        {
            if (Char.IsWhiteSpace(c)) return true;

            switch (c) {
                case '\0':
                case '\'':
                case '.':
                case ',':
                case '(':
                case ')':
                case '[':
                case ']':
                case '{':
                case '}':
                case '+':
                case '-':
                case '*':
                case '/':
                case '%':
                case '=':
                case '<':
                case '>':
                    return true;
            }

            return false;
        }

        private Type SearchType(string type_name)
        {
            Type t = null;
            for (int i = 0; i < asms.Length; i++) {
                t = asms[i].GetType(type_name, false, true);
                if (t != null) {
                    return t;
                } else {
                    foreach (string ns in myUsingNamespaces) {
                        t = asms[i].GetType(String.Concat(ns, ".", type_name), false, true);
                        if (t != null) return t;
                    }
                }
            }
            return t;
        }

        internal List<string> ExpandAlias(ProcedureInfo proc, List<string> tokens)
        {
            int index = 0;

            while (index < tokens.Count) {
                if (tokens[index] == "[") {
                    tokens[index] = "(";
                    if (index > 0) {
                        string last_token = tokens[index - 1];
                        if (last_token == "(" || last_token == "," || ProcedureInfo.IsOperator(last_token)) {
                            tokens.Insert(index, "GetElement");
                            List<string> sb = new List<string>(1);
                            EncodeString(String.Concat("\"", tokens[index + 2], "\""), 0, sb);
                            tokens[index + 2] = sb[0];
                            index += 2;
                        } else {
                            tokens.Insert(index, "Default");
                            tokens.Insert(index, ".");
                            index += 3;
                        }
                    } else {
                        tokens.Insert(index, "GetElement");
                        List<string> sb = new List<string>(1);
                        EncodeString(String.Concat("\"", tokens[index + 2], "\""), 0, sb);
                        tokens[index + 2] = sb[0];
                        index += 2;
                    }
                } else if (tokens[index] == "]") {
                    tokens[index] = ")";
                    index++;
                } else if (tokens[index] == "{") {
                    tokens[index] = "(";
                    tokens.Insert(index, "CreateList");
                    index += 2;
                } else if (tokens[index] == "}") {
                    tokens[index] = ")";
                    index++;
                } else {
                    index++;
                }
            }

            index = 0;
            while (index < tokens.Count) {
                string real_name = null;
                if (index >= 2 && tokens[index - 1] == "." && myConstants.ContainsKey(tokens[index - 2]) &&
                    myConstants[tokens[index - 2]].TryGetValue(tokens[index], out real_name)) {
                    tokens.RemoveRange(index - 2, 2);
                    index -= 2;
                    tokens[index] = real_name;
                } else {
                    string[] real_names = null;
                    if (myImports.TryGetValue(tokens[index], out real_names)) {
                        tokens.RemoveAt(index);
                        tokens.InsertRange(index, real_names);
                    }
                }
                index++;
            }

            index = 0;
            bool canStart = true;
            while (index < tokens.Count) {
                if (canStart) {
                    string s = tokens[index];
                    if (s != ")" && !ProcedureInfo.IsOperator(s) && !proc.IsValue(s)) {
                        Type t = SearchType(s);
                        int i = index;
                        while (i < tokens.Count - 2 && tokens[i + 1] == ".") {
                            s = String.Concat(s, ".", tokens[i + 2]);
                            Type _t = SearchType(s);
                            if (_t != null) {
                                t = _t;
                            } else if (t != null) {
                                break;
                            }
                            i += 2;
                        }
                        if (t == null) {
                            if (index >= tokens.Count - 2 || tokens[index + 1] != ".") {
                                index = i + 1;
                            } else {
                                tokens[index] = String.Concat(tokens[index], ".", tokens[index + 2]);
                                tokens.RemoveRange(index + 1, 2);
                                index++;
                            }
                        } else {
                            if (i > index) {
                                tokens.RemoveRange(index, i - index + 1);
                                tokens.Insert(index, t.FullName);
                            } else {
                                tokens[index] = t.FullName;
                            }
                            index += 3;
                        }
                        canStart = false;
                    } else {
                        index++;
                        canStart = (s == "(" || s == "," || (s != "." && ProcedureInfo.IsOperator(s)));
                    }
                } else {
                    string s = tokens[index++];
                    canStart = (s == "(" || s == "," || (s != "." && ProcedureInfo.IsOperator(s)));
                }
            }

            return tokens;
        }

        private void ParseImpl(StreamReader reader, Converter<string, Stream> stream_getter, ref int lineNumber)
        {
            int extln = 0;

            Stack<BlockMode> mode = new Stack<BlockMode>();
            Stack<string> lblIf = new Stack<string>();
            List<string> lblDo = new List<string>();
            List<string> lblFor = new List<string>();
            List<string> varFor = new List<string>();
            ProcedureInfo proc = null;
            string constHead = null;

            List<string> tokens = new List<string>(32);

            string line = reader.ReadLine();
            while (line != null) {
                lineNumber += 1;

                if (proc != null) {
                    proc.Emit(Opcodes.DBG, 0, new object[] { lineNumber });
                }

                tokens.Clear();

                Stack<int> brackets = new Stack<int>(16);
                int i = 0;
                while (i < line.Length) {
                    char c = line[i];
                    char nc = (i < line.Length - 1) ? line[i + 1] : '\0';
                    if (c == '\'') {
                        break;
                    } else if (c == '"') {
                        i = EncodeString(line, i, tokens);
                    } else if ('0' <= c && c <= '9') {
                        i = EncodeNumeric(line, i, tokens);
                    } else if (c == '_' && i == line.Length - 1 && i > 0 && Char.IsWhiteSpace(line[i - 1])) {
                        string nl = reader.ReadLine();
                        if (nl != null && nl.Length > 0) {
                            line = nl;
                            i = 0;
                            extln += 1;
                            continue;
                        } else {
                            throw new GustFrontException("連結する行がありません。");
                        }
                    } else if (IsSymbol(c)) {
                        switch (c) {
                            case '(':
                                brackets.Push(0);
                                tokens.Add("(");
                                break;
                            case ')':
                                if (brackets.Count > 0 && brackets.Pop() == 0) {
                                    tokens.Add(")");
                                } else {
                                    throw new GustFrontException("'( )' が一致しません。");
                                }
                                break;
                            case '[':
                                brackets.Push(1);
                                tokens.Add("[");
                                break;
                            case ']':
                                if (brackets.Count > 0 && brackets.Pop() == 1) {
                                    tokens.Add("]");
                                } else {
                                    throw new GustFrontException("'[ ]' が一致しません。");
                                }
                                break;
                            case '{':
                                brackets.Push(2);
                                tokens.Add("{");
                                break;
                            case '}':
                                if (brackets.Count > 0 && brackets.Pop() == 2) {
                                    tokens.Add("}");
                                } else {
                                    throw new GustFrontException("'{ }' が一致しません。");
                                }
                                break;
                            case '.':
                            case ',':
                            case '=':
                                tokens.Add(c.ToString());
                                break;
                            case '+':
                            case '-':
                            case '*':
                            case '/':
                            case '%':
                                if (nc != '=') {
                                    tokens.Add(c.ToString());
                                } else {
                                    tokens.Add(new string(new char[] { c, '=' }));
                                    i += 1;
                                }
                                break;
                            case '<':
                                if (nc != '=') {
                                    if (nc == '<') {
                                        if (i < line.Length - 2 && line[i + 2] == '=') {
                                            tokens.Add("<<=");
                                            i += 2;
                                        } else {
                                            tokens.Add("<<");
                                            i += 1;
                                        }
                                    } else if (nc == '>') {
                                        tokens.Add("<>");
                                        i += 1;
                                    } else {
                                        tokens.Add("<");
                                    }
                                } else {
                                    tokens.Add("<=");
                                    i += 1;
                                }
                                break;
                            case '>':
                                if (nc != '=') {
                                    if (nc == '>') {
                                        if (i < line.Length - 2 && line[i + 2] == '=') {
                                            tokens.Add(">>=");
                                            i += 2;
                                        } else {
                                            tokens.Add(">>");
                                            i += 1;
                                        }
                                    } else {
                                        tokens.Add(">");
                                    }
                                } else {
                                    tokens.Add(">=");
                                    i += 1;
                                }
                                break;
                        }
                    } else if (!Char.IsWhiteSpace(c)) {
                        int j = i + 1;
                        while (j < line.Length) {
                            nc = line[j];
                            if (IsSymbol(nc) || Char.IsWhiteSpace(line[j])) {
                                tokens.Add(line.Substring(i, j - i));
                                i = j - 1;
                                break;
                            } else if (nc == '\'') {
                                tokens.Add(line.Substring(i, j - i));
                                i = j - 1;
                                break;
                            }
                            j += 1;
                        }
                        if (j == line.Length) {
                            tokens.Add(line.Substring(i, line.Length - i));
                            break;
                        }
                    }
                    i += 1;
                }
                if (brackets.Count > 0) {
                    throw new GustFrontException("括弧が一致しません。");
                }

                if (tokens.Count == 0)
                    goto go_next_line;

                if (mode.Count > 0) {
                    switch (mode.Peek()) {
                        case BlockMode.Reference:
                            if (!Util.CaseInsensitiveEquals(tokens[0], "End")) {
                                string ref_alias = null;
                                string modkey = null;
                                if (tokens.Count >= 3 && tokens[1] == "=") {
                                    ref_alias = tokens[0];
                                    modkey = String.Concat(tokens.GetRange(2, tokens.Count - 2).ToArray());
                                } else {
                                    modkey = String.Concat(tokens.ToArray());
                                }
                                ModuleInfo ref_mod = null;
                                foreach (ModuleInfo rm in myRefMods) {
                                    if (Util.CaseInsensitiveEquals(rm.FilePath, modkey)) {
                                        ref_mod = rm;
                                        break;
                                    }
                                }
                                if (ref_mod == null) {
                                    ref_mod = ModuleInfo.Parse(stream_getter, modkey, reader.CurrentEncoding, myRefMods.ToArray());
                                    myRefMods.Add(ref_mod);
                                    if (ref_alias != null) {
                                        myModRootNs.Add(ref_alias, ref_mod);
                                    }
                                }
                                foreach (KeyValuePair<string, Dictionary<string, string>> kv in ref_mod.myConstants) {
                                    myConstants[kv.Key] = kv.Value;
                                }
                            } else {
                                mode.Pop();
                            }
                            break;
                        case BlockMode.Constant:
                            if (!Util.CaseInsensitiveEquals(tokens[0], "End")) {
                                if (tokens.Count == 3 && tokens[1] == "=" && tokens[2][0] == '0') {
                                    string v = tokens[tokens.Count - 1];
                                    myConstants[constHead].Add(tokens[0], v);
                                } else {
                                    throw new GustFrontException("定数の宣言が不正です。");
                                }
                            } else {
                                mode.Pop();
                            }
                            break;
                        case BlockMode.Import:
                            if (!Util.CaseInsensitiveEquals(tokens[0], "End")) {
                                if (tokens.Count >= 3 && tokens[1] == "=") {
                                    myImports.Add(tokens[0], tokens.GetRange(2, tokens.Count - 2).ToArray());
                                } else {
                                    myUsingNamespaces.Add(String.Concat(tokens.ToArray()));
                                }
                            } else {
                                mode.Pop();
                            }
                            break;
                        default:
                            if (!Util.CaseInsensitiveEquals(tokens[0], "End")) {
                                if (tokens[0].EndsWith(":")) {
                                    proc.DefineLabel(tokens[0].Substring(0, tokens[0].Length - 1), true);
                                    tokens.RemoveAt(0);
                                }
                                if (tokens.Count > 0) {
                                    proc.ParseStatement(tokens, mode, lblIf, lblDo, lblFor, varFor);
                                }
                            } else {
                                switch (mode.Pop()) {
                                    case BlockMode.Procedure:
                                        proc.Emit(Opcodes.LDI, null);
                                        proc.Emit(Opcodes.RET, 1);
                                        proc = null;
                                        break;
                                    case BlockMode.If:
                                        string atx = lblIf.Pop();
                                        do {
                                            proc.MarkLabel(atx);
                                            atx = lblIf.Pop();
                                        } while (atx != null);
                                        proc.LeaveVarBlock();
                                        break;
                                    case BlockMode.DoWhile:
                                        throw new GustFrontException("Do-While が閉じていません。");
                                    case BlockMode.Do:
                                        throw new GustFrontException("Do が閉じていません。");
                                    case BlockMode.For:
                                        throw new GustFrontException("For が閉じていません。");
                                    default:
                                        //ICE
                                        throw new GustFrontException("不正な End です。");
                                }
                            }
                            break;
                    }
                    //non-mode
                } else {
                    if (Util.CaseInsensitiveEquals(tokens[0], "Reference")) {
                        mode.Push(BlockMode.Reference);
                    } else if (Util.CaseInsensitiveEquals(tokens[0], "Constant")) {
                        if (tokens.Count == 2 && tokens[1].Length > 0) {
                            constHead = tokens[1];
                            if (!myConstants.ContainsKey(constHead)) {
                                myConstants.Add(constHead, CollectionUtil.CreateCaseInsensitiveDictionary<string>());
                            }
                            mode.Push(BlockMode.Constant);
                        } else {
                            throw new GustFrontException("定数ブロックの宣言が不正です。");
                        }
                    } else if (Util.CaseInsensitiveEquals(tokens[0], "Import")) {
                        mode.Push(BlockMode.Import);
                    } else if (Util.CaseInsensitiveEquals(tokens[0], "Procedure")) {
                        if (tokens.Count == 2 || (tokens.Count >= 4 && tokens[2] == "(" && tokens[tokens.Count - 1] == ")")) {
                            string name = tokens[1];
                            if (!myProcedures.ContainsKey(name)) {
                                List<string> args = new List<string>();
                                for (int j = 3; j < tokens.Count - 1; j += 2) {
                                    args.Add(tokens[j]);
                                    if (tokens[j + 1] != ")" && tokens[j + 1] != ",") {
                                        throw new GustFrontException("引数の指定に誤りがあります。");
                                    }
                                }
                                proc = new ProcedureInfo(this, name, args.ToArray());
                                myProcedures.Add(name, proc);
                                mode.Push(BlockMode.Procedure);
                            } else {
                                throw new GustFrontException(String.Format("プロシージャ {0} は既に存在しています。", name));
                            }
                        } else {
                            throw new GustFrontException("プロシージャの宣言が不正です。");
                        }
                    } else if (Util.CaseInsensitiveEquals(tokens[0], "Dim")) {
                        if (tokens.Count > 1) {
                            string name = tokens[1];
                            for (int j = 0; j < myGlobalVariables.Count; j++) {
                                if (Util.CaseInsensitiveEquals(myGlobalVariables[j], name)) {
                                    throw new GustFrontException("同じ名前を持つグローバル変数が既に宣言されています。");
                                }
                            }
                            if (tokens.Count > 2 && tokens[2] == "=") {
                                if (tokens.Count > 3) {
                                    myGlobalInitializer.ParseCondition(tokens.GetRange(3, tokens.Count - 3));
                                    myGlobalVariables.Add(name);
                                    myGlobalInitializer.EmitStore(name);
                                } else {
                                    throw new GustFrontException("代入する値がありません。");
                                }
                            } else {
                                myGlobalVariables.Add(name);
                            }
                        } else {
                            throw new GustFrontException("変数名がありません。");
                        }
                    } else {
                        throw new GustFrontException("ブロックを識別できません。");
                    }
                }
            go_next_line:

                lineNumber += extln;
                extln = 0;
                line = reader.ReadLine();
            }

            if (mode.Count > 0) {
                throw new GustFrontException("ブロックが閉じていません。");
            }

            myGlobalInitializer.Emit(Opcodes.LDI, null);
            myGlobalInitializer.Emit(Opcodes.RET, 0);

            myGlobalInitializer.RestoreStrings();
            foreach (ProcedureInfo _proc in myProcedures.Values) {
                _proc.RestoreStrings();
            }
        }

        private static ModuleInfo Parse(Converter<string, Stream> stream_getter, string source, Encoding enc, ModuleInfo[] ref_of_root)
        {
            ModuleInfo info = new ModuleInfo();
            info.FilePath = source;
            info.myGlobalInitializer = new ProcedureInfo(info, String.Concat("!", "init"), new string[0]);
            if (ref_of_root != null) {
                info.myRefMods.AddRange(ref_of_root);
            }

            int lineNumber = 0;

            using (StreamReader reader = new StreamReader(stream_getter(source), enc)) {
                try {
                    info.ParseImpl(reader, stream_getter, ref lineNumber);
                } catch (Exception ex) {
                    StringBuilder sb = new StringBuilder(256);
                    sb.AppendLine("スクリプトを解析中にエラーが発生しました。");
                    sb.Append("詳細:");
                    sb.AppendLine(ex.Message);
                    sb.Append("ファイル:");
                    sb.AppendLine(source);
                    sb.Append("行:");
                    sb.AppendLine(lineNumber.ToString());
                    Trace.Write(sb.ToString());
                    throw new GustFrontException(sb.ToString(), ex);
                }
            }

            return info;
        }

        internal static ModuleInfo Parse(Converter<string, Stream> stream_getter, string source, Encoding enc)
        {
            return Parse(stream_getter, source, enc, null);
        }
    }
}