﻿/*
Copyright (c) 2013, KAKUMOTO Masayuki
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of the outher KAKUMOTO Masayuki nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Globalization;
using System.Diagnostics;

namespace GustFront
{
    /// <summary>
    /// 文字列操作などの汎用ルーチンを提供する。
    /// </summary>
    public static class Util
    {
        /// <summary>
        /// ディレクトリセパレータで引数を連結する。
        /// </summary>
        public static string CombinePath(params string[] path)
        {
            if (path.Length > 0) {
                StringBuilder sb = new StringBuilder(256);
                sb.Append(path[0]);
                for (int i = 1; i < path.Length; i++) {
                    sb.Append(Path.DirectorySeparatorChar);
                    sb.Append(path[i]);
                }
                return sb.ToString();
            } else {
                return String.Empty;
            }
        }

        /// <summary>
        /// 文字の大小を区別せずに文字列を比較する。
        /// </summary>
        public static bool CaseInsensitiveEquals(string s1, string s2)
        {
            return StringComparer.InvariantCultureIgnoreCase.Equals(s1, s2);
        }

        /// <summary>
        /// 文字の大小を区別せずにコレクションに文字列が含まれるか調べる。
        /// </summary>
        public static bool CaseInsensitiveContains(IEnumerable<string> s1, string s2)
        {
            foreach (string s in s1) {
                if (StringComparer.InvariantCultureIgnoreCase.Equals(s, s2)) return true;
            }
            return false;
        }

        /// <summary>
        /// 文字の大小を区別せずにリストに文字列が含まれるか調べる。
        /// </summary>
        public static int CaseInsensitiveIndexOf(IList<string> s1, string s2)
        {
            for (int i = 0; i < s1.Count; i++) {
                if (StringComparer.InvariantCultureIgnoreCase.Equals(s1[i], s2)) return i;
            }
            return -1;
        }

        /// <summary>
        /// 文字の大小を区別せずにリストに文字列が含まれるか調べる。
        /// </summary>
        public static int CaseInsensitiveLastIndexOf(IList<string> s1, string s2)
        {
            for (int i = s1.Count - 1; i >= 0; i--) {
                if (StringComparer.InvariantCultureIgnoreCase.Equals(s1[i], s2)) return i;
            }
            return -1;
        }

        /// <summary>
        /// 引数を連結し、改行を付加してログに出力する。
        /// </summary>
        public static void AppendToLog(params string[] text)
        {
            Trace.WriteLine(String.Concat(text));
            Trace.Flush();
        }
    }

    /// <summary>
    /// リストおよびディクショナリの汎用ルーチンと、スクリプトで使用可能なクラスを提供する。
    /// </summary>
    public static class CollectionUtil
    {
        private class StdList : ArrayList, IHasMember
        {
            public StdList()
                : base()
            {
            }

            public StdList(ICollection c)
                : base(c)
            {
            }

            public StdList(int capacity)
                : base(capacity)
            {
            }

            public bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
            {
                switch (name) {
                    case "Count":
                        result = Count;
                        return true;
                    case "Default":
                    case "Item":
                        result = new ScriptValue(this[@params[0].AsInt32() - 1]);
                        return true;
                    case "Add":
                        result = Add(@params[0].Instance) + 1;
                        return true;
                    case "Clear":
                        Clear();
                        result = null;
                        return true;
                    case "Contains":
                        result = Contains(@params[0].Instance);
                        return true;
                    case "IndexOf":
                        result = IndexOf(@params[0].Instance) + 1;
                        return true;
                    case "Insert":
                        Insert(@params[0].AsInt32() - 1, @params[1].Instance);
                        result = null;
                        return true;
                    case "Remove":
                        Remove(@params[0].Instance);
                        result = null;
                        return true;
                    case "RemoveAt":
                        RemoveAt(@params[0].AsInt32() - 1);
                        result = null;
                        return true;
                }
                result = null;
                return false;
            }

            public bool SetMember(string name, ScriptValue[] @params)
            {
                switch (name) {
                    case "Default":
                    case "Item":
                        this[@params[0].AsInt32() - 1] = @params[1].Instance;
                        return true;
                }
                return false;
            }
        }

        private class StdDictionary : Hashtable, IHasMember
        {
            public StdDictionary()
                : base()
            {
            }

            public StdDictionary(IDictionary d)
                : base(d)
            {
            }

            public StdDictionary(int capacity)
                : base(capacity)
            {
            }

            public StdDictionary(IEqualityComparer comparer)
                : base(comparer)
            {
            }

            public StdDictionary(IDictionary d, IEqualityComparer comparer)
                : base(d, comparer)
            {
            }

            public StdDictionary(int capacity, IEqualityComparer comparer)
                : base(capacity, comparer)
            {
            }

            public bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
            {
                switch (name) {
                    case "Count":
                        result = Count;
                        return true;
                    case "Default":
                    case "Item":
                        result = new ScriptValue(this[@params[0].Instance]);
                        return true;
                    case "Keys":
                        result = new ScriptValue(new StdList(Keys));
                        return true;
                    case "Values":
                        result = new ScriptValue(new StdList(Values));
                        return true;
                    case "Add":
                        Add(@params[0].Instance, @params[1].Instance);
                        result = null;
                        return true;
                    case "Clear":
                        Clear();
                        result = null;
                        return true;
                    case "Contains":
                        result = Contains(@params[0].Instance);
                        return true;
                    case "Remove":
                        Remove(@params[0].Instance);
                        result = null;
                        return true;
                }
                result = null;
                return false;
            }

            public bool SetMember(string name, ScriptValue[] @params)
            {
                switch (name) {
                    case "Default":
                    case "Item":
                        this[@params[0].Instance] = @params[1].Instance;
                        return true;
                }
                return false;
            }
        }

        /// <summary>
        /// スクリプトで使用可能なリストを作成する。
        /// </summary>
        public static IList CreateStdList() { return new StdList(); }
        public static IList CreateStdList(ICollection c) { return new StdList(c); }
        public static IList CreateStdList(int capacity) { return new StdList(capacity); }

        /// <summary>
        /// スクリプトで使用可能なディクショナリを作成する。
        /// </summary>
        public static IDictionary CreateStdDictionary() { return new StdDictionary(); }
        public static IDictionary CreateStdDictionary(IDictionary d) { return new StdDictionary(d); }
        public static IDictionary CreateStdDictionary(int capacity) { return new StdDictionary(capacity); }
        public static IDictionary CreateStdDictionary(IEqualityComparer comparer) { return new StdDictionary(comparer); }
        public static IDictionary CreateStdDictionary(IDictionary d, IEqualityComparer comparer) { return new StdDictionary(d, comparer); }
        public static IDictionary CreateStdDictionary(int capacity, IEqualityComparer comparer) { return new StdDictionary(capacity, comparer); }

        /// <summary>
        /// 文字列をキーとする、文字の大小を区別しないディクショナリを作成する。
        /// </summary>
        public static Dictionary<string, T> CreateCaseInsensitiveDictionary<T>()
        {
            return new Dictionary<string, T>(StringComparer.InvariantCultureIgnoreCase);
        }
        public static Dictionary<string, T> CreateCaseInsensitiveDictionary<T>(int capacity)
        {
            return new Dictionary<string, T>(capacity, StringComparer.InvariantCultureIgnoreCase);
        }
        public static Dictionary<string, T> CreateCaseInsensitiveDictionary<T>(IDictionary<string, T> dictionary)
        {
            return new Dictionary<string, T>(dictionary, StringComparer.InvariantCultureIgnoreCase);
        }

        /// <summary>
        /// 全要素を含む配列を作成する。
        /// </summary>
        public static T[] ToArray<T>(ICollection<T> source)
        {
            T[] result = new T[source.Count];
            source.CopyTo(result, 0);
            return result;
        }
        /// <summary>
        /// 指定されたインデックス以降の要素を含む配列を作成する。
        /// </summary>
        public static T[] ToArray<T>(IList<T> source, int index)
        {
            return ToArray(source, index, source.Count - index);
        }
        /// <summary>
        /// 指定された範囲の要素を含む配列を作成する。
        /// </summary>
        public static T[] ToArray<T>(IList<T> source, int index, int count)
        {
            T[] result = new T[count];
            for (int i = 0; i < count; i++) {
                result[i] = source[index + i];
            }
            return result;
        }
    }
}