﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Drawing;
using GustFront;
using RVII;

public struct Vector
{
    public int Source;
    public int Length;
    public Vector(int source, int length)
    {
        this.Source = source;
        this.Length = length;
    }
    public static Vector Zero = new Vector(0, 0);
    public int Target
    {
        get { return Source + Length; }
        set { Length = value - Source; }
    }
    public bool IntersectsWith(Vector vec)
    {
        return (Source <= vec.Source && vec.Source < Target) || (vec.Source <= Source && Source < vec.Target);
    }
    public bool Contains(Vector vec)
    {
        return Source <= vec.Source && vec.Target <= Target;
    }
    public void Offset(int v)
    {
        Source += v;
    }
    public static bool operator ==(Vector a, Vector b)
    {
        return a.Source == b.Source && a.Length == b.Length;
    }
    public static bool operator !=(Vector a, Vector b)
    {
        return a.Source != b.Source || a.Length != b.Length;
    }
    public override bool Equals(object obj)
    {
        return this == (Vector)obj;
    }
    public override int GetHashCode()
    {
        return (new SharpDX.Vector2(Source, Length)).GetHashCode();
    }
}

public struct Cuboid
{
    public Vector X;
    public Vector Y;
    public Vector Z;
    public Cuboid(Vector x, Vector y, Vector z)
    {
        this.X = x;
        this.Y = y;
        this.Z = z;
    }
    public Cuboid(int posX, int posY, int posZ)
    {
        this.X = new Vector(posX, 1);
        this.Y = new Vector(posY, 1);
        this.Z = new Vector(posZ, 1);
    }
    public Cuboid(GeometryData geo)
    {
        this.X = new Vector(geo.X, 1);
        this.Y = new Vector(geo.Y, 1);
        this.Z = new Vector(geo.Z, 1);
    }
    public static Cuboid Zero = new Cuboid(Vector.Zero, Vector.Zero, Vector.Zero);
    public bool IntersectsWith(Cuboid cube)
    {
        return X.IntersectsWith(cube.X) && Y.IntersectsWith(cube.Y) && Z.IntersectsWith(cube.Z);
    }
    public bool Contains(Cuboid cube)
    {
        return (X.Contains(cube.X) && Y.Contains(cube.Y) && Z.Contains(cube.Z)) || (cube.X.Contains(X) && cube.Y.Contains(Y) && cube.Z.Contains(Z));
    }
    public void Offset(int x, int y, int z)
    {
        this.X.Offset(x);
        this.Y.Offset(y);
        this.Z.Offset(z);
    }
    public Point Location
    {
        get { return new Point(X.Source, Y.Source); }
        set
        {
            X.Source = value.X;
            Y.Source = value.Y;
        }
    }
    public static bool operator ==(Cuboid a, Cuboid b)
    {
        return a.X == b.X && a.Y == b.Y && a.Z == b.Z;
    }
    public static bool operator !=(Cuboid a, Cuboid b)
    {
        return a.X != b.X || a.Y != b.Y || a.Z != b.Z;
    }
    public override bool Equals(object obj)
    {
        return this == (Cuboid)obj;
    }
    public override int GetHashCode()
    {
        return (new SharpDX.Vector3(X.GetHashCode(), Y.GetHashCode(), Z.GetHashCode())).GetHashCode();
    }
}

public interface IBounds
{
    Cuboid Bounds { get; }
}

public interface IActor : IRenderable, IBounds, IHasMember
{
    Cuboid Location { get; set; }
    Point Offset { get; set; }
    ActorMaterial Material { get; set; }
    string Motion { get; set; }
    int Frame { get; set; }
    bool Visible { get; set; }
    int OrderInClass { get; set; }

    GeometryData[] GetCurrentGeometry();
    void DeclareData(string name);
}

public interface IGlobalActor : IActor
{
    event EventHandler PositionChanged;

    Area Area { get; set; }
}

public class ActorMaterial : Element, IRenderable
{
    private Dictionary<string, GeometryData[][]> myMaterial;

    public ActorMaterial(string tagName)
        : base(tagName)
    {
        myMaterial = CollectionUtil.CreateCaseInsensitiveDictionary<GeometryData[][]>(4);
    }

    public TextureInfo TextureInfo
    {
        get { return (TextureInfo)GetData("Texture").Instance; }
        set { SetData("Texture", new ScriptValue(value)); }
    }

    public int HasMotion(string name)
    {
        if (myMaterial.ContainsKey(name)) {
            return myMaterial[name].Length;
        } else {
            return 0;
        }
    }

    public GeometryData[] GetGeometry(string motion, int frame)
    {
        return myMaterial[motion][frame];
    }

    private static int ParseInt32(string s)
    {
        return Int32.Parse(s, NumberStyles.Integer, NumberFormatInfo.InvariantInfo);
    }

    public void LoadMaterial(Stream strm, string motion, int frame)
    {
        if (!myMaterial.ContainsKey(motion)) {
            myMaterial.Add(motion, new GeometryData[((System.Collections.IList)GetData(motion).Instance).Count][]);
        }

        List<GeometryData> theList = new List<GeometryData>();
        using (StreamReader reader = new StreamReader(strm, System.Text.Encoding.ASCII)) {
            string line = reader.ReadLine();
            while (line != null) {
                Match m = Regex.Match(line, "\\s*(-?\\d+)\\s*,\\s*(-?\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(-?\\d+)", RegexOptions.Compiled);
                if (m.Success) {
                    int destX = ParseInt32(m.Groups[1].Value);
                    int destY = ParseInt32(m.Groups[2].Value);
                    int destZ = ParseInt32(m.Groups[3].Value);
                    int col = ParseInt32(m.Groups[4].Value);
                    int tex = ParseInt32(m.Groups[5].Value);

                    GeometryData obj = new GeometryData(null);
                    obj.X = destX;
                    obj.Y = destY;
                    obj.Z = destZ;
                    obj.T = new GeometryTexture();
                    obj.T.M = 0;
                    obj.T.C = col;
                    obj.T.Tex = tex;

                    theList.Add(obj);
                }
                line = reader.ReadLine();
            }
        }
        myMaterial[motion][frame] = theList.ToArray();
    }

    public override Element Duplicate(string newTag)
    {
        throw new NotImplementedException();
    }
}

public class Actor : Element, IActor
{
    private GeometryData myGeometry;
    private Point myOffset;
    private ActorMaterial myMaterial;
    private string myCurrentMotion = null;
    private int myCurrentFrame = 0;
    private bool myVisible = true;
    private int myOrderInClass;

    public Actor(string tagName, Cuboid loc)
        : base(tagName)
    {
        myGeometry = new GeometryData(this);
        Location = loc;
    }

    public TextureInfo TextureInfo
    {
        get { return myMaterial.TextureInfo; }
        set { myMaterial.TextureInfo = value; }
    }

    public Cuboid Bounds
    {
        get { return Location; }
    }

    public Cuboid Location
    {
        get { return new Cuboid(myGeometry); }
        set
        {
            myGeometry.X = value.X.Source;
            myGeometry.Y = value.Y.Source;
            myGeometry.Z = value.Z.Source;
        }
    }

    public Point Offset
    {
        get { return myOffset; }
        set { myOffset = value; }
    }

    public ActorMaterial Material
    {
        get { return myMaterial; }
        set { myMaterial = value; }
    }

    public string Motion
    {
        get { return myCurrentMotion; }
        set { myCurrentMotion = value; }
    }

    public int Frame
    {
        get { return myCurrentFrame; }
        set { myCurrentFrame = value; }
    }

    public GeometryData[] GetCurrentGeometry()
    {
        if (myMaterial != null) {
            return myMaterial.GetGeometry(myCurrentMotion, myCurrentFrame);
        } else {
            return new GeometryData[0];
        }
    }

    public bool Visible
    {
        get { return myVisible; }
        set { myVisible = value; }
    }

    public int OrderInClass
    {
        get { return myOrderInClass; }
        set { myOrderInClass = value; }
    }

    private Dictionary<string, object> myData = CollectionUtil.CreateCaseInsensitiveDictionary<object>();

    public void DeclareData(string name)
    {
        myData.Add(name, null);
    }

    public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
    {
        object data = null;
        if (myData.TryGetValue(name, out data)) {
            result = new ScriptValue(data);
        } else if (Util.CaseInsensitiveEquals(name, "X")) {
            result = Location.X.Source;
        } else if (Util.CaseInsensitiveEquals(name, "Y")) {
            result = Location.Y.Source;
        } else if (Util.CaseInsensitiveEquals(name, "Z")) {
            result = Location.Z.Source;
        } else if (Util.CaseInsensitiveEquals(name, "OffsetX")) {
            result = Offset.X;
        } else if (Util.CaseInsensitiveEquals(name, "OffsetY")) {
            result = Offset.Y;
        } else if (Util.CaseInsensitiveEquals(name, "Material")) {
            result = new ScriptValue(Material);
        } else if (Util.CaseInsensitiveEquals(name, "Motion")) {
            result = Motion;
        } else if (Util.CaseInsensitiveEquals(name, "Frame")) {
            result = Frame;
        } else if (Util.CaseInsensitiveEquals(name, "Visible")) {
            result = Visible;
        } else if (Util.CaseInsensitiveEquals(name, "OrderInClass")) {
            result = OrderInClass;
        } else {
            return base.GetMember(name, @params, out result);
        }
        return true;
    }

    public override bool SetMember(string name, ScriptValue[] @params)
    {
        if (myData.ContainsKey(name)) {
            myData[name] = @params[0].Instance;
        } else if (Util.CaseInsensitiveEquals(name, "X")) {
            Location = new Cuboid(@params[0].AsInt32(), Location.Y.Source, Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Y")) {
            Location = new Cuboid(Location.X.Source, @params[0].AsInt32(), Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Z")) {
            Location = new Cuboid(Location.X.Source, Location.Y.Source, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "OffsetX")) {
            Offset = new Point(@params[0].AsInt32(), Offset.Y);
        } else if (Util.CaseInsensitiveEquals(name, "OffsetY")) {
            Offset = new Point(Offset.X, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "Material")) {
            Material = (ActorMaterial)@params[0].Instance;
        } else if (Util.CaseInsensitiveEquals(name, "Motion")) {
            Motion = @params[0].AsString();
        } else if (Util.CaseInsensitiveEquals(name, "Frame")) {
            Frame = @params[0].AsInt32();
        } else if (Util.CaseInsensitiveEquals(name, "Visible")) {
            Visible = @params[0].AsBoolean();
        } else if (Util.CaseInsensitiveEquals(name, "OrderInClass")) {
            OrderInClass = @params[0].AsInt32();
        } else {
            return base.SetMember(name, @params);
        }
        return true;
    }

    public override Element Duplicate(string newTag)
    {
        throw new NotImplementedException();
    }
}

public class Vehicle : Element, IGlobalActor
{
    private Actor myActor = null;

    public event EventHandler PositionChanged;

    protected virtual void OnPositionChanged(EventArgs e)
    {
        if (PositionChanged != null) {
            PositionChanged(this, e);
        }
    }

    public Vehicle(string tagName)
        : base(tagName)
    {
        myActor = new Actor("Internal", Cuboid.Zero);
    }

    public GeometryData[] GetCurrentGeometry()
    {
        if (Material != null) {
            return Material.GetGeometry(Motion, Frame);
        } else {
            return new GeometryData[0];
        }
    }

    public void DeclareData(string name)
    {
        myActor.DeclareData(name);
    }

    public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
    {
        if (base.GetMember(name, @params, out result)) {
            return true;
        } else {
            return myActor.GetMember(name, @params, out result);
        }
    }

    public override bool SetMember(string name, ScriptValue[] @params)
    {
        if (Util.CaseInsensitiveEquals(name, "X")) {
            Location = new Cuboid(@params[0].AsInt32(), Location.Y.Source, Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Y")) {
            Location = new Cuboid(Location.X.Source, @params[0].AsInt32(), Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Z")) {
            Location = new Cuboid(Location.X.Source, Location.Y.Source, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "OffsetX")) {
            Offset = new Point(@params[0].AsInt32(), Offset.Y);
        } else if (Util.CaseInsensitiveEquals(name, "OffsetY")) {
            Offset = new Point(Offset.X, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "Material")) {
            Material = (ActorMaterial)@params[0].Instance;
        } else if (Util.CaseInsensitiveEquals(name, "Motion")) {
            Motion = @params[0].AsString();
        } else if (Util.CaseInsensitiveEquals(name, "Frame")) {
            Frame = @params[0].AsInt32();
        } else if (Util.CaseInsensitiveEquals(name, "Visible")) {
            Visible = @params[0].AsBoolean();
        } else if (Util.CaseInsensitiveEquals(name, "OrderInClass")) {
            OrderInClass = @params[0].AsInt32();
        } else if (Util.CaseInsensitiveEquals(name, "Area")) {
            Area = (Area)@params[0].Instance;
        } else if (base.SetMember(name, @params)) {
            return true;
        } else {
            return myActor.SetMember(name, @params);
        }
        return true;
    }

    public TextureInfo TextureInfo
    {
        get { return (TextureInfo)GetData("Texture").Instance; }
        set { SetData("Texture", new ScriptValue(value)); }
    }

    public Cuboid Bounds
    {
        get { return Location; }
    }

    public Cuboid Location
    {
        get
        {
            Cuboid c;
            c.X = new Vector(GetData("X").AsInt32(), 1);
            c.Y = new Vector(GetData("Y").AsInt32(), 1);
            c.Z = new Vector(GetData("Z").AsInt32(), 1);
            return c;
        }
        set
        {
            Cuboid c = Location;
            SetData("X", value.X.Source);
            SetData("Y", value.Y.Source);
            SetData("Z", value.Z.Source);
            if (c != Location) {
                OnPositionChanged(EventArgs.Empty);
            }
        }
    }

    public Point Offset
    {
        get
        {
            return new Point(GetData("OffsetX").AsInt32(), GetData("OffsetY").AsInt32());
        }
        set
        {
            SetData("OffsetX", value.X);
            SetData("OffsetY", value.Y);
        }
    }

    public ActorMaterial Material
    {
        get { return (ActorMaterial)GetData("Material").Instance; }
        set { SetData("Material", new ScriptValue(value)); }
    }

    public string Motion
    {
        get { return GetData("Motion").AsString(); }
        set { SetData("Motion", value); }
    }

    public int Frame
    {
        get { return GetData("Frame").AsInt32(); }
        set { SetData("Frame", value); }
    }

    public bool Visible
    {
        get { return GetData("Visible").AsBoolean(); }
        set { SetData("Visible", value); }
    }

    public int OrderInClass
    {
        get { return GetData("OrderInClass").AsInt32(); }
        set { SetData("OrderInClass", value); }
    }

    public Area Area
    {
        get { return (Area)GetData("Area").Instance; }
        set
        {
            Area a = Area;
            SetData("Area", new ScriptValue(value));
            if (a != Area) {
                OnPositionChanged(EventArgs.Empty);
            }
        }
    }

    public bool AutoRide
    {
        get { return true; }
    }

    public bool IsHover
    {
        get { return GetData("Hover").AsBoolean(); }
    }

    public override Element Duplicate(string newTag)
    {
        throw new NotImplementedException();
    }
}

public class GeometryTexture
{
    public int M;
    public int C;
    public int Tex;
}

public class GeometryData
{
    public GeometryData(IRenderable @ref)
    {
        R = @ref;
    }
    public int X;
    public int Y;
    public int Z;
    public GeometryTexture T;
    public readonly IRenderable R;
}

public class Area : Element, IRenderable
{
    private static int ParseInt32(string s)
    {
        return Int32.Parse(s, NumberStyles.Integer, NumberFormatInfo.InvariantInfo);
    }

    public class AreaBlock
    {
        private string myName = null;
        private Area myArea = null;
        private Rectangle myGeometryBounds = new Rectangle();
        private GeometryTexture[][] myTowers = null;

        public AreaBlock(string name, Area parent)
        {
            myName = name;
            myArea = parent;
        }

        internal AreaBlock(string name, Area parent, Rectangle bounds)
        {
            myName = name;
            myArea = parent;
            myGeometryBounds = bounds;
        }

        public string Name { get { return myName; } }

        public Area Area { get { return myArea; } }

        public Rectangle GeometryBounds { get { return myGeometryBounds; } }

        public bool IsLoaded { get { return myTowers != null; } }

        public static AreaBlock Combine(AreaBlock[] blocks)
        {
            AreaBlock obj = new AreaBlock(null, null);
            if (blocks.Length == 0) return obj;

            obj.myArea = blocks[0].myArea;

            obj.myGeometryBounds = blocks[0].myGeometryBounds;
            for (int b = 1; b < blocks.Length; b++) {
                obj.myGeometryBounds = Rectangle.Union(obj.myGeometryBounds, blocks[b].myGeometryBounds);
            }
            obj.myTowers = new GeometryTexture[obj.myGeometryBounds.Width * obj.myGeometryBounds.Height][];

            for (int b = 0; b < blocks.Length; b++) {
                for (int i = 0; i < blocks[b].myTowers.Length; i++) {
                    for (int y = 0; y < blocks[b].myGeometryBounds.Height; y++) {
                        Array.Copy(blocks[b].myTowers, blocks[b].myGeometryBounds.Width * y, obj.myTowers,
                            obj.myGeometryBounds.Width * (blocks[b].myGeometryBounds.Top - obj.myGeometryBounds.Top + y) +
                            (blocks[b].myGeometryBounds.Left - obj.myGeometryBounds.Left), blocks[b].myGeometryBounds.Width);
                    }
                }
            }

            return obj;
        }

        public GeometryTexture GetThere(int x, int y, int z)
        {
            if (!IsLoaded) return null;

            if (myGeometryBounds.Left <= x && x < myGeometryBounds.Right) {
                if (myGeometryBounds.Top <= y && y < myGeometryBounds.Bottom) {
                    int ix = x - myGeometryBounds.Left;
                    int iy = y - myGeometryBounds.Top;
                    GeometryTexture[] tower = myTowers[myGeometryBounds.Width * iy + ix];
                    if (tower != null && 0 <= z && z < tower.Length) {
                        GeometryTexture t = tower[z];
                        if (t != null) {
                            GeometryData obj = new GeometryData(myArea);
                            obj.X = x;
                            obj.Y = y;
                            obj.Z = z;
                            obj.T = t;
                            return t;
                            //return obj;
                        }
                    }
                }
            }
            return null;
        }

        public void SetThere(int x, int y, int z, int m)
        {
            if (!IsLoaded) return;

            GeometryData[] gds = myArea.myMaterialTable[m];
            for (int i = 0; i < gds.Length; i++) {
                GeometryData gd = gds[i];
                int dx = x + gd.X;
                int dy = y + gd.Y;
                int dz = z + gd.Z;
                if (myGeometryBounds.Left <= dx && dx < myGeometryBounds.Right) {
                    if (myGeometryBounds.Top <= dy && dy < myGeometryBounds.Bottom) {
                        int ix = dx - myGeometryBounds.Left;
                        int iy = dy - myGeometryBounds.Top;
                        GeometryTexture[] tower = myTowers[myGeometryBounds.Width * iy + ix];
                        if (tower != null && 0 <= dz) {
                            if (tower.Length <= dz) {
                                Array.Resize(ref tower, dz + 1);
                                myTowers[myGeometryBounds.Width * iy + ix] = tower;
                            }
                            tower[dz] = gd.T;
                        }
                    }
                }
            }
        }

        public IList<GeometryData> GetVisibleGeometry(Rectangle vision, int bg_tex, Cuboid? sight)
        {
            if (!IsLoaded) return null;

            List<GeometryData> result = new List<GeometryData>(myTowers.Length * 2);

            GeometryTexture eg = new GeometryTexture();
            eg.M = 0;
            eg.C = -1;
            eg.Tex = bg_tex;

            Rectangle vi = new Rectangle(vision.X / Area.CellWidth, vision.Y / Area.CellHeight,
                vision.Width / Area.CellWidth, vision.Height / Area.CellHeight);
            Rectangle gi;
            Vector z_range;
            if (!sight.HasValue) {
                gi = myGeometryBounds;
                z_range = new Vector(0, Int32.MaxValue);
            } else {
                gi = Rectangle.Intersect(myGeometryBounds, new Rectangle(
                    sight.Value.X.Source, sight.Value.Y.Source, sight.Value.X.Length, sight.Value.Y.Length));
                z_range = sight.Value.Z;
            }

            for (int y = vi.Top; y < gi.Top; y++) {
                for (int x = vi.Left; x < vi.Right; x++) {
                    GeometryData geo = new GeometryData(myArea);
                    geo.X = x * Area.CellWidth;
                    geo.Y = y * Area.CellHeight;
                    geo.Z = 0;
                    geo.T = eg;
                    result.Add(geo);
                }
            }
            for (int y = gi.Top; y < gi.Bottom; y++) {
                int iy = y - myGeometryBounds.Top;
                for (int x = vi.Left; x < gi.Left; x++) {
                    GeometryData geo = new GeometryData(myArea);
                    geo.X = x * Area.CellWidth;
                    geo.Y = y * Area.CellHeight;
                    geo.Z = 0;
                    geo.T = eg;
                    result.Add(geo);
                }
                for (int x = gi.Left; x < gi.Right; x++) {
                    int ix = x - myGeometryBounds.Left;
                    GeometryTexture[] tower = myTowers[myGeometryBounds.Width * iy + ix];
                    for (int z = 0; z < tower.Length; z++) {
                        if (z_range.Source <= z && z < z_range.Target && tower[z] != null) {
                            Rectangle looked = new Rectangle(
                                x * Area.CellWidth, y * Area.CellHeight - (z / 5) * (CellHeight / 2),
                                Area.CellWidth, Area.CellHeight);
                            if (vision.IntersectsWith(looked)) {
                                GeometryData geo = new GeometryData(myArea);
                                geo.X = x * Area.CellWidth;
                                geo.Y = y * Area.CellHeight;
                                geo.Z = z;
                                geo.T = tower[z];
                                result.Add(geo);
                            }
                        }
                    }
                }
                for (int x = gi.Right; x < vi.Right; x++) {
                    GeometryData geo = new GeometryData(myArea);
                    geo.X = x * Area.CellWidth;
                    geo.Y = y * Area.CellHeight;
                    geo.Z = 0;
                    geo.T = eg;
                    result.Add(geo);
                }
            }
            for (int y = gi.Bottom; y < vi.Bottom; y++) {
                for (int x = vi.Left; x < vi.Right; x++) {
                    GeometryData geo = new GeometryData(myArea);
                    geo.X = x * Area.CellWidth;
                    geo.Y = y * Area.CellHeight;
                    geo.Z = 0;
                    geo.T = eg;
                    result.Add(geo);
                }
            }

            return result;
        }

        public void LoadGeometry(Stream strm)
        {
            List<GeometryData> geometries;

            bool strm_is_binary = false;
            byte[] data = null;

            using (BinaryReader reader = new BinaryReader(strm)) {
                data = reader.ReadBytes((int)reader.BaseStream.Length);
                if (data.Length >= 4) {
                    for (int i = 0; i < 4; i++) {
                        if (data[i] <= 0x8) {
                            strm_is_binary = true;
                            break;
                        }
                    }
                }
            }

            if (strm_is_binary) {
                geometries = new List<GeometryData>(data.Length * 2 / 16);
                for (int i = 0; i < data.Length; i += 16) {
                    int destX = BitConverter.ToInt32(data, i + 0);
                    int destY = BitConverter.ToInt32(data, i + 4);
                    int destZ = BitConverter.ToInt32(data, i + 8);
                    int tex = BitConverter.ToInt32(data, i + 12);
                    foreach (GeometryData geo in myArea.myMaterialTable[tex]) {
                        GeometryData obj = new GeometryData(myArea);
                        obj.X = geo.X + destX;
                        obj.Y = geo.Y + destY;
                        obj.Z = geo.Z + destZ;
                        obj.T = geo.T;
                        geometries.Add(obj);
                    }
                }
            } else {
                using (StreamReader reader = new StreamReader(new MemoryStream(data, false), System.Text.Encoding.ASCII)) {
                    geometries = new List<GeometryData>(65536);
                    string line = reader.ReadLine();
                    while (line != null) {
                        Match m = Regex.Match(line, "\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)", RegexOptions.Compiled);
                        if (m.Success) {
                            int destX = ParseInt32(m.Groups[1].Value);
                            int destY = ParseInt32(m.Groups[2].Value);
                            int destZ = ParseInt32(m.Groups[3].Value);
                            int tex = ParseInt32(m.Groups[4].Value);
                            foreach (GeometryData geo in myArea.myMaterialTable[tex]) {
                                GeometryData obj = new GeometryData(myArea);
                                obj.X = geo.X + destX;
                                obj.Y = geo.Y + destY;
                                obj.Z = geo.Z + destZ;
                                obj.T = geo.T;
                                geometries.Add(obj);
                            }
                        }
                        line = reader.ReadLine();
                    }
                }
            }

            myTowers = new GeometryTexture[myGeometryBounds.Width * myGeometryBounds.Height][];

            GeometryTexture eg = new GeometryTexture();
            eg.M = 0;
            eg.C = -1;
            eg.Tex = 0;
            for (int y = myGeometryBounds.Top; y < myGeometryBounds.Bottom; y++) {
                int iy = y - myGeometryBounds.Top;
                for (int x = myGeometryBounds.Left; x < myGeometryBounds.Right; x++) {
                    int ix = x - myGeometryBounds.Left;
                    myTowers[myGeometryBounds.Width * iy + ix] = new GeometryTexture[] { eg };
                }
            }

            for (int i = 0; i < geometries.Count; i++) {
                GeometryData geo = geometries[i];
                if (myGeometryBounds.Contains(geo.X, geo.Y)) {
                    int ti = myGeometryBounds.Width * (geo.Y - myGeometryBounds.Top) + (geo.X - myGeometryBounds.Left);
                    GeometryTexture[] tower = myTowers[ti];
                    if (tower.Length < geo.Z + 1) {
                        Array.Resize(ref tower, geo.Z + 1);
                        myTowers[ti] = tower;
                    }
                    tower[geo.Z] = geo.T;
                }
            }
        }

        public void Unload()
        {
            myTowers = null;
        }
    }

    public const int CellWidth = 32;
    public const int CellHeight = 32;

    private Dictionary<int, GeometryData[]> myMaterialTable = new Dictionary<int, GeometryData[]>();
    private AreaBlock[] myBlocks;

    public Area(string tagName)
        : base(tagName)
    {
        myMaterialTable = new Dictionary<int, GeometryData[]>();
        myBlocks = new AreaBlock[0];
    }

    private static Dictionary<int, GeometryData[]> ParseMaterialDictionary(Stream strm)
    {
        Dictionary<int, List<GeometryData>> materialTable = new Dictionary<int, List<GeometryData>>(100);
        materialTable[0] = new List<GeometryData>(new GeometryData[] { new GeometryData(null) });

        using (StreamReader reader = new StreamReader(strm, System.Text.Encoding.ASCII)) {
            string line = reader.ReadLine();
            while (line != null) {
                Match m = Regex.Match(line, "\\s*(\\d+)\\s*,\\s*(-?\\d+)\\s*,\\s*(-?\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(-?\\d+)", RegexOptions.Compiled);
                if (m.Success) {
                    int mapTo = ParseInt32(m.Groups[1].Value);
                    int destX = ParseInt32(m.Groups[2].Value);
                    int destY = ParseInt32(m.Groups[3].Value);
                    int destZ = ParseInt32(m.Groups[4].Value);
                    int col = ParseInt32(m.Groups[5].Value);
                    int tex = ParseInt32(m.Groups[6].Value);

                    GeometryData obj = new GeometryData(null);
                    obj.X = destX;
                    obj.Y = destY;
                    obj.Z = destZ;
                    obj.T = new GeometryTexture();
                    obj.T.M = mapTo;
                    obj.T.C = col;
                    obj.T.Tex = tex;

                    List<GeometryData> theList = null;
                    if (!materialTable.TryGetValue(mapTo, out theList)) {
                        theList = new List<GeometryData>(4);
                        materialTable.Add(mapTo, theList);
                    }
                    theList.Add(obj);
                }
                line = reader.ReadLine();
            }
        }

        Dictionary<int, GeometryData[]> temp = new Dictionary<int, GeometryData[]>(materialTable.Count);
        foreach (int mapTo in materialTable.Keys) {
            temp[mapTo] = materialTable[mapTo].ToArray();
        }

        return temp;
    }

    public TextureInfo TextureInfo
    {
        get { return (TextureInfo)GetData("Texture").Instance; }
        set { SetData("Texture", new ScriptValue(value)); }
    }

    public void SetAreaMaterial(int m, int i, int t)
    {
        myMaterialTable[m][i].T.Tex = t;
    }

    public void LoadMaterial(Stream strm)
    {
        myMaterialTable = ParseMaterialDictionary(strm);
    }

    public IList<AreaBlock> GetBlocks()
    {
        return Array.AsReadOnly(myBlocks);
    }

    public void LoadBlocks(Stream strm)
    {
        List<AreaBlock> theList = new List<AreaBlock>(4);

        using (StreamReader reader = new StreamReader(strm, System.Text.Encoding.ASCII)) {
            string line = reader.ReadLine();
            while (line != null) {
                Match m = Regex.Match(line, "\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\d+)\\s*,\\s*(\\w+)", RegexOptions.Compiled);
                if (m.Success) {
                    int left = ParseInt32(m.Groups[1].Value);
                    int top = ParseInt32(m.Groups[2].Value);
                    int right = ParseInt32(m.Groups[3].Value) + 1;
                    int bottom = ParseInt32(m.Groups[4].Value) + 1;
                    theList.Add(new AreaBlock(m.Groups[5].Value, this, 
                        new Rectangle(left, top, right - left, bottom - top)));
                }
                line = reader.ReadLine();
            }
        }

        myBlocks = theList.ToArray();
    }

    public override Element Duplicate(string newTag)
    {
        throw new NotImplementedException();
    }
}
