﻿using System;
using System.Collections.Generic;
using System.Drawing;
using GustFront;
using RVII;

public abstract class Fighter : Element, IRenderable
{
    protected Fighter(string tagName)
        : base(tagName)
    {
    }

    public TextureInfo TextureInfo
    {
        get { return (TextureInfo)GetData("Texture").Instance; }
        set { SetData("Texture", new ScriptValue(value)); }
    }
}

public class Hero : Fighter, IGlobalActor
{
    private Actor myActor = null;

    public event EventHandler PositionChanged;

    protected virtual void OnPositionChanged(EventArgs e)
    {
        if (PositionChanged != null) {
            PositionChanged(this, e);
        }
    }

    public Hero(string tagName)
        : base(tagName)
    {
        myActor = new Actor("Internal", Cuboid.Zero);
    }

    public GeometryData[] GetCurrentGeometry()
    {
        if (Material != null) {
            return Material.GetGeometry(Motion, Frame);
        } else {
            return new GeometryData[0];
        }
    }

    public void DeclareData(string name)
    {
        myActor.DeclareData(name);
    }

    public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
    {
        if (base.GetMember(name, @params, out result)) {
            return true;
        } else {
            return myActor.GetMember(name, @params, out result);
        }
    }

    public override bool SetMember(string name, ScriptValue[] @params)
    {
        if (Util.CaseInsensitiveEquals(name, "X")) {
            Location = new Cuboid(@params[0].AsInt32(), Location.Y.Source, Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Y")) {
            Location = new Cuboid(Location.X.Source, @params[0].AsInt32(), Location.Z.Source);
        } else if (Util.CaseInsensitiveEquals(name, "Z")) {
            Location = new Cuboid(Location.X.Source, Location.Y.Source, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "OffsetX")) {
            Offset = new Point(@params[0].AsInt32(), Offset.Y);
        } else if (Util.CaseInsensitiveEquals(name, "OffsetY")) {
            Offset = new Point(Offset.X, @params[0].AsInt32());
        } else if (Util.CaseInsensitiveEquals(name, "Material")) {
            Material = (ActorMaterial)@params[0].Instance;
        } else if (Util.CaseInsensitiveEquals(name, "Motion")) {
            Motion = @params[0].AsString();
        } else if (Util.CaseInsensitiveEquals(name, "Frame")) {
            Frame = @params[0].AsInt32();
        } else if (Util.CaseInsensitiveEquals(name, "Visible")) {
            Visible = @params[0].AsBoolean();
        } else if (Util.CaseInsensitiveEquals(name, "OrderInClass")) {
            OrderInClass = @params[0].AsInt32();
        } else if (Util.CaseInsensitiveEquals(name, "Area")) {
            Area = (Area)@params[0].Instance;
        } else if (base.SetMember(name, @params)) {
            return true;
        } else {
            return myActor.SetMember(name, @params);
        }
        return true;
    }

    public override Element Duplicate(string newTag)
    {
        Hero obj = new Hero(newTag);
        CopyTo(obj);
        return obj;
    }

    public Cuboid Bounds
    {
        get { return Location; }
    }

    public Cuboid Location
    {
        get
        {
            Cuboid c;
            c.X = new Vector(GetData("X").AsInt32(), 1);
            c.Y = new Vector(GetData("Y").AsInt32(), 1);
            c.Z = new Vector(GetData("Z").AsInt32(), 1);
            return c;
        }
        set
        {
            Cuboid c = Location;
            SetData("X", value.X.Source);
            SetData("Y", value.Y.Source);
            SetData("Z", value.Z.Source);
            if (c != Location) {
                OnPositionChanged(EventArgs.Empty);
            }
        }
    }

    public Point Offset
    {
        get
        {
            return new Point(GetData("OffsetX").AsInt32(), GetData("OffsetY").AsInt32());
        }
        set
        {
            SetData("OffsetX", value.X);
            SetData("OffsetY", value.Y);
        }
    }

    public ActorMaterial Material
    {
        get { return (ActorMaterial)GetData("Material").Instance; }
        set { SetData("Material", new ScriptValue(value)); }
    }

    public string Motion
    {
        get { return GetData("Motion").AsString(); }
        set { SetData("Motion", value); }
    }

    public int Frame
    {
        get { return GetData("Frame").AsInt32(); }
        set { SetData("Frame", value);  }
    }

    public bool Visible
    {
        get { return GetData("Visible").AsBoolean(); }
        set { SetData("Visible", value); }
    }

    public int OrderInClass
    {
        get { return GetData("OrderInClass").AsInt32(); }
        set { SetData("OrderInClass", value); }
    }

    public Area Area
    {
        get { return (Area)GetData("Area").Instance; }
        set
        {
            Area a = Area;
            SetData("Area", new ScriptValue(value));
            if (a != Area) {
                OnPositionChanged(EventArgs.Empty);
            }
        }
    }
}

public class Enemy : Fighter
{
    public Enemy(string tagName)
        : base(tagName)
    {
    }

    public override Element Duplicate(string newTag)
    {
        Enemy obj = new Enemy(newTag);
        CopyTo(obj);
        return obj;
    }
}
