/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

var Config	= h5glib.Config;
var Debug	= h5glib.Debug;

var PI		= Math.PI;
var PI2		= Math.PI * 2;

var SCENE_NAME		= "stage";
var ACTOR_MVAL_BASE	= 10000;

/**
 * RenderInfo
 */
var RenderInfo =
{
	fov			: ( 60 * PI / 180 ),
	rayPixels	: 2,	//  8,   4,   2,   1
	samples		: 200,	// 50, 100, 200, 400
	gridPixels	: 64
};


/**
 * Process
 */
var Process = function()
{
	this.command	= null;
};

/**
 * StageProcess
 */
var StageProcess = function( scene )
{
	this.command	= scene.input;

	this.data		= null,
	this.getHeight	= function() { return this.data.length; };
	this.getWidth	= function() { return this.data[0].length; };

	this.res		= null;

	var RaycastingData = function()
	{
		this.dist	= 0;
		this.val	= 0;
		this.tx		= 0;
	};
	this.rd = [];
	for ( var i = 0; i < RenderInfo.samples; i++ )
	{
		this.rd[i] = new RaycastingData();
	}
};
StageProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.loadData = function( scene, json )
	{
		this.data	= json.data;
		this.res	= json.res;
		this.res.load();
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		return false;
	};

	/**
	 * 
	 */
	pt.draw = function( scene, canvas, context )
	{
		this.castRay( scene );

		var viewPoint	= scene.viewPoint;
		var image;
		// sky
		image		= this.res.skyImage;
		var skyW2	= image.width - canvas.width;
		context.drawImage(
			image,
			Math.floor( skyW2 - ( viewPoint.dir / PI2 * skyW2 ) ),	// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			0,				// dy
			canvas.width,	// dw
			image.height	// dh
		);
		// floor
		image = this.res.floorImage;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			canvas.height - image.height,		// dy
			canvas.width,	// dw
			image.height	// dh
		);

		// wall
		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var sx	= this.rd[i].tx * ( RenderInfo.gridPixels - 1 );
			var sy	= ( this.rd[i].val - 1 ) * RenderInfo.gridPixels;
			var sw	= 1;
			var sh	= RenderInfo.gridPixels;

			var dist= this.rd[i].dist * Math.cos( theta - viewPoint.dir );
			var dh	= canvas.height / dist;
			var dx	= i * RenderInfo.rayPixels;
			var dy	= ( canvas.height / 2 ) - ( dh / 2 );
			var dw	= RenderInfo.rayPixels;

			context.drawImage( this.res.wallImage, sx, sy, sw, sh, dx, dy, dw, dh );
		}
	};
	/**
	 * 
	 */
	pt.castRay = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var mapX = Math.floor( viewPoint.x );
			var mapY = Math.floor( viewPoint.y );

			var cosTheta = Math.cos( theta );
			var sinTheta = Math.sin( theta );

			var distX, distY;
			var stepX, stepY;

			if ( cosTheta == 0 )
			{
				distX = 100;
			}
			else if ( cosTheta > 0 )
			{
				stepX = 1;
				distX = ( mapX + 1 - viewPoint.x ) / cosTheta;
			}
			else
			{
				stepX = -1;
				cosTheta *= -1.0;
				distX = ( viewPoint.x - mapX ) / cosTheta;
			}

			if ( sinTheta == 0 )
			{
				distY = 100;
			}
			else if ( sinTheta > 0 )
			{
				stepY = -1;
				distY = ( viewPoint.y - mapY ) / sinTheta;
			}
			else
			{
				stepY = 1;
				sinTheta *= -1.0;
				distY = ( mapY + 1 - viewPoint.y ) / sinTheta;
			}

			while ( true )
			{
				if ( distX < distY )
				{
					mapX += stepX;
					var mapVal = ( 0 <= mapX || mapX < this.getWidth() ) ? this.data[ mapY ][ mapX ] : 1;
					if ( mapX < 0 || this.getWidth() <= mapX || mapVal )
					{
						// not actor
						if ( mapVal < ACTOR_MVAL_BASE )
						{
							this.rd[i].dist	= distX;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= 1 - ( viewPoint.y + distX * sinTheta * stepY ) % 1;
							break;
						}
					}
					distX += ( 1 / cosTheta );	// + dist per grid
				}
				else
				{
					mapY += stepY;
					var mapVal = this.data[ mapY ][ mapX ];
					var mapVal = ( 0 <= mapY || mapY < this.getHeight() ) ? this.data[ mapY ][ mapX ] : 1;
					if ( mapY < 0 || this.getHeight() <= mapY || mapVal )
					{
						// not actor
						if ( mapVal < ACTOR_MVAL_BASE )
						{
							this.rd[i].dist	= distY;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= ( viewPoint.x + distY * cosTheta * stepX ) % 1;
							break;
						}
					}
					distY += ( 1 / sinTheta );	// + dist per grid
				}
			}
		}
	};
})( StageProcess.prototype );

/**
 * ActorSetProcess
 */
var ActorSetProcess = function( scene )
{
	this.command	= scene.input;

	this.children	= [];
	this.visible	= [];
};
ActorSetProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.loadData = function( scene, json )
	{
		// init actors
		this.children.length = 0;
		for ( var i = 0; i < json.actors.length; i++ )
		{
			var actor	= new ActorProcess();
			var adata	= json.actors[i];

			actor.id	= i;
			actor.moveTo( scene.stage, adata.x, adata.y );
			actor.z		= adata.z;
			actor.dir	= adata.dir;
			actor.event	= adata.event;
			if ( i == 0 )
			{
				actor.command	= scene.input;
			}
			else
			{
				actor.command	= new AICommand();
			}
			// add
			this.children.push( actor );
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		for ( var i = 0; i < this.children.length; i++ )
		{
			if ( this.children[i].update( scene ) )
			{
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene, canvas, context )
	{
		this.updateVisible( scene );

		var viewPoint	= scene.viewPoint;
		var stage		= scene.stage;

		for ( var i = 0; i < this.visible.length; i++ )
		{
			var actor = this.visible[i];
			// if behind wall
			if ( stage.rd[ actor.rd.index ].dist <= actor.rd.dist )
			{
				continue;
			}

			// angle
			var sdir = viewPoint.dir - actor.dir;
			if ( sdir < 0 ) { sdir = PI2 + sdir; }
			sdir = sdir / ( PI / 4 );
			if ( sdir < 1 || sdir >= 7 )
			{
				sdir = 0;
			}
			else
			{
				sdir = Math.floor( ( sdir + 1 ) / 2 );
			} 
			// animation
			var offsx = Math.floor( ( scene.ticks % 20 ) / 10 );

			var dh	= canvas.height / actor.rd.dist;
			var dw	= dh;
			var dx	= actor.rd.index * RenderInfo.rayPixels - dw / 2;
			var dy	= canvas.height / 2 - dh / 2;

			var rx	= dx / RenderInfo.rayPixels;
			var rw	= dw / RenderInfo.rayPixels;

			for ( var n = 0; n < rw; n++ )
			{
				var rid = Math.floor( rx + n );
				if ( rid < 0 ) { continue; }
				if ( rid >= RenderInfo.samples ) { break; }
				// draw
				if ( stage.rd[ rid ].dist > actor.rd.dist )
				{
					var sx = ( offsx * RenderInfo.gridPixels ) + ( n / rw ) * ( RenderInfo.gridPixels - 1 );
					var sy = sdir * RenderInfo.gridPixels;
					var sw = RenderInfo.gridPixels / rw;
					var sh = RenderInfo.gridPixels;
					context.drawImage(
						stage.res.actorImage, sx, sy, sw, sh, dx + ( n * RenderInfo.rayPixels ), dy, RenderInfo.rayPixels, dh );
				}
			}
		}
	};
	/**
	 * 
	 */
	pt.updateVisible = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var stage		= scene.stage;

		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		// search visible actors
		this.visible.length = 0;
		for ( var i = 1; i < this.children.length; i++ )
		{
			actor		= this.children[i];
			var x		= actor.x - viewPoint.x;
			var y		= viewPoint.y - actor.y;
			var thetaV	= Math.atan2( y, x );	// PI ... PI * -1
			if ( thetaV < 0 ) { thetaV += PI2; }

			var diff	= startTheta - thetaV;
			if ( 0 <= diff && diff <= RenderInfo.fov )
			{
				actor.rd.index	= Math.floor( diff / ( RenderInfo.fov / RenderInfo.samples ) );
				actor.rd.dist	= Math.sqrt( x * x + y * y );
				this.visible.push( actor );
			}
		}
		this.visible.sort( function( a, b ) { return b.rd.dist - a.rd.dist;  } );
	};
})( ActorSetProcess.prototype );

/**
 * ActorProcess
 */
var ActorProcess = function()
{
	this.ROTATE_VAL	= 0.07;
	this.STEP_VAL	= 0.25;

	this.id			= 0;

	this.x			= 0;
	this.y			= 0;
	this.z			= 0;
	this.dir		= 0;

	this.event		= null;

	this.velocity	= 0;

	this.rd =
	{
		index	: 0,
		dist	: 0
	};
};
ActorProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.moveTo = function( stage, x, y )
	{
		// restore map value
		stage.data[ Math.floor( this.y ) ][ Math.floor( this.x ) ] = 0;
		this.x = x;
		this.y = y;
		// update map value
		stage.data[ Math.floor( this.y ) ][ Math.floor( this.x ) ] = ACTOR_MVAL_BASE + this.id;
	};
	/**
	 * 
	 */
	pt.rotate = function( val )
	{
		this.dir	+= val;
		this.dir	%= PI2;
		if ( this.dir < 0 ) { this.dir += PI2; }
	};
	/**
	 * 
	 */
	pt.isInnerWall = function( stage, x, y )
	{
		var arr = [ -0.5, 0.5 ];

		for ( var i in arr )
		{
			var my = Math.floor( y + i );
			if ( my < 0 || stage.getHeight() <= my ) { return true; }

			for ( var j in arr )
			{
				var mx = Math.floor( x + j );
				if ( mx < 0 || stage.getWidth() <= mx ) { return true; }

				var value = stage.data[ my ][ mx ];
				if ( value != 0 && value != ACTOR_MVAL_BASE + this.id )
				{
					return true;
				}
			}
		}
		return false;
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		return this.updateMotion( scene );
	};
	/**
	 * 
	 */
	pt.updateMotion = function( scene )
	{
		var upd = false;

		this.command.update();

		// left right
		if ( this.command.left )
		{
			this.rotate( this.ROTATE_VAL );
			upd = true;
		}
		else if ( this.command.right )
		{
			this.rotate( this.ROTATE_VAL * -1 );
			upd = true;
		}

		// go back
		if ( this.command.go )
		{
			this.velocity = this.STEP_VAL;
			upd = true;
		}
		else if ( this.command.back )
		{
			this.velocity = this.STEP_VAL * -1;
			upd = true;
		}
		else
		{
			if ( this.velocity != 0 )
			{
				this.velocity /= 2;
				if ( this.velocity < this.STEP_VAL / 8 )
				{
					this.velocity = 0;
				}
				upd = true;
			}
		}

		if ( this.velocity != 0 )
		{
			var newX	= this.x + Math.cos( this.dir ) * this.velocity;
			var newY	= this.y - Math.sin( this.dir ) * this.velocity;

			if ( !this.isInnerWall( scene.stage, newX, this.y ) )
			{
				this.moveTo( scene.stage, newX, this.y );
				upd = true;
			}
			if ( !this.isInnerWall( scene.stage, this.x, newY ) )
			{
				this.moveTo( scene.stage, this.x, newY );
				upd = true;
			}
		}
		return upd;
	};
})( ActorProcess.prototype );

/**
 * EventProcess
 */
var EventProcess = function( scene )
{
	this.command	= scene.input;

	this.EVENT_DIST	= 1;
	this.prevDist	= [];

	this.eventActor	= null;

	this.LAG_VAL	= 10;
	this.lag		= 0;
};
EventProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		if ( scene.focus == this )
		{
			if ( this.lag > 0 )
			{
				this.lag--;
				return false;
			}
			if ( this.command.go || this.command.back || this.command.left || this.command.right )
			{
				scene.focus		= null;
				this.eventActor	= null;
				return true;
			}
			return false;
		}

		var upd = false;
		var actors = scene.actors;

		for ( var i = 0; i < actors.visible.length; i++ )
		{
			var actor = actors.visible[i];
			if ( actor.rd.dist < this.EVENT_DIST )
			{
				var dist = this.prevDist[ actor.id ];
				if ( dist != undefined && dist > this.EVENT_DIST )
				{
					this.eventActor	= actor;
					this.lag		= this.LAG_VAL;
					scene.focus		= this;
					upd = true;
					break;
				}
			}
		}
		// copy dist
		this.prevDist.length = 0;
		for ( var i = 0; i < actors.visible.length; i++ )
		{
			var actor = actors.visible[i];
			this.prevDist[ actor.id ] = actor.rd.dist;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene, canvas, context )
	{
		if ( this.eventActor == null ) { return; }

		scene.stage.res.noticeSound.play();

		var mx = 10;
		var my = 10;
		var mh = 50;

		context.save();

		context.fillStyle	= "rgba(192, 80, 77, 0.7)";
		context.beginPath();
		context.fillRect( mx, my, canvas.width - mx * 2, mh );

		context.strokeStyle	= "rgb(192, 192, 192)";
		context.beginPath();
		context.strokeRect( mx, my, canvas.width - mx * 2, mh );

		context.font		= "bold 18px 'ＭＳ Ｐゴシック'";
		context.fillStyle	= "rgb(255, 255, 255)";
		context.fillText( this.eventActor.event.message, mx + 10, my + 25 );

		context.restore();
	};
})( EventProcess.prototype );

/**
 * FrontViewProcess
 */
var FrontViewProcess = function( scene )
{
	this.command	= scene.input;

	this.FIRE_VAL	= 3;
	this.fire		= 0;
};
FrontViewProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// fire
		if ( this.fire )
		{
			this.fire--;
			upd = true;
		}
		else
		{
			if ( this.command.fire )
			{
				this.fire	= this.FIRE_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene, canvas, context )
	{
		var viewPoint	= scene.viewPoint;
		var stage		= scene.stage;
		var image;

		// gun
		var gx = canvas.width  - stage.res.gunImage.width;
		var gy = canvas.height - stage.res.gunImage.height;
		gx += 20 + Math.cos( scene.ticks / 6 ) * viewPoint.velocity * 70;
		gy += 20 + Math.cos( scene.ticks / 5 ) * viewPoint.velocity * 70;

		// fire
		if ( this.fire )
		{
			if ( this.fire == this.FIRE_VAL )
			{
				stage.res.shootSound.play();
			}
			context.save();
			context.strokeStyle = "#FFFF00";
			context.beginPath();
			context.moveTo( gx, gy + 16 );
			context.lineTo( gx + 32, gy + 48 );
			context.closePath();
			context.stroke();
			context.restore();
		}

		image = stage.res.gunImage;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			gx,				// dx
			gy,				// dy
			image.width,	// dw
			image.height	// dh
		);
	};
})( FrontViewProcess.prototype );

/**
 * DebugMessageProcess
 */
var DebugMessageProcess = function( scene )
{
	this.command	= scene.input;
	this.message	= "";
};
DebugMessageProcess.prototype = new Process();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		this.message = Debug.str;
		return false;
	};

	/**
	 * 
	 */
	pt.draw = function( scene, canvas, context )
	{
		if ( !this.message ) { return; }

		context.save();
		context.font		= "18px 'ＭＳ Ｐゴシック'";
		context.fillStyle	= "red";
		context.fillText( this.message, 4, 22 );
		context.restore();
	};
})( DebugMessageProcess.prototype );


/**
 * Command
 */
var Command = function()
{
	this.left	= 0;
	this.right	= 0;
	this.go		= 0;
	this.back	= 0;
	this.fire	= 0;
	this.seek	= 0;
	this.escape	= 0;
};

(function( pt )
{
	/**
	 * 
	 */
	pt.clear = function()
	{
		this.left	= 0;
		this.right	= 0;
		this.go		= 0;
		this.back	= 0;
		this.fire	= 0;
		this.seek	= 0;
		this.escape	= 0;
	};
})( Command.prototype );

/**
 * AICommand
 */
var AICommand = function() {};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 15 );
		switch ( r )
		{
			case 0: this.left	= 1; break;
			case 1: this.right	= 1; break;
			case 2: this.go		= 1; break;
			case 3: this.back	= 1; break;
		}
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function() {};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.left	= value; break;	// ← | A
				case 39: case 68:	this.right	= value; break;	// → | D
				case 38: case 87:	this.go		= value; break;	// ↑ | W
				case 40: case 83:	this.back	= value; break;	// ↓ | S
				case 66:			this.fire	= value; break;	// B
				case 32:			this.seek	= value; break;	// space;
				case 27:			this.escape	= value; break;	// ESC
			}
		}
	};
})( InputCommand.prototype );


/**
 * Scene
 */
var Scene = function( app )
{
	this.app		= app;
	this.name		= SCENE_NAME;
	this.loaded		= false;

	this.ticks		= 0;

	this.canvas		= null;
	this.context	= null;

	this.input		= new InputCommand();
	this.viewPoint	= null;

	this.focus		= null;
	this.stage		= new StageProcess( this );
	this.actors		= new ActorSetProcess( this );
	this.event		= new EventProcess( this );
	this.fview		= new FrontViewProcess( this );
	this.dmsg		= new DebugMessageProcess( this );

	this.processes	= [ this.stage, this.actors, this.event, this.fview, this.dmsg ];
};

(function( pt )
{
	/**
	 * 
	 */
	pt.show = function()
	{
		if ( !this.loaded ) { Debug.alert( "Scene#loaded=" + this.loaded ); return; }

		this.input.clear();

		this.canvas		= window.document.getElementById( Config.canvasName );
		this.context	= this.canvas.getContext("2d");

		for ( var i = 0; i < this.processes.length; i++ )
		{
			this.processes[i].draw( this, this.canvas, this.context );
		}
	};
	/**
	 * 
	 */
	pt.update = function()
	{
		this.ticks++;

		// change scene
		if ( this.input.escape )
		{
			this.app.sceneManager.pop();
			this.app.sceneManager.current.show();
			return;
		}

		if ( !this.loaded ) { Debug.alert( "Scene#loaded=" + this.loaded ); return; }

		// update
		var upd = false;
		for ( var i = 0; i < this.processes.length; i++ )
		{
			if ( this.focus == null || this.focus == this.processes[i] )
			{
				if ( this.processes[i].update( this ) ) { upd = true; }
			}
		}
		// draw
		if ( upd )
		{
			for ( var i = 0; i < this.processes.length; i++ )
			{
				this.processes[i].draw( this, this.canvas, this.context );
			}
		}
	};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		this.input.handleSysEvent( event );
	};
	/**
	 * 
	 */
	pt.loadData = function( json )
	{
		try
		{
			this.stage.loadData( this, json );
			this.actors.loadData( this, json );
			this.viewPoint = this.actors.children[0];

			this.loaded = true;
		}
		catch ( e )
		{
			Debug.alert( e );
		}
	};
})( Scene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene = Scene;


// Anonymous function end
//
})( window );
