/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= h5glib.Config;
var Debug			= h5glib.Debug;
var Command			= h5glib.Command;
var Task			= h5glib.Task;
var Message			= h5glib.Message;
var MessageHandler	= h5glib.MessageHandler;
var MessageManager	= h5glib.MessageManager;
var SceneStatus		= h5glib.SceneStatus;
var Scene			= h5glib.Scene;

var STAGE_RECT		= { x :  10, y :  10, w : 256, h : 256 };
var CHIP_RECT		= { x : 290, y :  10, w :  32, h : 256 };
var ACTOR_RECT		= { x : 346, y :  10, w :  32, h : 256 };

var USAGE_HTML			= "<p>" +
						  "マウスでチップを選択して、マップ画面をクリック・ドラッグします。<br>" +
						  "スクロールバーのボタンをクリックするとスクロールします。<br>" +
						  "（キャラクタの設定、データの保存は未実装。）" +
						  "</p>";

var EDIT_BUTTON_TEXT	= "ゲームに戻る";

/**
 * RenderInfo
 */
var RenderInfo =
{
	mapGridPixels	: 16,
	chipGridPixels	: 32,
	chipPixels		: 64
};

/**
 * ColorRes
 */
var ColorRes =
{
	panel			: "gainsboro",
	background		: "black",
	border			: "gray",
	borderButtonOn	: "black",
	borderButtonOff	: "gray",
	gridLine		: "white",
	selected		: "crimson",
	fontReady		: "white"
};

/**
 * FontRes
 */
var FontRes =
{
	large		: "bold 18px 'ＭＳ Ｐゴシック'",
	small		: "bold 14px 'ＭＳ Ｐゴシック'"
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1
};


/**
 * PanelTask
 */
var PanelTask = function( scene )
{
	this.command	= scene.command;	// input
};
PanelTask.prototype = new Task();

(function( pt )
{
	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.fillStyle	= ColorRes.panel;
		context.fillRect( 0, 0, canvas.width, canvas.height );

		context.restore();
	};
})( PanelTask.prototype );

/**
 * EditorTask
 */
var EditorTask = function()
{
	this.resource		= null;
	this.gridPixels		= RenderInfo.mapGridPixels;

	this.rect			= null;		// ピクセル単位 ( x, y, w, h ), Canvas 基準
	this.scroll			= null;		// グリッド単位 ( x, y )
	this.selected		= null;		// グリッド単位 ( x, y )

	this.getVisibleSize	= null;		// グリッド単位 ( w, h )
	this.getMaxScroll	= null;		// グリッド単位 ( w, h )
};
EditorTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setScroll = function( sx, sy )
	{
		var ms	= this.getMaxScroll();
		if ( 0 <= sx && sx <= ms.w &&
			 0 <= sy && sy <= ms.h )
		{
			this.scroll.x = sx;
			this.scroll.y = sy;
		}
	};
	/**
	 * 
	 */
	pt.drawBackground = function( context )
	{
		context.fillStyle = ColorRes.background;
		context.fillRect( this.rect.x, this.rect.y, this.rect.w, this.rect.h );
	};
	/**
	 * 
	 */
	pt.drawSelected = function( context )
	{
		// selected
		context.strokeStyle	= ColorRes.selected;
		context.beginPath();
		context.strokeRect(
			this.rect.x + this.selected.x * this.gridPixels,
			this.rect.y + this.selected.y * this.gridPixels,
			this.gridPixels,
			this.gridPixels
		);
		context.beginPath();
		context.strokeRect(
			this.rect.x + this.selected.x * this.gridPixels + 1,
			this.rect.y + this.selected.y * this.gridPixels + 1,
			this.gridPixels - 2,
			this.gridPixels - 2
		);
	};
	/**
	 * 
	 */
	pt.drawBorder = function( context )
	{
		context.strokeStyle	= ColorRes.border;
		context.beginPath();
		context.strokeRect( this.rect.x, this.rect.y, this.rect.w, this.rect.h );
	};
})( EditorTask.prototype );

/**
 * ScrollType
 */
var ScrollType =
{
	HORIZONTAL	: 0,
	VERTICAL	: 1
};

/**
 * ScrollTask
 */
var ScrollTask = function( scene, type, target )
{
	this.command		= scene.command;	// input

	this.type			= type;
	this.target			= target;
	this.rect			= {};		// ピクセル単位 ( x, y, w, h ), Canvas 基準
	this.buttons =					// ピクセル単位 ( x, y, w, h ), Canvas 基準
	[
		{ value : -1, on : 0 },
		{ value :  1, on : 0 }
	]
	this.initRect();

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ScrollTask.prototype = new Task();

(function( pt )
{
	pt.BAR_LEN		= 18;
	pt.BUTTON_LEN	= 64;

	/**
	 * 
	 */
	pt.initRect = function()
	{
		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			// rect
			this.rect.x	= this.target.rect.x;
			this.rect.y	= this.target.rect.y + this.target.rect.h;
			this.rect.w	= this.target.rect.w;
			this.rect.h	= this.BAR_LEN;
			// buttons
			this.buttons[0].x	= this.rect.x;
			this.buttons[0].y	= this.rect.y;
			this.buttons[0].w	= this.BUTTON_LEN;
			this.buttons[0].h	= this.rect.h;
			this.buttons[1].x	= this.rect.x + this.rect.w - this.BUTTON_LEN;
			this.buttons[1].y	= this.rect.y;
			this.buttons[1].w	= this.BUTTON_LEN;
			this.buttons[1].h	= this.rect.h;
			break;
		case ScrollType.VERTICAL :
			this.rect.x	= this.target.rect.x + this.target.rect.w;
			this.rect.y	= this.target.rect.y;
			this.rect.w	= this.BAR_LEN;
			this.rect.h	= this.target.rect.h;
			// buttons
			this.buttons[0].x	= this.rect.x;
			this.buttons[0].y	= this.rect.y;
			this.buttons[0].w	= this.rect.w;
			this.buttons[0].h	= this.BUTTON_LEN;
			this.buttons[1].x	= this.rect.x;
			this.buttons[1].y	= this.rect.y + this.rect.h - this.BUTTON_LEN;
			this.buttons[1].w	= this.rect.w;
			this.buttons[1].h	= this.BUTTON_LEN;
			break;
		}
	};
	/**
	 * 
	 */
	pt.getSliderRect = function()
	{
		var vs		= this.target.getVisibleSize();
		var ms		= this.target.getMaxScroll();
		var scroll	= this.target.scroll;
		var sarea	= {};
		var slider	= {};

		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			// sarea
			sarea.x		= this.rect.x + this.BUTTON_LEN;
			sarea.w		= this.rect.w - ( this.BUTTON_LEN * 2 );
			// slider
			slider.w	= Math.floor( sarea.w * vs.w / ( vs.w + ms.w ) );
			slider.h	= this.rect.h;
			slider.x	= ( ms.w > 0 ) ? ( sarea.x + ( sarea.w - slider.w ) * scroll.x / ms.w ) : sarea.x;
			slider.y	= this.rect.y;
			break;
		case ScrollType.VERTICAL :
			// sarea
			sarea.y		= this.rect.y + this.BUTTON_LEN;
			sarea.h		= this.rect.h - ( this.BUTTON_LEN * 2 );
			// slider
			slider.w	= this.rect.w;
			slider.h	= Math.floor( sarea.h * vs.h / ( vs.h + ms.h ) );
			slider.x	= this.rect.x;
			slider.y	= ( ms.h > 0 ) ? ( sarea.y + ( sarea.h - slider.h ) * scroll.y / ms.h ) : sarea.y;
			break;
		}
		return slider;
	};
	/**
	 * 
	 */
	pt.addScroll = function( value )
	{
		var scroll	= this.target.scroll;

		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			this.target.setScroll( scroll.x + value, scroll.y );
			break;
		case ScrollType.VERTICAL :
			this.target.setScroll( scroll.x, scroll.y + value );
			break;
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var mi	= this.command.mouseInfo;

			// update button status
			for ( var i = 0; i < this.buttons.length; i++ )
			{
				var button = this.buttons[i];
				if ( button.x <= mi.x && mi.x <= button.x + button.w &&
					 button.y <= mi.y && mi.y <= button.y + button.h )
				{
					button.on	= 1;
					this.addScroll( button.value );
					upd = true;
					break;
				}
			}
		}
		else
		{
			// set button off
			for ( var i = 0; i < this.buttons.length; i++ )
			{
				var button = this.buttons[i];
				if ( button.on ) { upd = true; }
				button.on	= 0;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		// background
		context.fillStyle	= ColorRes.background;
		context.fillRect( this.rect.x + 2, this.rect.y + 2, this.rect.w - 4, this.rect.h - 4 );

		// border
		context.strokeStyle	= ColorRes.border;
		context.beginPath();
		context.strokeRect( this.rect.x + 2, this.rect.y + 2, this.rect.w - 4, this.rect.h - 4 );

		// slider
		var slider = this.getSliderRect();
		context.fillStyle	= ColorRes.panel;
		context.fillRect( slider.x, slider.y, slider.w, slider.h );

		context.strokeStyle	= ColorRes.border;
		context.beginPath();
		context.strokeRect( slider.x, slider.y, slider.w, slider.h );

		// button
		for ( var i = 0; i < this.buttons.length; i++ )
		{
			var button = this.buttons[i];
			context.fillStyle	= ColorRes.panel;
			context.fillRect( button.x, button.y, button.w, button.h );

			context.strokeStyle	= ( button.on ) ? ColorRes.borderButtonOn : ColorRes.borderButtonOff;
			context.beginPath();
			context.strokeRect( button.x, button.y, button.w, button.h );
		}

		context.restore();
	};
})( ScrollTask.prototype );

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.mapGridPixels;
	this.rect			= STAGE_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };

	this.map			= null;
	this.getMapHeight	= function() { return this.map.length; };
	this.getMapWidth	= function() { return this.map[0].length; };

	this.actors			= null;

	// scroll task
	this.hscrollTask	= new ScrollTask( scene, ScrollType.HORIZONTAL, this );
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.vscrollTask	.link( this.hscrollTask );
	this.child			= this.hscrollTask;

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
StageTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, map, actors, resource )
	{
		this.map		= map;
		this.actors		= actors;
		this.resource	= resource;

		// set actor value
		for ( var i = 0; i < this.actors.length; i++ )
		{
			var actor	= this.actors[i];
			var x		= Math.floor( actor.x );
			var y		= Math.floor( actor.y );
			this.map[ y ] [ x ] = Config.actorValBase + i;
		}
	};

	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vw	= this.rect.w / this.gridPixels;
		var vh	= this.rect.h / this.gridPixels;
		var mw	= this.getMapWidth();
		var mh	= this.getMapHeight();
		if ( mw < vw )
		{
			vw = mw;
		}
		if ( mh < vh )
		{
			vh = mh;
		}
		return { w : vw, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var mw	= this.getMapWidth()  - this.rect.w / this.gridPixels;
		var mh	= this.getMapHeight() - this.rect.h / this.gridPixels;
		if ( mw < 0 )
		{
			mw = 0;
		}
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : mw, h : mh };
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sx	= Math.floor( this.command.mouseInfo.ox / this.gridPixels );
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );

			if ( sx < vs.w && sy < vs.h )
			{
				// update selection
				if ( ( sx != this.selected.x || sy != this.selected.y ) )
				{
					this.selected.x = sx;
					this.selected.y = sy;
					upd = true;
				}
				// update map
				var value	= scene.chipTask.selected.y;
				var mx		= sx + this.scroll.x;
				var my		= sy + this.scroll.y;
				if ( this.map[ my ][ mx ] != value )
				{
					// actor
					if ( this.map[ my ][ mx ] >= Config.actorValBase )
					{
					}
					// floor, wall
					else
					{
						if ( ( mx == 0 || mx == this.getMapWidth() - 1 ||
							   my == 0 || my == this.getMapHeight() - 1 ) &&
							   value == 0 )
						{
						}
						else
						{
							this.map[ my ][ mx ] = value;
						}
					}
					upd = true;
				}
			}
		}
		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// map
		for ( var i = 0; i < vs.h; i++ )
		{
			for ( var j = 0; j < vs.w; j++ )
			{
				var value = this.map[ i + this.scroll.y ][ j + this.scroll.x ];
				if ( value <= 0 ) { continue; }

				var image, sx, sy;
				// actor
				if ( value >= Config.actorValBase )
				{
					var id	= value - Config.actorValBase;
					image	= this.resource.actorImages[ this.actors[ id ].type ];
					sx		= 0;
					sy		= 2 * RenderInfo.chipPixels;
				}
				// wall
				else
				{
					image	= this.resource.wallImage;
					sx		= 0;
					sy		= ( value - 1 ) * RenderInfo.chipPixels;
				}

				context.drawImage(
					image,
					sx,											// sx
					sy,											// sy
					RenderInfo.chipPixels,						// sw
					RenderInfo.chipPixels,						// sh
					this.rect.x + ( j * this.gridPixels ),		// dx
					this.rect.y + ( i * this.gridPixels ),		// dy
					this.gridPixels,							// dw
					this.gridPixels								// dh
				);
			}
		}
		// grid line
		context.strokeStyle = ColorRes.gridLine;
		for ( var i = 1; i < vs.h; i++ )
		{
			var y = this.rect.y + ( i * this.gridPixels );
			context.beginPath();
			context.moveTo( this.rect.x, y );
			context.lineTo( this.rect.x + this.rect.w, y );
			context.stroke();
		}
		for ( var j = 1; j < vs.w; j++ )
		{
			var x = this.rect.x + ( j * this.gridPixels );
			context.beginPath();
			context.moveTo( x, this.rect.y );
			context.lineTo( x, this.rect.y + this.rect.h );
			context.stroke();
		}
		// selected
		this.drawSelected( context );
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( StageTask.prototype );

/**
 * ChipTask
 */
var ChipTask = function( scene )
{
	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.chipGridPixels;
	this.rect			= CHIP_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };

	// scroll task
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.child			= this.vscrollTask;

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ChipTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vh	= this.rect.h / this.gridPixels;
		var ih	= this.resource.wallImage.height / RenderInfo.chipPixels + 1;
		if ( ih < vh )
		{
			vh = ih;
		}
		return { w : 1, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var vh	= this.rect.h / this.gridPixels;
		var ih	= this.resource.wallImage.height / RenderInfo.chipPixels + 1;
		var mh	= ih - vh;
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : 0, h : mh };
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );
			if ( sy != this.selected.y && sy < vs.h )
			{
				this.selected.y = sy;
				upd = true;
			}
		}
		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// chip
		for ( var i = 1; i < vs.h; i++ )
		{
			context.drawImage(
				this.resource.wallImage,
				0,									// sx
				( i + this.scroll.y - 1 ) * RenderInfo.chipPixels,	// sy
				RenderInfo.chipPixels,				// sw
				RenderInfo.chipPixels,				// sh
				this.rect.x,						// dx
				this.rect.y + ( i * this.gridPixels ),	// dy
				this.gridPixels,					// dw
				this.gridPixels						// dh
			);
		}

		// selected
		this.drawSelected( context );
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( ChipTask.prototype );

/**
 * ActorTask
 */
var ActorTask = function( scene )
{
	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.chipGridPixels;
	this.rect			= ACTOR_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };

	// scroll task
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.child			= this.vscrollTask;

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ActorTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vh		= this.rect.h / this.gridPixels;
		var images	= this.resource.actorImages.length;
		if ( images < vh )
		{
			vh = images;
		}
		return { w : 1, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var vh		= this.rect.h / this.gridPixels;
		var images	= this.resource.actorImages.length;
		var mh	= images - vh;
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : 0, h : mh };
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );
			if ( sy != this.selected.y && sy < vs.h )
			{
				this.selected.y = sy;
				upd = true;
			}
		}
		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var actors		= scene.stageTask.actors;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// actor
		for ( var i = 0; i < vs.h; i++ )
		{
			context.drawImage(
				this.resource.actorImages[ i + this.scroll.y ],
				0,								// sx
				2 * RenderInfo.chipPixels,		// sy
				RenderInfo.chipPixels,			// sw
				RenderInfo.chipPixels,			// sh
				this.rect.x,					// dx
				this.rect.y + ( i * this.gridPixels ),	// dy
				this.gridPixels,				// dw
				this.gridPixels					// dh
			);
		}

		// selected
		this.drawSelected( context );
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( ActorTask.prototype );

/**
 * ReadyTask
 */
var ReadyTask = function( scene )
{
	this.command	= scene.command;	// input
	this.text		= "Now Loading...";
};
ReadyTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.clearRect( 0, 0, canvas.width, canvas.height );

		context.font		= FontRes.large;
		context.fillStyle	= ColorRes.fontReady;
		context.fillText( this.text, 10, canvas.height / 2 + 20 );

		context.restore();
	};
})( ReadyTask.prototype );


/**
 * MouseArea
 */
var MouseArea = function( target, rect )
{
	this.target	= target;
	this.rect	= rect;
};

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.escape	= 0;
	this.tbl.mouse	= 0;
	this.mouseInfo	=
	{
		target	: null,
		x		: 0,
		y		: 0,
		ox		: 0,
		oy		: 0
	};
	this.mouseAreas	= [];
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 27:	this.tbl.escape	= value; break;	// ESC
			}
		}
		else if ( type == "mousedown" )
		{
			if ( this.getMouseArea( event.layerX, event.layerY ) )
			{
				this.mouse = 1;
			}
			else
			{
				this.mouse = 0;
			}
		}
		else if ( type == "mousemove" )
		{
			if ( this.mouse )
			{
				if ( !this.getMouseArea( event.layerX, event.layerY ) )
				{
					this.mouse = 0;
				}
			}
		}
		else if ( type == "mouseup" || type == "mouseout" )
		{
			this.mouse = 0;
		}
	};
	/**
	 * 
	 */
	pt.getMouseArea = function( x, y )
	{
		for ( var i = this.mouseAreas.length - 1; i >= 0; i-- )
		{
			var area = this.mouseAreas[i];
			if ( area.rect.x <= x && x <= area.rect.x + area.rect.w &&
				 area.rect.y <= y && y <= area.rect.y + area.rect.h )
			{
				this.mouseInfo.target	= area.target;
				this.mouseInfo.x		= x;
				this.mouseInfo.y		= y;
				this.mouseInfo.ox		= x - area.rect.x;
				this.mouseInfo.oy		= y - area.rect.y;
//Debug.print( "x=" + x + ", y=" + y + ", ox=" + this.mouseInfo.ox + ", oy=" + this.mouseInfo.oy );
				return area;
			}
		}
		return null;
	};
	/**
	 * 
	 */
	pt.addMouseArea = function( area )
	{
		this.mouseAreas.push( area );
	};
	/**
	 * 
	 */
	pt.removeMouseArea = function( area )
	{
		for ( var i = 0; i < this.mouseAreas.length; i++ )
		{
			if ( this.mouseAreas[i] == area )
			{
				this.mouseAreas.splice( i, 1 );
				break;
			}
		}
	};
})( InputCommand.prototype );

/**
 * EditorScene
 */
var EditorScene = function( app, name )
{
	this.app		= app;
	this.name		= name;
	this.json			= null;
	this.stageName	= null;

	this.command	= new InputCommand();

	// create task
	this.readyTask	= new ReadyTask( this );
	this.panelTask	= new PanelTask( this );
	this.stageTask	= new StageTask( this );
	this.chipTask	= new ChipTask( this );
	this.actorTask	= new ActorTask( this );
	// create list
	this.stageTask	.link( this.panelTask );
	this.chipTask	.link( this.stageTask );
	this.actorTask	.link( this.chipTask );

	this.headTasks	=
	[
		this.readyTask,
		this.panelTask,
		this.readyTask
	];
	this.setStatus( SceneStatus.READY );

	// message handler
	this.msgManager	= new MessageManager();
	this.msgManager.addHandler( new MessageHandler( MessageType.RET_PREV_SCENE,	this, this.handleRetPrevScene ) );
};
EditorScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleRetPrevScene = function( scene, message )
	{
		this.app.sceneManager.pop();
		var prevScene = this.app.sceneManager.current;
		prevScene.setData( this.json );	// for StageScene
		prevScene.show();
	};

	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.child	= this.headTasks[ this.status ];
	};

	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
		var self	= this;
		var button	= this.app.getDomElement( Config.editId );
		if ( button )
		{
			this.app.addSysEventHandler( button, "click", function()
			{
				try
				{
					if ( self.app.sceneManager.current == self )
					{
						var message = new Message( MessageType.RET_PREV_SCENE );
						self.msgManager.postBreakMessage( message );
					}
				}
				catch ( e )
				{
					self.app.kill();
					Debug.alert( e );
				}
			});
		}
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		var button	= this.app.getDomElement( Config.editId );
		if ( button )
		{
			button.value = EDIT_BUTTON_TEXT;
		}
		this.setUsage( USAGE_HTML );

		this.command.clear();
		this.holdCanvas();
		this.drawChildren( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.json	= data;

		// set parameters
		this.stageTask.setData( this, data.map, data.actors, data.resource );
		this.chipTask.resource	= data.resource;
		this.actorTask.resource	= data.resource;
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.json;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			data.resource.load();
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, 500 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alert( e );
		}
	};
})( EditorScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.EditorScene	= EditorScene;


// Anonymous function end
//
})( window );
