/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= h5glib.Config;
var Debug			= h5glib.Debug;
var Command			= h5glib.Command;
var Task			= h5glib.Task;
var Message			= h5glib.Message;
var MessageHandler	= h5glib.MessageHandler;
var MessageManager	= h5glib.MessageManager;
var SceneStatus		= h5glib.SceneStatus;
var Scene			= h5glib.Scene;

var USAGE_HTML			= "<p>" +
						  "[←][→]: 回転 [↑]: ジャンプ [↓]: しゃがみ [Z]: 発声<br>" +
						  "</p>";

/**
 * RenderInfo
 */
var RenderInfo =
{
	chipPixelsW	: 512,
	chipPixelsH	: 384,
};

/**
 * Priority
 */
/*
var Priority =
{
	NONE	: 0,
	STAGE	: 1,
	ACTOR	: 2,
	EVENT	: 3,
	FRONT	: 4,
	DEBUG	: 5
};
*/
/**
 * ColorRes
 */
var ColorRes =
{
	backGround		: "rgb(0, 0, 0)",
	fontReady		: "white"
};

/**
 * FontRes
 */
var FontRes =
{
	large		: "bold 18px 'ＭＳ Ｐゴシック'",
	small		: "bold 14px 'ＭＳ Ｐゴシック'"
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command	= scene.command;	// input
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.fillStyle	= ColorRes.backGround;
		context.fillRect( 0, 0, canvas.width, canvas.height );

		context.restore();
	};
})( StageTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	NEUTRAL	: 0,
	JUMP	: 1,
	SQUAT	: 2,
	TURN_L	: 3,
	TURN_R	: 4
};

/**
 * ActorTask
 */
var ActorTask = function( scene )
{
	this.command		= scene.command;

	this.updateFuncs	=
	[
		this.updateNeutral,
		this.updateActive,
		this.updateActive,
		this.updateActive,
		this.updateActive,
	];
	this.setStatus( ActorStatus.NEUTRAL );

	this.lag		= 0;
	this.anim		= 0;
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.update	= this.updateFuncs[ this.status ];
	};
	/**
	 * 
	 */
	pt.updateNeutral = function( scene )
	{
		var upd = false;

		if ( this.command.tbl.up )
		{
			this.setStatus( ActorStatus.JUMP );
		}
		else if ( this.command.tbl.down )
		{
			this.setStatus( ActorStatus.SQUAT );
		}
		else if ( this.command.tbl.right )
		{
			this.setStatus( ActorStatus.TURN_L );
		}
		else if ( this.command.tbl.left )
		{
			this.setStatus( ActorStatus.TURN_R );
		}
		else if ( this.command.tbl.voice )
		{
			scene.resource.voiceSound.play();
		}

		if ( this.status != ActorStatus.NEUTRAL )
		{
			this.anim	= 0;
			this.lag	= this.LAG_VAL;
			return true;
		}

		// neutral animation
		if ( this.lag )
		{
			this.lag--;
			if ( this.lag == 0 )
			{
				this.anim = 0;
				upd = true;
			}
		}
		else
		{
			var r = Math.floor( Math.random() * 20 );
			if      ( r == 1 ) { this.anim = 1; }
			else if ( r == 2 ) { this.anim = 2; }

			if ( this.anim != 0 )
			{
				this.lag = 20;
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var upd = false;

		if ( this.lag )
		{
			if ( this.lag % 4 == 0 )
			{
				this.anim = 3 - ( this.lag / 4 );
				upd = true;
			}
			this.lag--;
		}
		else
		{
			this.setStatus( ActorStatus.NEUTRAL );
			this.anim = 0;
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;
		var image;

		image = resource.actorImages[ this.status ];
		context.drawImage(
			image,
			0,						// sx
			this.anim * RenderInfo.chipPixelsH,				// sy
			RenderInfo.chipPixelsW,	// sw
			RenderInfo.chipPixelsH,	// sh
			0,						// dx
			0,						// dy
			RenderInfo.chipPixelsW,	// dw
			RenderInfo.chipPixelsH	// dh
		);
	};
})( ActorTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "scene.actorTask.anim=" + scene.actorTask.anim;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );

/**
 * ReadyTask
 */
var ReadyTask = function( scene )
{
	this.command	= scene.command;	// input
	this.lag		= 0;
	this.text		= "Now Loading...";
};
ReadyTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lagF
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			this.lag = this.LAG_VAL;
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.clearRect( 0, 0, canvas.width, canvas.height );

		context.font		= FontRes.large;
		context.fillStyle	= ColorRes.fontReady;
		context.fillText( this.text, 10, canvas.height / 2 + 20 );

		context.restore();
	};
})( ReadyTask.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.up		= 0;
	this.tbl.down	= 0;
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.voice	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.tbl.left	= value; break;	// ← | A
				case 39: case 68:	this.tbl.right	= value; break;	// → | D
				case 38: case 87:	this.tbl.up		= value; break;	// ↑ | W
				case 40: case 83:	this.tbl.down	= value; break;	// ↓ | S
				case 90:			this.tbl.voice	= value; break;	// Z
				case 27:			this.tbl.escape	= value; break;	// ESC
				case 81:			this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * StageScene
 */
var StageScene = function( app, name )
{
	this.app			= app;
	this.name			= name;
	this.json			= null;

	this.command		= new InputCommand();
	this.resource		= null;

	// create task
	this.readyTask		= new ReadyTask( this );
	this.stageTask		= new StageTask( this );
	this.actorTask		= new ActorTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.actorTask		.link( this.stageTask );
	this.debugTask		.link( this.actorTask );

	this.headTasks		=
	[
		this.readyTask,
		this.stageTask,
		this.readyTask
	];
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager		= new MessageManager();
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.child	= this.headTasks[ this.status ];
	};

	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( USAGE_HTML );

		this.command.clear();
		this.holdCanvas();
		this.drawChildren( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.json	= data;
		// set resource (before loading actor)
		this.resource	= data.resource;
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.json;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			data.resource.load();
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, 500 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alert( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene	= StageScene;

// Anonymous function end
//
})( window );
