/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= h5glib.Config;
var Debug			= h5glib.Debug;
var Command			= h5glib.Command;
var Task			= h5glib.Task;
var Message			= h5glib.Message;
var MessageHandler	= h5glib.MessageHandler;
var MessageManager	= h5glib.MessageManager;
var SceneStatus		= h5glib.SceneStatus;
var Scene			= h5glib.Scene;

var USAGE_HTML			= "<p>" +
						  "右側のキャラクターを操作します。<br>" +
						  "[←][→]: 移動 [Z]: 攻撃<br>" +
						  "</p>";

/**
 * RenderInfo
 */
var RenderInfo =
{
	chipPixelsW	: 256,
	chipPixelsH	: 256
};

/**
 * ColorRes
 */
var ColorRes =
{
	backGround		: "rgb(0, 0, 0)",
	fontReady		: "white"
};

/**
 * FontRes
 */
var FontRes =
{
	large		: "bold 18px 'ＭＳ Ｐゴシック'",
	small		: "bold 14px 'ＭＳ Ｐゴシック'"
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command	= scene.command;	// input
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;

		context.save();

		var image = resource.image.back.media;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			( canvas.width  - image.width  ) / 2,	// dx
			( canvas.height - image.height ) / 2,	// dy
			image.width,	// dw
			image.height	// dh
		);

		context.restore();
	};
})( StageTask.prototype );

/**
 * ActorListTask
 */
var ActorListTask = function( scene )
{
	this.command	= scene.command;
	this.actorL		= null;
	this.actorR		= null;
	this.actors		= [];
};
ActorListTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene )
	{
		this.child	= null;

		// crreate actor

		this.actorL	= new ActorTask();
		this.actorR	= new ActorTask();
		this.actorL.init( scene, new AICommand(), ActorType.LEFT,  this.actorR );
		this.actorR.init( scene, scene.command,   ActorType.RIGHT, this.actorL );

		// link
		this.child = this.actorL;
		this.actorR.link( this.actorL );

		this.actors.push( this.actorL );
		this.actors.push( this.actorR );
	};

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = this.updateChildren( scene );

		if ( this.actorL.status == ActorStatus.PUNCH )
		{
			this.actors.length = 0;
			this.actors.push( this.actorR );
			this.actors.push( this.actorL );
		}
		if ( this.actorR.status == ActorStatus.PUNCH )
		{
			this.actors.length = 0;
			this.actors.push( this.actorL );
			this.actors.push( this.actorR );
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		for ( var i = 0; i < this.actors.length; i++ )
		{
			this.actors[i].draw( scene );
		}
	};
})( ActorListTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	NEUTRAL	: 0,
	STEP	: 1,
	PUNCH	: 2,
	DAMAGE	: 3
};

/**
 * ActorType
 */
var ActorType =
{
	LEFT	: 0,
	RIGHT	: 1
};

/**
 * ActorTask
 */
var ActorTask = function( scene )
{
	this.updateFuncs	=
	[
		this.updateNeutral,
		this.updateActive,
		this.updateActive,
		this.updateActive,
	];
	this.setStatus( ActorStatus.NEUTRAL );

	this.type		= 0;
	this.enemy		= null;

	this.trans		= null;	// center position
	this.step		= 0;

	this.lag		= 0;
	this.anim		= 0;
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 6;
	pt.STEP_VAL	= 30;
	pt.HIT_DIST	= ( RenderInfo.chipPixelsW / 2 ) - 10;

	/**
	 * 
	 */
	pt.init = function( status, command, type, enemy )
	{
		var arr =
		[
			{ x : 128, y : 224 },
			{ x : 384, y : 224 }
		];

		this.command	= command;
		this.type		= type;
		this.enemy		= enemy;
		this.trans		= arr[ this.type ];
	};
	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.update	= this.updateFuncs[ this.status ];
	};
	/**
	 * 
	 */
	pt.updateNeutral = function( scene )
	{
		var upd = false;

		this.command.update();

		if ( this.command.tbl.right )
		{
			this.setStatus( ActorStatus.STEP );
			this.step = this.STEP_VAL;
		}
		else if ( this.command.tbl.left )
		{
			this.setStatus( ActorStatus.STEP );
			this.step = this.STEP_VAL * -1;
		}
		else if ( this.command.tbl.punch )
		{
			var dist = Math.abs( this.trans.x - this.enemy.trans.x );

			if ( this.HIT_DIST - 20 < dist )
			{
				this.setStatus( ActorStatus.PUNCH );
				if ( dist < this.HIT_DIST + 20 && this.enemy.status == ActorStatus.NEUTRAL )
				{
					// damage
					this.enemy.setStatus( ActorStatus.DAMAGE );
					this.enemy.anim	= 0;
					this.enemy.lag	= this.LAG_VAL;
					scene.resource.sound.damage.media.play();

					// effect
					var pm		= ( this.type == ActorType.RIGHT ) ? -1 : 1;
					var ex		= this.trans.x + ( RenderInfo.chipPixelsW / 2 * pm );
					var ey		= this.trans.y;
					var task	= new EffectTask( scene, ex, ey );
					task.link( scene.actorListTask );
				}
			}
		}

		if ( this.status != ActorStatus.NEUTRAL )
		{
			this.anim	= 0;
			this.lag	= this.LAG_VAL;
			return true;
		}

		// neutral animation
		if ( this.lag )
		{
			this.lag--;
			if ( this.lag == 0 )
			{
				this.anim = 0;
				upd = true;
			}
		}
		else
		{
			var r = Math.floor( Math.random() * 10 );
			if ( r == 1 ) { this.anim = 1; }

			if ( this.anim != 0 )
			{
				this.lag = 20;
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var canvas		= scene.canvas;

		var upd = false;

		if ( this.lag )
		{
			if ( this.lag % 4 == 0 )
			{
				this.anim = 2 - ( this.lag / 4 );
				upd = true;
			}
			// move
			if ( this.step != 0 )
			{
				var x = this.trans.x + this.step;
				if ( ( RenderInfo.chipPixelsW / 4 ) <= x && x <= canvas.width - ( RenderInfo.chipPixelsW / 4 ) &&
					 Math.abs( x - this.enemy.trans.x ) >= ( RenderInfo.chipPixelsW / 4 ) )
				{
					this.trans.x = x;
					upd = true;
				}

				this.step /= 2;
				if ( this.step <= this.STEP_VAL / 8 )
				{
					this.step = 0;
				}
			}
			this.lag--;
		}
		else
		{
			this.setStatus( ActorStatus.NEUTRAL );
			this.anim = 0;
			this.step = 0;
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;

		context.save();

		if ( this.anim < 0 ) { Debug.alert( "this.anim=" + this.anim ); }
		var image	= resource.image.actors[ this.status ].media;
		// 画像を反転する。
		if ( this.type == ActorType.LEFT )
		{
			 context.setTransform( -1, 0, 0, 1, this.trans.x * 2, 0 );
		}
		context.drawImage(
			image,
			0,												// sx
			this.anim * RenderInfo.chipPixelsH,				// sy
			RenderInfo.chipPixelsW,							// sw
			RenderInfo.chipPixelsH,							// sh
			this.trans.x - ( RenderInfo.chipPixelsW / 2 ),	// dx
			this.trans.y - ( RenderInfo.chipPixelsH / 2 ),	// dy
			RenderInfo.chipPixelsW,							// dw
			RenderInfo.chipPixelsH							// dh
		);

		context.restore();
	};
})( ActorTask.prototype );

/**
 * EffectTask
 */
var EffectTask = function( scene, cx, cy )
{
	this.lag	= this.LAG_VAL;
	this.anim	= 0;
	this.trans	= { x : cx, y : cy };	// center position
};
EffectTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL		= 6;
	pt.chipPixelsW	= 96;
	pt.chipPixelsH	= 96;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			if ( this.lag % 2 == 0 )
			{
				this.anim = 3 - ( this.lag / 2 );
				upd = true;
			}
			this.lag--;
		}
		else
		{
			this.unlink();
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;

		context.save();

		var image	= resource.image.fire.media;
		context.drawImage(
			image,
			0,										// sx
			this.anim * this.chipPixelsH,			// sy
			this.chipPixelsW,						// sw
			this.chipPixelsH,						// sh
			this.trans.x - ( this.chipPixelsW / 2 ),// dx
			this.trans.y - ( this.chipPixelsH / 2 ),// dy
			this.chipPixelsW,						// dw
			this.chipPixelsH						// dh
		);

		context.restore();
	};
})( EffectTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "x=" + scene.actorListTask.child.trans.x;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );

/**
 * ReadyTask
 */
var ReadyTask = function( scene )
{
	this.command	= scene.command;	// input
	this.lag		= 0;
	this.text		= "Now Loading...";
};
ReadyTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lagF
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			this.lag = this.LAG_VAL;
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();

		context.clearRect( 0, 0, canvas.width, canvas.height );

		context.font		= FontRes.large;
		context.fillStyle	= ColorRes.fontReady;
		context.fillText( this.text, 10, canvas.height / 2 + 20 );

		context.restore();
	};
})( ReadyTask.prototype );

/**
 * AICommand
 */
var AICommand = function()
{
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.punch	= 0;
};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 54 );
		if      ( 0 <= r && r <= 2 )	{ this.tbl.left		= 1; }
		else if ( 3 <= r && r <= 4 )	{ this.tbl.right	= 1; }
		else if ( 5 <= r && r <= 6 )	{ this.tbl.punch	= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.left	= 0;
	this.tbl.right	= 0;
	this.tbl.punch	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.tbl.left	= value; break;	// ← | A
				case 39: case 68:	this.tbl.right	= value; break;	// → | D
				case 90:			this.tbl.punch	= value; break;	// Z
				case 27:			this.tbl.escape	= value; break;	// ESC
				case 81:			this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * StageScene
 */
var StageScene = function( app, name )
{
	this.app			= app;
	this.name			= name;
	this.json			= null;

	this.command		= new InputCommand();
	this.resource		= null;

	// create task
	this.readyTask		= new ReadyTask( this );
	this.stageTask		= new StageTask( this );
	this.actorListTask	= new ActorListTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.actorListTask	.link( this.stageTask );
	this.debugTask		.link( this.actorListTask );

	this.headTasks		=
	[
		this.readyTask,
		this.stageTask,
		this.readyTask
	];
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager		= new MessageManager();
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.child	= this.headTasks[ this.status ];
	};

	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( USAGE_HTML );

		this.command.clear();
		this.holdCanvas();
		this.drawChildren( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.json		= data;
		this.resource	= data.resource;
		// create actorList
		this.actorListTask.setData( this );
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.json;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			this.app.loadResource( data.baseUrl, data.resource );
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, 500 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene	= StageScene;

// Anonymous function end
//
})( window );
