/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jsfps;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.Date;
import java.util.Collections;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.cache.Cache;
import javax.cache.CacheManager;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author hoge
 */
@SuppressWarnings("serial")
public class JsFpsServerServlet extends HttpServlet
{
	/**  */
	private static final String STR_ERR_UNKNOWN		= "Unknown Error.";
	/**  */
	private static final String STR_ERR_PROTOCOL	= "Protocol Error.";
	/**  */
	private static final String STR_ERR_CLIENTID	= "ClientId already exists.";
	/**  */
	private static final String STR_ERR_MAXUSERS	= "Too many users.";
	/**  */
	private static final String STR_ERR_NOACTOR		= "Actor not found.";

	/**  */
	private static final String STR_METHOD_JOIN		= "join";
	/**  */
	private static final String STR_METHOD_UPDATE	= "update";

	/**  */
	private static final String KEY_KEYS			= "keys";
	/**  */
	private static final int	MAX_USERS			= 20;
	/**  */
	private static final int	MAX_STRP_LEN		= 256;
	/**  */
	private static final int	LOGIN_TIMEOUT		= ( 3 * 60 * 1000 );
	/**  */
	private static final int	MVAL_ACTOR			= 1000;

	/**
	 *
	 */
	private static final Logger logger = Logger.getLogger( JsFpsServerServlet.class.getName() );

	/**
	 *
	 */
	@Override
	public void doGet( HttpServletRequest req, HttpServletResponse resp )
		throws IOException
	{
		try
		{
			Cache		cache	= this.getCache();
			List<Actor>	list	= this.getActorList( cache );
			this.output( req, resp, MapData.mapToString(), list );
		}
		catch ( Exception e )
		{
			this.outputError( req, resp, STR_ERR_UNKNOWN + ", " + e.toString() );
			logger.log( Level.ALL, e.getMessage() );
		}
	}

	/**
	 *
	 */
	@Override
	public void doPost( HttpServletRequest req, HttpServletResponse resp )
		throws IOException
	{
		try
		{
			Cache		cache	= this.getCache();
			List<Actor>	list	= this.getActorList( cache );

			req.setCharacterEncoding("UTF-8");
			String method		= req.getParameter("method");
			String clientId		= req.getParameter("clientId");
	
			if ( method == null ||
				 clientId == null || clientId.equals("") || clientId.length() > MAX_STRP_LEN )
			{
				this.outputError( req, resp, STR_ERR_PROTOCOL );
				return;
			}
	
			if ( method.equals( STR_METHOD_JOIN ) )
			{
				String err = this.handleJoin( cache, list, clientId );
				if ( err == null )
				{
					// 再取得
					list	= this.getActorList( cache );
					this.output( req, resp, MapData.mapToString(), list );
				}
				else
				{
					this.outputError( req, resp, err );
				}
			}
			else if ( method.equals( STR_METHOD_UPDATE ) )
			{
				String messageText	= req.getParameter("messageText");
				String strX			= req.getParameter("x");
				String strY			= req.getParameter("y");
				String strDir		= req.getParameter("dir");
				String err = this.handleUpdate( cache, list, clientId, strX, strY, strDir, messageText  );
				if ( err == null )
				{
					// 再取得
					list	= this.getActorList( cache );
					this.output( req, resp, null, list );
				}
				else
				{
					this.outputError( req, resp, err );
				}
			}
			else
			{
				this.outputError( req, resp, STR_ERR_PROTOCOL );
			}
		}
		catch ( Exception e )
		{
			this.outputError( req, resp, STR_ERR_UNKNOWN + ", " + e.toString() );
			logger.log( Level.ALL, e.getMessage() );
		}
	}

	/**
	 *
	 */
	private Cache getCache()
		throws Exception
	{
		return CacheManager.getInstance().getCacheFactory().createCache( Collections.emptyMap() );
	}

	/**
	 *
	 */
	private List<Actor> getActorList( Cache cache )
		throws Exception
	{
		long		time	= (new Date()).getTime();
		List<Actor>	list	= new ArrayList<Actor>();
		boolean		dirty	= false;

		ArrayList<String> keyList = (ArrayList<String>)cache.get( KEY_KEYS );
		if ( keyList == null )
		{
			keyList = new ArrayList<String>();
			dirty = true;
		}
		for ( Iterator<String> it = keyList.iterator(); it.hasNext(); )
		{
			String	key		= it.next();
			Actor	actor	= (Actor)cache.get( key );
			// check timeout
			if ( time - actor.getLastUpdate() > LOGIN_TIMEOUT )
			{
				// delete key and value
				it.remove();
				cache.remove( key );
				dirty = true;
			}
			else
			{
				list.add( actor );
			}
		}
		if ( dirty )
		{
			cache.put( KEY_KEYS, keyList );
		}
		return list;
	}

	/**
	 *
	 */
	private boolean putActorToMap( int[][] map, Actor actor )
		throws Exception
	{
		int x = (int)Math.floor( actor.getX() );
		int y = (int)Math.floor( actor.getX() );
		if ( 0 <= x && x < map[0].length && 0 <= y && y < map.length )
		{
			if ( map[y][x] == 0 )
			{
				map[y][x] = MVAL_ACTOR;
				return true;
			}
			// 周囲１セルだけサーチ＞要修正
			int ltx = x - 1;
			int lty = y - 1;
			for ( int i = 0; i < 3; i++ )
			{
				int cy = lty + i;
				for ( int j = 0; j < 3; j++ )
				{
					int cx = ltx + j;
					if ( 0 <= cx && cx < map[0].length && 0 <= cy && cy < map.length )
					{
						if ( map[cy][cx] == 0 )
						{
							map[cy][cx] = MVAL_ACTOR;
							actor.setX( (double)cx + 0.5 );
							actor.setY( (double)cy + 0.5 );
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	/**
	 *
	 */
	private String handleJoin(
		Cache cache, List<Actor> list,
		String clientId )
		throws Exception
	{
		// ユーザ数をチェック
		if ( list.size() > MAX_USERS )
		{
			return STR_ERR_MAXUSERS;
		}
		// 既に存在？
		if ( cache.get( clientId ) != null )
		{
			return STR_ERR_CLIENTID;
		}

		// update map
		int[][]	map	= MapData.getCopyMap();
		for ( Iterator<Actor> it = list.iterator(); it.hasNext(); )
		{
			Actor tmpActor = it.next();
			this.putActorToMap( map, tmpActor );
			// this.putActor() が失敗した場合...
		}
		// new actor
		long	time	= (new Date()).getTime();
		Actor	actor	= new Actor();
		actor.setClientId( clientId );
		actor.setLastUpdate( time );
		actor.setX( Math.random() * ( map[0].length - 2 ) + 1.0 );
		actor.setY( Math.random() * ( map.length - 2 ) + 1.0 );
		actor.setDir( Math.PI * 3.0 / 2.0 );
		this.putActorToMap( map, actor );
		// add key and value
		ArrayList<String> keyList = (ArrayList<String>)cache.get( KEY_KEYS );
		keyList.add( clientId );
		cache.put( KEY_KEYS, keyList );
		cache.put( clientId, actor );

		return null;
	}

	/**
	 *
	 */
	private String handleUpdate(
		Cache cache, List<Actor> list,
		String clientId, String strX, String strY, String strDir, String messageText )
		throws Exception
	{
		long	time	= (new Date()).getTime();
		Actor	actor	= (Actor)cache.get( clientId );
		if ( actor == null )
		{
			return STR_ERR_NOACTOR;
		}

		actor.setLastUpdate( time );
		if ( messageText != null )
		{
			if ( messageText.length() > MAX_STRP_LEN )
			{
				return STR_ERR_PROTOCOL;
			}
			actor.setMessageText( messageText );
			actor.setMessageTime( time );
		}
		if ( strX != null )
		{
			actor.setX( Double.valueOf( strX ) );
		}
		if ( strY != null )
		{
			actor.setY( Double.valueOf( strY ) );
		}
		if ( strDir != null )
		{
			actor.setDir( Double.valueOf( strDir ) );
		}

		// update map
		int[][]	map	= MapData.getCopyMap();
		for ( Iterator<Actor> it = list.iterator(); it.hasNext(); )
		{
			Actor tmpActor = it.next();
			this.putActorToMap( map, tmpActor );
			// this.putActor() が失敗した場合...
		}
		// update actor
		cache.put( clientId, actor );
		return null;
	}

	/**
	 *
	 */
	private void outputError( HttpServletRequest req, HttpServletResponse resp, String errorMessage )
	{
		try
		{
			resp.setContentType("text/plain; charset=UTF-8");
			PrintWriter out = resp.getWriter();
	
			out.print("{ errorMessage : \"" + errorMessage + "\"}" );
			out.close();
		}
		catch ( Throwable e )
		{
			logger.log( Level.ALL, e.getMessage() );
		}
	}
	/**
	 *
	 */
	private void output(
		HttpServletRequest req, HttpServletResponse resp, String map, List<Actor> list  )
		throws Exception
	{
		resp.setContentType("text/plain; charset=UTF-8");
		PrintWriter out = resp.getWriter();
		
		out.print("{\n");
		if ( map != null )
		{
			out.print( map );
			out.print(",\n");
		}
		out.print("\tactors :\n\t[");
		for ( int i = 0; i < list.size(); i++ )
		{
			if ( i > 0 ) { out.print(","); }
			out.print("\n");

			Actor actor = list.get( i );
			out.print( actor.toJsString() );
		}
		out.print("\n\t]");

		out.print("\n}");

		out.close();
	}
};

/**
 * 
 */
@SuppressWarnings("serial")
class Actor
	implements Serializable
{
	private int		type		= 0;
	private String	name		= "";
	private int		status		= 1;
	private double	x			= 0.0;
	private double	y			= 0.0;
	private double	dir			= 0.0;
	private String	clientId	= "";
	private long	messageTime	= 0;
	private String	messageText	= "";

	private long	lastUpdate	= 0;
	
	public Actor() {}

	public String toJsString()
	{
		StringBuilder builder = new StringBuilder();

		builder.append("\t\t{");
		builder.append("\n\t\t\ttype : " + this.type );
		builder.append(",\n\t\t\tname : \"" + this.name + "\"" );
		builder.append(",\n\t\t\tstatus : " + this.status );
		builder.append(",\n\t\t\tx : " + this.x + ", y : " + this.y + ", z : 1, dir : " + this.dir );
		builder.append(",\n\t\t\tclientId : \"" + this.clientId + "\"" );
		builder.append(",\n\t\t\tmessage : { time : " + this.messageTime + ", text : \"" + this.messageText + "\" }" );
		builder.append(",\n\t\t\tevent : { type : \"none\" }" );
		builder.append("\n\t\t}");

		return builder.toString();
	}

	public int getType()
	{
		return type;
	}
	public void setType( int type )
	{
		this.type = type;
	}
	public String getName()
	{
		return name;
	}
	public void setName( String name )
	{
		this.name = name;
	}
	public int getStatus()
	{
		return status;
	}
	public void setStatus( int status )
	{
		this.status = status;
	}
	public double getX()
	{
		return x;
	}
	public void setX( double x )
	{
		this.x = x;
	}
	public double getY()
	{
		return y;
	}
	public void setY( double y )
	{
		this.y = y;
	}
	public double getDir()
	{
		return dir;
	}
	public void setDir( double dir )
	{
		this.dir = dir;
	}
	public String getClientId()
	{
		return clientId;
	}
	public void setClientId( String clientId )
	{
		this.clientId = clientId;
	}
	public long getMessageTime()
	{
		return messageTime;
	}
	public void setMessageTime( long messageTime )
	{
		this.messageTime = messageTime;
	}
	public String getMessageText()
	{
		return messageText;
	}
	public void setMessageText( String messageText )
	{
		this.messageText = messageText;
	}

	public long getLastUpdate()
	{
		return lastUpdate;
	}
	public void setLastUpdate( long lastUpdate )
	{
		this.lastUpdate = lastUpdate;
	}
};

/**
 * @author hoge
 */
class MapData
{
	/**
	 *
	 */
	private static int[][] map =
	{
		{2,2,1,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,3,1,1,1,1,1,1,1,1,1,1,1,2,2},
		{2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
		{2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2},
		{2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2}
	};

	/**
	 *
	 */
	public static int[][] getCopyMap()
	{
		return getCopyArray( map );
	}

	/**
	 *
	 */
	protected static int[][] getCopyArray( int[][] array )
	{
		int[][] out = new int[ array.length ][];
		for ( int i = 0; i < array.length; i++ )
		{
			out[i] = new int[ array[i].length ];
			System.arraycopy( array[i], 0, out[i], 0, array[i].length );
		}
		return out;
	}

	/**
	 *
	 */
	public static String mapToString()
	{
		return arrayToString( map );
	}

	/**
	 *
	 */
	protected static String arrayToString( int[][] array )
	{
		StringBuilder builder = new StringBuilder();
		builder.append("\tmap :\n\t[");
		for ( int i = 0; i < array.length; i++ )
		{
			if ( i > 0 ) { builder.append(","); }
			builder.append("\n\t\t[");
			for ( int j = 0; j < array[i].length; j++ )
			{
				if ( j > 0 ) { builder.append(","); }
				builder.append( array[i][j] );
			}
			builder.append("]");
		}
		builder.append("\n\t]");
		return builder.toString();
	}
};

