/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

/**
 * RenderInfo
 */
var RenderInfo =
{
	chipPixels	: 32
};

/**
 * MapInfo
 */
var MapInfo =
{
	maxValue		: 1000,
	chipLevel		: 0,
	chipTrLevel		: 1,
	actorLevel		: 2
};

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE		: "<p>" +
						  "＜キー入力ができない場合はゲームの画面をクリック＞<br>" +
						  "[↑]: 左上, [←]: 左下, [↓]: 右下, [→]: 右上<br>" +
						  "[Z] アイテム<br>" +
						  "</p>",
		CONTINUE_MSG	: "(hit any key to continue)"
	},
	Color :
	{
		TEXTBOX			: "rgba(192, 80, 77, 0.7)",
		ITEMBOX			: "rgba(72, 61, 139, 0.7)",
		BORDER			: "rgb(192, 192, 192)",
		FONT_TEXTBOX	: "rgb(255, 255, 255)",
		FONT_ITEMBOX	: "rgb(255, 255, 255)",
		ITEM_FOCUS		: "rgba(100, 149, 237, 0.7)",
	},
	Font :
	{
		LARGE			: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL			: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1
};

/**
 * EventType
 */
var EventType =
{
	NONE	: "none",
	TEXT	: "text"
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command		= scene.command;	// input

	this.map			= null;
	this.getMapHeight	= function() { return this.map.length; };
	this.getMapWidth	= function() { return this.map[0].length; };

	this.scroll			= { x : 0, y : 0 };
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, map )
	{
		this.map	= map;
	};

	/**
	 * 
	 */
	pt.setChipValue = function( x, y, level, value )
	{
		var values	= [];
		for ( var i = 0; i < 3; i++ )
		{
			values[i] = ( level == i ) ? value : this.getChipValue( x, y, i );
			for ( var n = 0; n < i; n++ )
			{
				values[i] *= MapInfo.maxValue;
			}
		}
		var retValue = 0;
		for ( var i = 0; i < values.length; i++ )
		{
			retValue += values[i];
		}
		this.map[ y ][ x ] = retValue;
	};
	/**
	 * 
	 */
	pt.getChipValue = function( x, y, level )
	{
		var val = this.map[ y ][ x ];
		for ( var i = 0; i < level; i++ )
		{
			val /= MapInfo.maxValue;
		}
		return Math.floor( val ) % MapInfo.maxValue;
	};

	/**
	 * 
	 */
	pt.setActorId = function( x, y, id )
	{
		this.setChipValue( x, y, MapInfo.actorLevel, ( id + 1 ) );
	};
	/**
	 * 
	 */
	pt.clearActorId = function( x, y )
	{
		this.setChipValue( x, y, MapInfo.actorLevel, 0 );
	};
	/**
	 * 
	 */
	pt.getActorId = function( x, y )
	{
		var val = this.getChipValue( x, y, MapInfo.actorLevel );
		return ( val - 1 );
	};

	/**
	 * 
	 */
	pt.updateScrollValue = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;
		var viewPoint	= scene.viewPoint;

		var cx		= canvas.width  / RenderInfo.chipPixels / 2;
		var cy		= canvas.height / RenderInfo.chipPixels / 2;
		var max_sx	= this.getMapWidth()  - ( cx * 2 );
		var max_sy	= this.getMapHeight() - ( cy * 2 );

		// scroll.x
		this.scroll.x	= viewPoint.ad.x - cx;
		if ( this.scroll.x < 0 )
		{
			this.scroll.x	= 0;
		}
		else if ( max_sx < this.scroll.x )
		{
			this.scroll.x	= max_sx;
		}
		// scroll.y
		this.scroll.y	= viewPoint.ad.y - cy;
		if ( this.scroll.y < 0 )
		{
			this.scroll.y	= 0;
		}
		else if ( max_sy < this.scroll.y )
		{
			this.scroll.y	= max_sy;
		}
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;
		var actorList	= scene.actorListTask;

		context.save();
		//context.clearRect( 0, 0, canvas.width, canvas.height );

		// update scroll
		this.updateScrollValue( scene );

		var cells_w	= canvas.width  / RenderInfo.chipPixels;
		var cells_h	= canvas.height / RenderInfo.chipPixels;

		var scroll_x	= Math.floor( this.scroll.x );
		var scroll_y	= Math.floor( this.scroll.y );
		var offset_x	= Math.floor( ( this.scroll.x % 1 ) * RenderInfo.chipPixels );
		var offset_y	= Math.floor( ( this.scroll.y % 1 ) * RenderInfo.chipPixels );

		// チップを描画
		for ( var i = 0; i <= cells_h; i++ )
		{
			for ( var j = 0; j <= cells_w; j++ )
			{
				if ( i == cells_h && offset_y == 0 ) { break; }
				if ( j == cells_w && offset_x == 0 ) { continue; }

				var mx = scroll_x + j;
				var my = scroll_y + i;

				var image	= resource.image.mchip.data;
				var value	= this.getChipValue( mx, my, MapInfo.chipLevel );
				var cw		= image.width / RenderInfo.chipPixels;
				var cx		= value % cw;
				var cy		= Math.floor( value / cw );
				this.drawChip( context, image, i, j, offset_x, offset_y, cx, cy );
			}
		}
		// 透明チップ、アクターを描画
		for ( var i = 0; i <= cells_h; i++ )
		{
			for ( var j = 0; j <= cells_w; j++ )
			{
				if ( i == cells_h && offset_y == 0 ) { break; }
				if ( j == cells_w && offset_x == 0 ) { continue; }

				var mx = scroll_x + j;
				var my = scroll_y + i;

				// draw chip
				var image	= resource.image.mchip_tr.data;
				var value	= this.getChipValue( mx, my, MapInfo.chipTrLevel );
				if ( value > 0 )
				{
					var cw		= image.width / RenderInfo.chipPixels;
					var cx		= value % cw;
					var cy		= Math.floor( value / cw );
					this.drawChip( context, image, i, j, offset_x, offset_y, cx, cy );
				}
				// draw actor
				var id		= this.getActorId( mx, my );
				if ( id >= 0 )
				{
					var actor = actorList.getActor( id );
					if ( actor )
					{
						actor.draw( scene );
						actor.drawFlag = true;
					}
				}
			}
		}
		context.restore();
	};
	/**
	 * 
	 */
	pt.drawChip = function( context, image, i, j, offset_x, offset_y, cx, cy )
	{
		var canvas		= context.canvas;

		var sx	= cx * RenderInfo.chipPixels;
		var sy	= cy * RenderInfo.chipPixels;
		var dx	= j * RenderInfo.chipPixels;
		var dy	= i * RenderInfo.chipPixels;
		var fw	= RenderInfo.chipPixels;
		var fh	= RenderInfo.chipPixels;

		// x
		dx -= offset_x;
		if ( dx < 0 )
		{
			dx = 0;
			sx += offset_x;
			fw -= offset_x;
		}
		if ( ( dx + fw ) > canvas.width )
		{
			fw = offset_x;
		}
		// y
		dy -= offset_y;
		if ( dy < 0 )
		{
			dy = 0;
			sy += offset_y;
			fh -= offset_y;
		}
		if ( ( dy + fh ) > canvas.height )
		{
			fh = offset_y;
		}
		// draw
		context.drawImage( image, sx, sy, fw, fh, dx, dy, fw, fh );
	};
})( StageTask.prototype );

/**
 * ActorListTask
 */
var ActorListTask = function( scene )
{
	this.command	= scene.command;	// input
};
ActorListTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, actors )
	{
		this.child	= null;

		// load actorList
		for ( var i = 0; i < actors.length; i++ )
		{
			var task	= new ActorTask();
			task.init( scene, i, actors[i] );
			// add
			if ( this.child == null )
			{
				this.child = task;
			}
			else
			{
				this.child.append( task );
			}
		}
	};
	/**
	 * 
	 */
	pt.getActor = function( id )
	{
		for ( var actor = this.child; actor != null; actor = actor.next )
		{
			if ( actor.id == id )
			{
				return actor;
			}
		}
		return null;
	};
	/**
	 * 
	 */
	pt.processAction = function( scene )
	{
		for ( var actor = this.child; actor != null; actor = actor.next )
		{
			actor.processAction( scene );
			// clear draw flag
			actor.drawFlag	= false;
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		this.processAction( scene );

		return this.updateChildren( scene );
	};
})( ActorListTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	STATIC	: 0,
	ACTIVE	: 1,
	MORTAL	: 2
};

/**
 * ActorTask
 */
var ActorTask = function()
{
	this.animator	= null;
	this.id			= 0;
	// actor data
	this.ad			= null;
	// step data
	this.step		=
	{
		goal	: { x : 0, y : 0, dir : 0 },
		count	: 0,
		locked	: false
	}

	this.drawFlag	= false;
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.DIR2STEP		= [ { x : 1, y : 1 }, { x : 1, y : -1 }, { x : -1, y : -1 }, { x : -1, y : 1 } ];
	pt.DIR2SY		= [ 1, 3, 2, 0 ];
	pt.STEP_VAL		= 0.4;
	pt.STEP_COUNT	= 4;

	/**
	 * 
	 */
	pt.init = function( scene, id, ad )
	{
		this.id		= id;
		this.ad		= ad;

		this.moveTo( scene.stageTask, this.ad.x, this.ad.y );
		this.command	= ( id == 0 ) ? scene.command : new AICommand();

		var image		= scene.data.resource.image.actors[ this.ad.type ].data;
		var animation	= scene.data.animation;
		this.interfaces	=
		[
			{ update : this.updateActive,	animator : new ImageAnimator( image, animation.actorAction ) },
			{ update : this.updateActive,	animator : new ImageAnimator( image, animation.actorAction ) }
		];
		this.setStatus( this.ad.status );
	};
	/**
	 * 
	 */
	pt.getGoalPos = function( val )
	{
		var step = this.DIR2STEP[ this.ad.dir ];
		var pos =
		{
			x : this.ad.x + val * step.x,
			y : this.ad.y + val / 2 * step.y
		};
		return pos;
	};
	/**
	 * 
	 */
	pt.moveTo = function( stage, x, y )
	{
		stage.clearActorId( Math.floor( this.ad.x ), Math.floor( this.ad.y ) );
		this.ad.x = x;
		this.ad.y = y;
		stage.setActorId( Math.floor( this.ad.x ), Math.floor( this.ad.y ), this.id );
	};
	/**
	 * 
	 */
	pt.canMoveTo = function( stage, x, y )
	{
		var arr = [ -0.3, 0.3 ];

		for ( var i = 0; i < arr.length; i++ )
		{
			var my = Math.floor( y + arr[i] );
			if ( my < 0 || stage.getMapHeight() <= my ) { return false; }

			for ( var j = 0; j < arr.length; j++ )
			{
				var mx = Math.floor( x + arr[j] );
				if ( mx < 0 || stage.getMapWidth() <= mx ) { return false; }

				var id = stage.getActorId( mx, my );
				if ( id >= 0 && id != this.id )
				{
					return false;
				}
			}
		}
		return true;
	};
	/**
	 * 
	 */
	pt.processAction = function( scene )
	{
		if ( this.status == ActorStatus.ACTIVE && !this.step.locked )
		{
			this.command.update();

			// key
			if ( this.command.tbl.up )
			{
				this.ad.dir			= 2;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.left )
			{
				this.ad.dir			= 3;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.down )
			{
				this.ad.dir			= 0;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.right )
			{
				this.ad.dir			= 1;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
		}
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var upd = false;
		var stage	= scene.stageTask;

		// step
		if ( this.step.count != 0 )
		{
			this.step.count--;
			if ( this.step.count == 0 ) { this.step.locked = false; }
			var newX = ( this.step.count ) ? ( this.ad.x + this.step.goal.x ) / 2 : this.step.goal.x;
			var newY = ( this.step.count ) ? ( this.ad.y + this.step.goal.y ) / 2 : this.step.goal.y;

			if ( this.canMoveTo( scene.stageTask, newX, this.ad.y ) )
			{
				this.moveTo( stage, newX, this.ad.y );
				upd = true;
			}
			if ( this.canMoveTo( scene.stageTask, this.ad.x, newY ) )
			{
				this.moveTo( stage, this.ad.x, newY );
				upd = true;
			}
		}
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			var r = Math.floor( Math.random() * 10 );
			if ( r == 1 )
			{
				this.animator.start();
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateMortal = function( scene )
	{
		var upd = false;
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context	= scene.context;
		var canvas	= context.canvas;
		var stage	= scene.stageTask;

		var image	= this.animator.image;
		var x		= Math.floor( ( this.ad.x - stage.scroll.x ) * RenderInfo.chipPixels );
		var y		= Math.floor( ( this.ad.y - stage.scroll.y ) * RenderInfo.chipPixels );
		var sy		= this.DIR2SY[ this.ad.dir ] * this.animator.fh;

		context.drawImage(
			image,
			this.animator.fx,			// sx
			sy,							// sy
			this.animator.fw,			// sw
			this.animator.fh,			// sh
			x - this.animator.fw / 2,	// dx
			y - this.animator.fh / 2,	// dy
			this.animator.fw,			// dw
			this.animator.fh			// dh
		);
	};
})( ActorTask.prototype );

/**
 * EventTask
 */
var EventTask = function( scene )
{
	this.command	= scene.command;	// input
	this.prevId		= -1;
};
EventTask.prototype = new Task();

(function( pt )
{
	pt.EVENT_DIST	= 1.5;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd			= false;

		var stage		= scene.stageTask;
		var actorList	= scene.actorListTask;
		var viewPoint	= scene.viewPoint;
		var msgManager	= scene.msgManager;
		var resource	= scene.data.resource;
		var sound		= scene.soundTask;

		// 画面遷移
		if ( this.command.tbl.escape )
		{
			var message	= new Message( MessageType.RET_PREV_SCENE );
			msgManager.postBreakMessage( message );

			return true;
		}
		// アイテム
		else if ( this.command.tbl.item )
		{
			sound.prepare( resource.sound.notice.data );
			// add task
			var task = new ItemTask( scene );
			task.link( this );
			// setFocus
			scene.setFocus( task );

			return true;
		}

		var ax		= Math.floor( viewPoint.ad.x );
		var ay		= Math.floor( viewPoint.ad.y );
		var step	= viewPoint.DIR2STEP[ viewPoint.ad.dir ];
		var pvals	=
		[
			{ x : step.x, y : step.y }, { x : 0, y : step.y }, { x : step.x, y : 0 }
		];
		var mw		= stage.getMapWidth();
		var mh		= stage.getMapHeight();
		var curId	= -1;

		// check near
		for ( var i = 0; i < pvals.length; i++ )
		{
			var mx	= ax + pvals[i].x;
			var my	= ay + pvals[i].y;
			if ( mx < 0 || mw <= mx || my < 0 || mh <= my ) { continue; }

			var id	= stage.getActorId( mx, my );
			// check id
			if ( id > 0 )
			{
				if ( id == this.prevId )
				{
					curId	= id;
					continue;
				}

				var actor	= actorList.getActor( id );
				// check distance
				if ( Math.abs( viewPoint.ad.x - actor.ad.x ) < this.EVENT_DIST &&
					 Math.abs( viewPoint.ad.y - actor.ad.y ) < this.EVENT_DIST )
				{
					curId	= id;

					// text box
					if ( actor.ad.event.type == EventType.TEXT )
					{
						sound.prepare( resource.sound.notice.data );

						// add task
						var task = new TextBoxTask( scene );
						task.link( this );
						task.texts.push( actor.ad.event.text );
						// setFocus
						scene.setFocus( task );

						upd = true;
					}
					break;
				}
			}
		}
		this.prevId	= curId;

		return upd;
	};
})( EventTask.prototype );

/**
 * TextBoxTask
 */
var TextBoxTask = function( scene )
{
	this.command	= scene.command;	// input
	this.texts		= [];

	this.lag		= this.LAG_VAL;
};
TextBoxTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 5;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( scene.getFocus() == this )
		{
			if ( this.lag > 0 )
			{
				this.lag--;
			}
			else if ( this.command.tbl.any )
			{
				scene.setFocus( null );
				this.unlink();
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;

		var hh = 10;
		var fh = 40;
		var th = 20;
		var bx = 10;
		var by = 10;
		var bh = hh + th * this.texts.length + fh;
		var bw = canvas.width - bx * 2;

		context.save();

		// rect
		context.fillStyle	= Res.Color.TEXTBOX;
		context.fillRect( bx, by, bw, bh );
		// border
		context.strokeStyle	= Res.Color.BORDER;
		context.beginPath();
		context.strokeRect( bx, by, bw, bh );
		// text
		context.font		= Res.Font.SMALL;
		context.fillStyle	= Res.Color.FONT_TEXTBOX;
		for ( var i = 0; i < this.texts.length; i++ )
		{
			context.fillText( this.texts[i], bx + 10, by + hh + th * ( i + 1 ) );
		}
		// continue message
		context.fillText( Res.String.CONTINUE_MSG, bx + 10, by + bh - 10 );

		context.restore();
	};
})( TextBoxTask.prototype );

/**
 * ItemTask
 */
var ItemTask = function( scene )
{
	this.command	= scene.command;	// input
	this.select		= 0;

	this.lag		= this.LAG_VAL;
};
ItemTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 5;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		var items	= scene.data.items;

		if ( scene.getFocus() == this )
		{
			if ( this.lag > 0 )
			{
				this.lag--;
			}
			else if ( this.command.tbl.up )
			{
				this.select--;
				if ( this.select < 0 )
				{
					this.select = items.length - 1;
				}
				this.lag = this.LAG_VAL;
				upd = true;
			}
			else if ( this.command.tbl.down )
			{
				this.select++;
				if ( this.select >= items.length )
				{
					this.select = 0;
				}
				this.lag = this.LAG_VAL;
				upd = true;
			}
			else if ( this.command.tbl.enter )
			{
				var item	= items[ this.select ];
				// add task
				var task = new TextBoxTask( scene );
				task.link( this );
				task.texts.push( item.msg );
				// setFocus
				scene.setFocus( task );

				this.unlink();
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var items		= scene.data.items;

		var hh = 30;
		var th = 20;
		var bx = 10;
		var by = 10;
		var bh = hh + th * items.length + 10;
		var bw = canvas.width - bx * 2;

		context.save();

		// rect
		context.fillStyle	= Res.Color.ITEMBOX;
		context.fillRect( bx, by, bw, bh );
		// border
		context.strokeStyle	= Res.Color.BORDER;
		context.beginPath();
		context.strokeRect( bx, by, bw, bh );
		// text
		context.fillStyle	= Res.Color.FONT_ITEMBOX;
		context.font		= Res.Font.LARGE;
		context.fillText( "アイテム", bx + 10, by + th );
		context.font		= Res.Font.SMALL;
		context.fillText( "[↑][↓] 選択  [Enter] 決定", bx + 100, by + th );
		for ( var i = 0; i < items.length; i++ )
		{
			if ( i == this.select )
			{
				context.fillStyle	= Res.Color.ITEM_FOCUS;
				context.fillRect( bx + 5, by + hh + 5 + th * i, bw - 10, th );
			}
			context.fillStyle	= Res.Color.FONT_ITEMBOX;
			context.fillText( items[i].name, bx + 10, by + hh + th * ( i + 1 ) );
		}
		context.restore();
	};
})( ItemTask.prototype );


/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "viewPoint.ad.x=" + scene.viewPoint.ad.x + ", viewPoint.ad.y=" + scene.viewPoint.ad.y;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );


/**
 * AICommand
 */
var AICommand = function()
{
	this.tbl.up		= 0;
	this.tbl.left	= 0;
	this.tbl.down	= 0;
	this.tbl.right	= 0;
};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 12 );
		if      ( r == 0 )	{ this.tbl.up		= 1; }
		else if ( r == 1 )	{ this.tbl.left		= 1; }
		else if ( r == 2 )	{ this.tbl.down		= 1; }
		else if ( r == 3 )	{ this.tbl.right	= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.up		= 0;
	this.tbl.left	= 0;
	this.tbl.down	= 0;
	this.tbl.right	= 0;

	this.tbl.item	= 0;
	this.tbl.seek	= 0;
	this.tbl.enter	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;

	this.tbl.any	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 38: this.tbl.up	= value; event.prevent( event.orgEvent ); break;	// ↑
				case 37: this.tbl.left	= value; event.prevent( event.orgEvent ); break;	// ←
				case 40: this.tbl.down	= value; event.prevent( event.orgEvent ); break;	// ↓
				case 39: this.tbl.right	= value; event.prevent( event.orgEvent ); break;	// →

				case 90: this.tbl.item	= value; break;	// Z
				case 32: this.tbl.seek	= value; break;	// space;
				case 13: this.tbl.enter	= value; break;	// ENTER
				case 27: this.tbl.escape= value; break;	// ESC
				case 81: this.tbl.debug	= value; break;	// Q
			}
			this.tbl.any	= value;	// any key
		}
	};
})( InputCommand.prototype );

/**
 * StageScene
 */
var StageScene = function( app, name, online )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;

	this.command		= new InputCommand();
	this.viewPoint		= null;

	// create task
	this.stageTask		= new StageTask( this );
	this.actorListTask	= new ActorListTask( this );
	this.eventTask		= new EventTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.stageTask.append( this.actorListTask );
	this.stageTask.append( this.eventTask );
	this.stageTask.append( this.soundTask );
	this.stageTask.append( this.debugTask );
	// head of task list
	this.child		= this.stageTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	this.msgManager		= new MessageManager();
	this.msgManager.addHandler( new MessageHandler( MessageType.RET_PREV_SCENE,	this, this.handleRetPrevScene ) );
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleRetPrevScene = function( scene, message )
	{
		this.app.sceneManager.pop();
		var prevScene = this.app.sceneManager.current;
		prevScene.show();
	};

	/**
	 * 
	 */
	pt.init = function()
	{
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data	= data;

		// load stage
		this.stageTask.setData( this, data.map );
		// load actorList
		this.actorListTask.setData( this, data.actors );
		// init viewPoint
		this.viewPoint	= this.actorListTask.child;
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			var self = this;

			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval + 300 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene	= StageScene;

// Anonymous function end
//
})( window );
