/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

var STAGE_RECT		= { x :  10, y :  10, w : 256, h : 256 };
var CHIP_RECT		= { x : 290, y :  10, w : 128, h : 256 };
var CHIP_TR_RECT	= { x : 440, y :  10, w : 128, h : 256 };
var ACTOR_RECT		= { x : 590, y :  10, w :  32, h : 256 };

//...[透過チップ][チップ]
//...000 000
//... actorid 要+1

/**
 * RenderInfo
 */
var RenderInfo =
{
	mapGridPixels	: 16,
	chipGridPixels	: 32,
	chipPixels		: 32,
	actorFramesW	: 3,
	actorFramesH	: 4
};

/**
 * MapInfo
 */
var MapInfo =
{
	maxValue		: 1000,
	chipLevel		: 0,
	chipTrLevel		: 1,
	actorLevel		: 2
};

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE		: "<p>" +
						  "マウスでチップを選択して、マップ画面をクリック・ドラッグします。<br>" +
						  "スクロールバーのボタンをクリックするとスクロールします。<br>" +
						  "（詳しい使い方はページ下の「help」を参照してください。）" +
						  "</p>",
		ID_DATA			: "data",
		DEFAULT_NAME	: "サンダー虎島"
	},
	Color :
	{
		PANEL				: "gainsboro",
		BACKGROUND			: "black",
		BORDER				: "gray",
		BORDER_BUTTON_ON	: "black",
		BORDER_BUTTON_OFF	: "gray",
		GRIDLINE			: "white",
		SELECTED			: "crimson",
		ACTOR_0				: "springgreen"
	},
	Font :
	{
		LARGE		: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL		: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1
};

/**
 * EventType
 */
var EventType =
{
	NONE	: "none",
	TEXT	: "text",
	BATTLE	: "battle",
	SHOP	: "shop",
	WARP	: "warp"
};


/**
 * PanelTask
 */
var PanelTask = function( scene )
{
	this.command	= scene.command;	// input
};
PanelTask.prototype = new Task();

(function( pt )
{
	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;

		context.save();

		context.fillStyle	= Res.Color.PANEL;
		context.fillRect( 0, 0, canvas.width, canvas.height );

		context.restore();
	};
})( PanelTask.prototype );

/**
 * EditorTask
 */
var EditorTask = function()
{
	this.resource		= null;
	this.gridPixels		= RenderInfo.mapGridPixels;

	this.rect			= null;		// ピクセル単位 ( x, y, w, h ), Canvas 基準
	this.scroll			= null;		// グリッド単位 ( x, y )
	this.selected		= null;		// グリッド単位 ( x, y )

	this.getVisibleSize	= null;		// グリッド単位 ( w, h )
	this.getMaxScroll	= null;		// グリッド単位 ( w, h )
	this.getSelectedVal	= null;		// y * w + x
};
EditorTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setScroll = function( sx, sy )
	{
		var ms	= this.getMaxScroll();
		if ( 0 <= sx && sx <= ms.w &&
			 0 <= sy && sy <= ms.h )
		{
			this.scroll.x = sx;
			this.scroll.y = sy;
		}
	};
	/**
	 * 
	 */
	pt.drawBackground = function( context )
	{
		context.fillStyle = Res.Color.BACKGROUND;
		context.fillRect( this.rect.x, this.rect.y, this.rect.w, this.rect.h );
	};
	/**
	 * 
	 */
	pt.drawGridLine = function( context )
	{
		var vs	= this.getVisibleSize();
		var lw	= vs.w * this.gridPixels;
		var lh	= vs.h * this.gridPixels;

		context.strokeStyle = Res.Color.GRIDLINE;
		for ( var i = 1; i <= vs.h; i++ )
		{
			var y = this.rect.y + ( i * this.gridPixels );
			context.beginPath();
			context.moveTo( this.rect.x, y );
			context.lineTo( this.rect.x + lw, y );
			context.stroke();
		}
		for ( var j = 1; j <= vs.w; j++ )
		{
			var x = this.rect.x + ( j * this.gridPixels );
			context.beginPath();
			context.moveTo( x, this.rect.y );
			context.lineTo( x, this.rect.y + lh );
			context.stroke();
		}
	};
	/**
	 * 
	 */
	pt.drawSelected = function( context )
	{
		// selected
		context.strokeStyle	= Res.Color.SELECTED;
		context.beginPath();
		context.strokeRect(
			this.rect.x + this.selected.x * this.gridPixels,
			this.rect.y + this.selected.y * this.gridPixels,
			this.gridPixels,
			this.gridPixels
		);
		context.beginPath();
		context.strokeRect(
			this.rect.x + this.selected.x * this.gridPixels + 1,
			this.rect.y + this.selected.y * this.gridPixels + 1,
			this.gridPixels - 2,
			this.gridPixels - 2
		);
	};
	/**
	 * 
	 */
	pt.drawBorder = function( context )
	{
		context.strokeStyle	= Res.Color.BORDER;
		context.beginPath();
		context.strokeRect( this.rect.x, this.rect.y, this.rect.w, this.rect.h );
	};
})( EditorTask.prototype );

/**
 * ScrollType
 */
var ScrollType =
{
	HORIZONTAL	: 0,
	VERTICAL	: 1
};

/**
 * ScrollTask
 */
var ScrollTask = function( scene, type, target )
{
	this.command		= scene.command;	// input

	this.type			= type;
	this.target			= target;
	this.rect			= {};		// ピクセル単位 ( x, y, w, h ), Canvas 基準
	this.buttons =					// ピクセル単位 ( x, y, w, h ), Canvas 基準
	[
		{ value : -1, on : 0 },
		{ value :  1, on : 0 }
	]
	this.initRect();

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ScrollTask.prototype = new Task();

(function( pt )
{
	pt.BAR_LEN		= 18;
	pt.BUTTON_LEN	= 64;

	/**
	 * 
	 */
	pt.initRect = function()
	{
		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			// rect
			this.rect.x	= this.target.rect.x;
			this.rect.y	= this.target.rect.y + this.target.rect.h;
			this.rect.w	= this.target.rect.w;
			this.rect.h	= this.BAR_LEN;
			// buttons
			this.buttons[0].x	= this.rect.x;
			this.buttons[0].y	= this.rect.y;
			this.buttons[0].w	= this.BUTTON_LEN;
			this.buttons[0].h	= this.rect.h;
			this.buttons[1].x	= this.rect.x + this.rect.w - this.BUTTON_LEN;
			this.buttons[1].y	= this.rect.y;
			this.buttons[1].w	= this.BUTTON_LEN;
			this.buttons[1].h	= this.rect.h;
			break;
		case ScrollType.VERTICAL :
			this.rect.x	= this.target.rect.x + this.target.rect.w;
			this.rect.y	= this.target.rect.y;
			this.rect.w	= this.BAR_LEN;
			this.rect.h	= this.target.rect.h;
			// buttons
			this.buttons[0].x	= this.rect.x;
			this.buttons[0].y	= this.rect.y;
			this.buttons[0].w	= this.rect.w;
			this.buttons[0].h	= this.BUTTON_LEN;
			this.buttons[1].x	= this.rect.x;
			this.buttons[1].y	= this.rect.y + this.rect.h - this.BUTTON_LEN;
			this.buttons[1].w	= this.rect.w;
			this.buttons[1].h	= this.BUTTON_LEN;
			break;
		}
	};
	/**
	 * 
	 */
	pt.getSliderRect = function()
	{
		var vs		= this.target.getVisibleSize();
		var ms		= this.target.getMaxScroll();
		var scroll	= this.target.scroll;
		var sarea	= {};
		var slider	= {};

		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			// sarea
			sarea.x		= this.rect.x + this.BUTTON_LEN;
			sarea.w		= this.rect.w - ( this.BUTTON_LEN * 2 );
			// slider
			slider.w	= Math.floor( sarea.w * vs.w / ( vs.w + ms.w ) );
			slider.h	= this.rect.h;
			slider.x	= ( ms.w > 0 ) ? ( sarea.x + ( sarea.w - slider.w ) * scroll.x / ms.w ) : sarea.x;
			slider.y	= this.rect.y;
			break;
		case ScrollType.VERTICAL :
			// sarea
			sarea.y		= this.rect.y + this.BUTTON_LEN;
			sarea.h		= this.rect.h - ( this.BUTTON_LEN * 2 );
			// slider
			slider.w	= this.rect.w;
			slider.h	= Math.floor( sarea.h * vs.h / ( vs.h + ms.h ) );
			slider.x	= this.rect.x;
			slider.y	= ( ms.h > 0 ) ? ( sarea.y + ( sarea.h - slider.h ) * scroll.y / ms.h ) : sarea.y;
			break;
		}
		return slider;
	};
	/**
	 * 
	 */
	pt.addScroll = function( value )
	{
		var scroll	= this.target.scroll;

		switch ( this.type )
		{
		case ScrollType.HORIZONTAL :
			this.target.setScroll( scroll.x + value, scroll.y );
			break;
		case ScrollType.VERTICAL :
			this.target.setScroll( scroll.x, scroll.y + value );
			break;
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var mi	= this.command.mouseInfo;

			// update button status
			for ( var i = 0; i < this.buttons.length; i++ )
			{
				var button = this.buttons[i];
				if ( button.x <= mi.x && mi.x <= button.x + button.w &&
					 button.y <= mi.y && mi.y <= button.y + button.h )
				{
					button.on	= 1;
					this.addScroll( button.value );
					upd = true;
					break;
				}
			}
		}
		else
		{
			// set button off
			for ( var i = 0; i < this.buttons.length; i++ )
			{
				var button = this.buttons[i];
				if ( button.on ) { upd = true; }
				button.on	= 0;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;

		context.save();

		// background
		context.fillStyle	= Res.Color.BACKGROUND;
		context.fillRect( this.rect.x + 2, this.rect.y + 2, this.rect.w - 4, this.rect.h - 4 );

		// border
		context.strokeStyle	= Res.Color.BORDER;
		context.beginPath();
		context.strokeRect( this.rect.x + 2, this.rect.y + 2, this.rect.w - 4, this.rect.h - 4 );

		// slider
		var slider = this.getSliderRect();
		context.fillStyle	= Res.Color.PANEL;
		context.fillRect( slider.x, slider.y, slider.w, slider.h );

		context.strokeStyle	= Res.Color.BORDER;
		context.beginPath();
		context.strokeRect( slider.x, slider.y, slider.w, slider.h );

		// button
		for ( var i = 0; i < this.buttons.length; i++ )
		{
			var button = this.buttons[i];
			context.fillStyle	= Res.Color.PANEL;
			context.fillRect( button.x, button.y, button.w, button.h );

			context.strokeStyle	= ( button.on ) ? Res.Color.BORDER_BUTTON_ON : Res.Color.BORDER_BUTTON_OFF;
			context.beginPath();
			context.strokeRect( button.x, button.y, button.w, button.h );
		}

		context.restore();
	};
})( ScrollTask.prototype );

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.mapGridPixels;
	this.rect			= STAGE_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };
	this.currentTool	= null;

	this.map			= null;
	this.getMapHeight	= function() { return this.map.length; };
	this.getMapWidth	= function() { return this.map[0].length; };
	this.actors			= null;

	this.draggedValue	= -1;

	// scroll task
	this.hscrollTask	= new ScrollTask( scene, ScrollType.HORIZONTAL, this );
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	// create list
	this.child			= this.hscrollTask;
	this.child.append( this.vscrollTask );

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
StageTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, map, actors, resource )
	{
		this.map		= map;
		this.actors		= actors;
		this.resource	= resource;

		// set actor value
		this.putActors();
		// default pallete
		this.currentTool	= scene.chipTask;
	};

	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vw	= this.rect.w / this.gridPixels;
		var vh	= this.rect.h / this.gridPixels;
		var mw	= this.getMapWidth();
		var mh	= this.getMapHeight();
		if ( mw < vw )
		{
			vw = mw;
		}
		if ( mh < vh )
		{
			vh = mh;
		}
		return { w : vw, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var mw	= this.getMapWidth()  - this.rect.w / this.gridPixels;
		var mh	= this.getMapHeight() - this.rect.h / this.gridPixels;
		if ( mw < 0 )
		{
			mw = 0;
		}
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : mw, h : mh };
	};
	/**
	 * 
	 */
	pt.getSelectedVal = function()
	{
		var x	= this.scroll.x + this.selected.x;
		var y	= this.scroll.y + this.selected.y;
		var w	= this.getMapWidth();
		return ( ( y * w ) + x );
	};

	/**
	 * 
	 */
	pt.setChipValue = function( x, y, level, value )
	{
		var values	= [];
		for ( var i = 0; i < 3; i++ )
		{
			values[i] = ( level == i ) ? value : this.getChipValue( x, y, i );
			for ( var n = 0; n < i; n++ )
			{
				values[i] *= MapInfo.maxValue;
			}
		}
		var retValue = 0;
		for ( var i = 0; i < values.length; i++ )
		{
			retValue += values[i];
		}
		this.map[ y ][ x ] = retValue;
	};
	/**
	 * 
	 */
	pt.getChipValue = function( x, y, level )
	{
		var val = this.map[ y ][ x ];
		for ( var i = 0; i < level; i++ )
		{
			val /= MapInfo.maxValue;
		}
		return Math.floor( val ) % MapInfo.maxValue;
	};

	/**
	 * 
	 */
	pt.setActorId = function( x, y, id )
	{
		this.setChipValue( x, y, MapInfo.actorLevel, ( id + 1 ) );
	};
	/**
	 * 
	 */
	pt.clearActorId = function( x, y )
	{
		this.setChipValue( x, y, MapInfo.actorLevel, 0 );
	};
	/**
	 * 
	 */
	pt.getActorId = function( x, y )
	{
		var val = this.getChipValue( x, y, MapInfo.actorLevel );
		return ( val - 1 );
	};

	/**
	 * 
	 */
	pt.putActors = function()
	{
		for ( var i = 0; i < this.actors.length; i++ )
		{
			var actor	= this.actors[i];
			var x		= Math.floor( actor.x );
			var y		= Math.floor( actor.y );
			this.setActorId( x, y, i );
		}
	};

	/**
	 * 
	 */
	pt.writeChip = function( mx, my, level, value )
	{
		var upd = false;

		// drag actor
		if ( this.draggedValue >= 0 )
		{
			var actor	= this.actors[ this.draggedValue ];
			var x	= Math.floor( actor.x );
			var y	= Math.floor( actor.y );
			if ( x != mx || y != my )
			{
				// clear old
				this.clearActorId( x, y );
				// update new
				actor.x		= mx + 0.5;
				actor.y		= my + 0.5;
				this.setActorId( mx, my, this.draggedValue );

				upd = true;
			}
		}
		else
		{
			// チップの更新
			var curValue = this.getChipValue( mx, my, level );
			if ( curValue != value )
			{
				this.setChipValue( mx, my, level, value );
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.writeActor = function( mx, my, value )
	{
		var upd = false;

		var id		= this.getActorId( mx, my );
		var type	= value - 1;

		// no actor
		if ( id < 0 )
		{
			// not erase >> add
			if ( value > 0 )
			{
				// copy
				var actorSrc	= this.actors[0];
				var actorDst	= {};
				for ( var key in actorSrc )
				{
					actorDst[ key ] = actorSrc[ key ];
				}
				// customize
				actorDst.type	= type;
				actorDst.name	= Res.String.DEFAULT_NAME;
				actorDst.x		= mx + 0.5;
				actorDst.y		= my + 0.5;
				// add
				this.actors.push( actorDst );
				this.putActors();
	
				upd = true;
			}
		}
		// actor exists
		else
		{
			// erase
			if ( value == 0 )
			{
				// can not erase actors[0]
				if ( id > 0 )
				{
					this.clearActorId( mx, my );
					this.actors.splice( id, 1 );
					this.putActors();
					upd = true;
				}
			}
			// replace
			else
			{
				this.actors[ id ].type	= type;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sx	= Math.floor( this.command.mouseInfo.ox / this.gridPixels );
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );

			if ( sx < vs.w && sy < vs.h )
			{
				// update selection
				if ( ( sx != this.selected.x || sy != this.selected.y ) )
				{
					this.selected.x = sx;
					this.selected.y = sy;
					upd = true;
				}

				// update map
				var value	= this.currentTool.getSelectedVal();
				var mx		= sx + this.scroll.x;
				var my		= sy + this.scroll.y;
				if ( this.currentTool == scene.chipTask )
				{
					if ( this.writeChip( mx, my, MapInfo.chipLevel, value ) )	{ upd = true; }
				}
				else if ( this.currentTool == scene.chipTrTask )
				{
					if ( this.writeChip( mx, my, MapInfo.chipTrLevel, value ) )	{ upd = true; }
				}
				else
				{
					if ( this.writeActor( mx, my, value ) )	{ upd = true; }
				}

				// set drag param
				if ( this.draggedValue == -1 )
				{
					this.draggedValue	= this.getActorId( mx, my );
				}
			}
		}
		else
		{
			// clear drag param
			if ( this.draggedValue != -1 )
			{
				this.draggedValue	= -1;
			}
		}

		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// map
		for ( var i = 0; i < vs.h; i++ )
		{
			for ( var j = 0; j < vs.w; j++ )
			{
				var mx		= this.scroll.x + j;
				var my		= this.scroll.y + i;

				var value;
				var image;
				var sx, sy;
				var sw	= RenderInfo.chipPixels;
				var sh	= RenderInfo.chipPixels;
				var dx	= this.rect.x + ( j * this.gridPixels );
				var dy	= this.rect.y + ( i * this.gridPixels );
				var dw	= this.gridPixels;
				var dh	= this.gridPixels;
				var cw;

				// draw chip
				value	= this.getChipValue( mx, my, MapInfo.chipLevel );
				image	= this.resource.image.mchip.data;
				cw		= image.width / RenderInfo.chipPixels;
				sx		= ( value % cw ) * RenderInfo.chipPixels;
				sy		= Math.floor( value / cw ) * RenderInfo.chipPixels;
				context.drawImage( image, sx, sy, sw, sh, dx, dy, dw, dh );
				// draw tr chip
				value	= this.getChipValue( mx, my, MapInfo.chipTrLevel );
				if ( value > 0 )
				{
					image	= this.resource.image.mchip_tr.data;
					cw		= image.width / RenderInfo.chipPixels;
					sx		= ( value % cw ) * RenderInfo.chipPixels;
					sy		= Math.floor( value / cw ) * RenderInfo.chipPixels;
					context.drawImage( image, sx, sy, sw, sh, dx, dy, dw, dh );
				}
				// draw actor
				var id	= this.getActorId( mx, my );
				if ( id >= 0 )
				{
					image	= this.resource.image.actors[ this.actors[ id ].type ].data;
					sx		= 0;
					sy		= 0;
					sw		= Math.floor( image.width / RenderInfo.actorFramesW );
					sh		= Math.floor( image.height / RenderInfo.actorFramesH );
					if ( id == 0 )
					{
						context.fillStyle	= Res.Color.ACTOR_0;
						context.fillRect( dx + 3, dy + 3, dw - 6, dh - 6 );
					}
					context.drawImage( image, sx, sy, sw, sh, dx, dy, dw, dh );
				}
			}
		}
		// grid line
		this.drawGridLine( context );
		// selected
		this.drawSelected( context );
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( StageTask.prototype );

/**
 * ChipType
 */
var ChipType =
{
	NORMAL		: 0,
	TRANSPARENT	: 1
};

/**
 * ChipTask
 */
var ChipTask = function( scene, type )
{
	this.type			= type;
	this.image			= null;

	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.chipGridPixels;
	this.rect			= ( this.type == ChipType.NORMAL ) ? CHIP_RECT : CHIP_TR_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };

	// scroll task
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.child			= this.vscrollTask;


	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ChipTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, resource )
	{
		this.resource	= resource;

		this.image		= ( this.type == ChipType.NORMAL ) ?
							resource.image.mchip.data : resource.image.mchip_tr.data;
	};
	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vw	= this.rect.w / this.gridPixels;
		var iw	= this.image.width / RenderInfo.chipPixels;
		if ( iw < vw )
		{
			vw = iw;
		}
		var vh	= this.rect.h / this.gridPixels;
		var ih	= this.image.height / RenderInfo.chipPixels;
		if ( ih < vh )
		{
			vh = ih;
		}
		return { w : vw, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var vw	= this.rect.w / this.gridPixels;
		var iw	= this.image.width / RenderInfo.chipPixels;
		var mw	= iw - vw;
		if ( mw < 0 )
		{
			mw = 0;
		}
		var vh	= this.rect.h / this.gridPixels;
		var ih	= this.image.height / RenderInfo.chipPixels;
		var mh	= ih - vh;
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : mw, h : mh };
	};
	/**
	 * 
	 */
	pt.getSelectedVal = function()
	{
		var x	= this.scroll.x + this.selected.x;
		var y	= this.scroll.y + this.selected.y;
		var w	= this.image.width / RenderInfo.chipPixels;
		return ( ( y * w ) + x );
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sx	= Math.floor( this.command.mouseInfo.ox / this.gridPixels );
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );
			if ( ( sx != this.selected.x || sy != this.selected.y || scene.stageTask.currentTool != this )
				 && sx < vs.w && sy < vs.h )
			{
				this.selected.x				= sx;
				this.selected.y				= sy;
				scene.stageTask.currentTool	= this;
				upd = true;
			}
		}
		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// chip
		context.drawImage(
			this.image,
			this.scroll.x * RenderInfo.chipPixels,	// sx
			this.scroll.y * RenderInfo.chipPixels,	// sy
			vs.w * RenderInfo.chipPixels,			// sw
			vs.h * RenderInfo.chipPixels,			// sh
			this.rect.x,							// dx
			this.rect.y,							// dy
			vs.w * this.gridPixels,					// dw
			vs.h * this.gridPixels					// dh
		);
		// grid line
		this.drawGridLine( context );
		// selected
		if ( scene.stageTask.currentTool == this )
		{
			this.drawSelected( context );
		}
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( ChipTask.prototype );

/**
 * ActorTask
 */
var ActorTask = function( scene )
{
	this.command		= scene.command;	// input
	this.gridPixels		= RenderInfo.chipGridPixels;
	this.rect			= ACTOR_RECT;
	this.scroll			= { x : 0, y : 0 };
	this.selected		= { x : 0, y : 0 };

	// scroll task
	this.vscrollTask	= new ScrollTask( scene, ScrollType.VERTICAL, this );
	this.child			= this.vscrollTask;

	// add mouse area
	this.command.addMouseArea( new MouseArea( this, this.rect ) );
};
ActorTask.prototype = new EditorTask();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, resource )
	{
		this.resource	= resource;
	};
	/**
	 * 
	 */
	pt.getVisibleSize = function()
	{
		var vh		= this.rect.h / this.gridPixels;
		var images	= this.resource.image.actors.length + 1;	// +空白セル
		if ( images < vh )
		{
			vh = images;
		}
		return { w : 1, h : vh };
	};
	/**
	 * 
	 */
	pt.getMaxScroll = function()
	{
		var vh		= this.rect.h / this.gridPixels;
		var images	= this.resource.image.actors.length + 1;	// +空白セル
		var mh	= images - vh;
		if ( mh < 0 )
		{
			mh = 0;
		}
		return { w : 0, h : mh };
	};
	/**
	 * 
	 */
	pt.getSelectedVal = function()
	{
		return this.scroll.y + this.selected.y;
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		if ( this.command.mouse && this.command.mouseInfo.target == this )
		{
			var vs	= this.getVisibleSize();
			var sy	= Math.floor( this.command.mouseInfo.oy / this.gridPixels );
			if ( ( sy != this.selected.y || scene.stageTask.currentTool != this ) && sy < vs.h )
			{
				this.selected.y				= sy;
				scene.stageTask.currentTool	= this;
				upd = true;
			}
		}
		// update scrollbar
		if ( !upd )
		{
			upd = this.updateChildren( scene );
		}
		return upd;
	};

	/**
	 * 描画する。
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var actors		= scene.stageTask.actors;
		var vs			= this.getVisibleSize();

		context.save();

		// background
		this.drawBackground( context );

		// actor
		for ( var i = 1; i < vs.h; i++ )	// 空白セルはスキップ
		{
			var image	= this.resource.image.actors[ i - 1 + this.scroll.y ].data;
			var sw		= Math.floor( image.width / RenderInfo.actorFramesW );
			var sh		= Math.floor( image.height / RenderInfo.actorFramesH );
			context.drawImage(
				image,
				0,								// sx
				0,								// sy
				sw,								// sw
				sh,								// sh
				this.rect.x,					// dx
				this.rect.y + ( i * this.gridPixels ),	// dy
				this.gridPixels,				// dw
				this.gridPixels					// dh
			);
		}
		// grid line
		this.drawGridLine( context );
		// selected
		if ( scene.stageTask.currentTool == this )
		{
			this.drawSelected( context );
		}
		// border
		this.drawBorder( context );

		context.restore();

		// draw scrollbar
		this.drawChildren( scene );
	};
})( ActorTask.prototype );

/**
 * FormTask
 */
var FormTask = function( scene )
{
	this.command		= scene.command;	// input
	this.currentValue	= -1;
	this.currentActor	= null;
};
FormTask.prototype = new Task();

(function( pt )
{
	pt.ID_ACTOR_NAME	= "actorName";
	pt.ID_EVENT_TYPE	= "eventType";
	pt.ID_EVENT_PARAM	= "eventParam";

	pt.eventNameId	= {};
	pt.eventNameId[ EventType.NONE ]	= 0;
	pt.eventNameId[ EventType.TEXT ]	= 1;
	pt.eventNameId[ EventType.BATTLE ]	= 2;
	pt.eventNameId[ EventType.SHOP ]	= 3;
	pt.eventNameId[ EventType.WARP ]	= 4;
	pt.eventIdName	=
	{
		0 : EventType.NONE,
		1 : EventType.TEXT,
		2 : EventType.BATTLE,
		3 : EventType.SHOP,
		4 : EventType.WARP
	};

	/**
	 * 
	 */
	pt.formToActor = function( app, actor )
	{
		var actorName	= app.getDomElement( this.ID_ACTOR_NAME );
		var eventType	= app.getDomElement( this.ID_EVENT_TYPE );
		var eventParam	= app.getDomElement( this.ID_EVENT_PARAM );
		if ( !actorName || !eventType || !eventParam )
		{
			return;
		}
		// set actor
		actor.name			= actorName.value;
		actor.event			= {};
		actor.event.type	= this.eventIdName[ eventType.selectedIndex ];
		switch ( actor.event.type )
		{
		case  EventType.TEXT :
			actor.event.text	= eventParam.value;
			break;
		case EventType.SHOP :
			actor.event.shopId	= eventParam.value;
			break;
		case EventType.WARP :
			actor.event.baseUrl	= eventParam.value;
			break;
		}
	};
	/**
	 * 
	 */
	pt.actorToForm = function( app, actor )
	{
		var actorName	= app.getDomElement( this.ID_ACTOR_NAME );
		var eventType	= app.getDomElement( this.ID_EVENT_TYPE );
		var eventParam	= app.getDomElement( this.ID_EVENT_PARAM );
		if ( !actorName || !eventType || !eventParam )
		{
			return;
		}
		// set form
		actorName.value			= actor.name;
		eventType.selectedIndex	= this.eventNameId[ actor.event.type ];
		eventParam.value		= "";
		switch ( actor.event.type )
		{
		case EventType.TEXT :
			eventParam.value	= actor.event.text;
			break;
		case EventType.SHOP :
			eventParam.value	= actor.event.shopId;
			break;
		case EventType.WARP :
			eventParam.value	= actor.event.baseUrl;
			break;
		}
	};
	/**
	 * 
	 */
	pt.clearForm = function( app )
	{
		var actorName	= app.getDomElement( this.ID_ACTOR_NAME );
		var eventType	= app.getDomElement( this.ID_EVENT_TYPE );
		var eventParam	= app.getDomElement( this.ID_EVENT_PARAM );
		if ( !actorName || !eventType || !eventParam )
		{
			return;
		}
		// clear
		actorName.value			= "";
		eventType.selectedIndex	= 0;
		eventParam.value		= "";
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var app			= scene.app;
		var stage		= scene.stageTask;
		var selectedVal	= stage.getSelectedVal();
		var id			= stage.getActorId(
			stage.scroll.x + stage.selected.x,
			stage.scroll.y + stage.selected.y
		);

		if ( this.currentValue != selectedVal )
		{
			// if old is actor
			if ( this.currentActor != null )
			{
				this.formToActor( app, this.currentActor );
			}
			// if new is actor
			if ( id >= 0 )
			{
				this.currentActor	= stage.actors[ id ];
				this.actorToForm( app, this.currentActor );
			}
			else
			{
				this.currentActor	= null;
				this.clearForm( app );
			}
			// set current
			this.currentValue = selectedVal;
		}
		return false;
	};
})( FormTask.prototype );


/**
 * MouseArea
 */
var MouseArea = function( target, rect )
{
	this.target	= target;
	this.rect	= rect;
};

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.escape	= 0;
	this.tbl.mouse	= 0;
	this.mouseInfo	=
	{
		target	: null,
		x		: 0,
		y		: 0,
		ox		: 0,
		oy		: 0
	};
	this.mouseAreas	= [];
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 27:	this.tbl.escape	= value; break;	// ESC
			}
		}
		else if ( type == "mousedown" )
		{
			if ( this.getMouseArea( event.layerX, event.layerY ) )
			{
				this.mouse = 1;
			}
			else
			{
				this.mouse = 0;
			}
		}
		else if ( type == "mousemove" )
		{
			if ( this.mouse )
			{
				if ( !this.getMouseArea( event.layerX, event.layerY ) )
				{
					this.mouse = 0;
				}
			}
		}
		else if ( type == "mouseup" || type == "mouseout" )
		{
			this.mouse = 0;
		}
	};
	/**
	 * 
	 */
	pt.getMouseArea = function( x, y )
	{
		for ( var i = this.mouseAreas.length - 1; i >= 0; i-- )
		{
			var area = this.mouseAreas[i];
			if ( area.rect.x <= x && x <= area.rect.x + area.rect.w &&
				 area.rect.y <= y && y <= area.rect.y + area.rect.h )
			{
				this.mouseInfo.target	= area.target;
				this.mouseInfo.x		= x;
				this.mouseInfo.y		= y;
				this.mouseInfo.ox		= x - area.rect.x;
				this.mouseInfo.oy		= y - area.rect.y;
//Debug.print( "x=" + x + ", y=" + y + ", ox=" + this.mouseInfo.ox + ", oy=" + this.mouseInfo.oy );
				return area;
			}
		}
		return null;
	};
	/**
	 * 
	 */
	pt.addMouseArea = function( area )
	{
		this.mouseAreas.push( area );
	};
	/**
	 * 
	 */
	pt.removeMouseArea = function( area )
	{
		for ( var i = 0; i < this.mouseAreas.length; i++ )
		{
			if ( this.mouseAreas[i] == area )
			{
				this.mouseAreas.splice( i, 1 );
				break;
			}
		}
	};
})( InputCommand.prototype );

/**
 * EditorScene
 */
var EditorScene = function( app, name )
{
	this.app		= app;
	this.name		= name;
	this.data		= null;
	this.stageName	= null;

	this.command	= new InputCommand();

	// create task
	this.panelTask	= new PanelTask( this );
	this.stageTask	= new StageTask( this );
	this.chipTask	= new ChipTask( this, ChipType.NORMAL );
	this.chipTrTask	= new ChipTask( this, ChipType.TRANSPARENT );
	this.actorTask	= new ActorTask( this );
	this.formTask	= new FormTask( this );
	// create list
	this.panelTask.append( this.stageTask );
	this.panelTask.append( this.chipTask );
	this.panelTask.append( this.chipTrTask );
	this.panelTask.append( this.actorTask );
	this.panelTask.append( this.formTask );
	// head of task list
	this.child		= this.panelTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager	= new MessageManager();
};
EditorScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
		var self	= this;
		var button;

		// data
		button	= this.app.getDomElement( Res.String.ID_DATA );
		if ( button )
		{
			this.app.addSysEventHandler( button, "click", function()
			{
				try
				{
					if ( self.app.sceneManager.current == self )
					{
						var str	= self.serializeData( self.data );
						var win	= window.open( "", "_blank" );
						win.document.write( "<pre>" + str + "</pre>" );
						win.document.close();
					}
				}
				catch ( e )
				{
					self.app.kill();
					Debug.alertError( e );
				}
			});
		}
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data	= data;

		// set parameters
		this.stageTask.setData( this, data.map, data.actors, data.resource );
		this.chipTask.setData( this, data.resource );
		this.chipTrTask.setData( this, data.resource );
		this.actorTask.setData( this, data.resource );
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.serializeData = function( data )
	{
		return "window.h5glib.loadDataCB(\n{\n" + this.serializeJson( data ) + "});";
	}
	/**
	 * 
	 */
	pt.serializeJson = function( data )
	{
		var stringify = function( obj )
		{
			var arr = [];
			arr.push("{ ");
			var i = 0;
			for ( var key in obj )
			{
				if ( i > 0 ) { arr.push(", "); }
				arr.push( pair( key, obj[ key ] )  );
				i++;
			}
			arr.push(" }");
			return arr.join("");
		};
		var pair = function( key, value )
		{
			// object
			if ( typeof( value ) == "object" )
			{
				return ( key + " : " + stringify( value ) );
			}
			// string
			else if ( typeof( value ) == "string" )
			{
				return ( key + " : \"" + value + "\"" );
			}
			// primitive ?
			else
			{
				return ( key + " : " + value );
			}
		};

		var arr = [];

		// map
		arr.push("\tmap :\n\t[");
		for ( var i = 0; i < data.map.length; i++ )
		{
			if ( i > 0 ) { arr.push(","); }

			arr.push("\n\t\t[");
			for ( var j = 0; j < data.map[i].length; j++ )
			{
				if ( j > 0 ) { arr.push(","); }

				var val = Math.floor( data.map[i][j] );
				arr.push( val );
			}
			arr.push("]");
		}
		arr.push("\n\t],\n");

		// actors
		arr.push("\tactors :\n\t[");
		for ( var i = 0; i < data.actors.length; i++ )
		{
			if ( i > 0 ) { arr.push(","); }

			var actor = data.actors[i];
			arr.push("\n\t\t{\n");

			arr.push("\n\t\t\t"		+ pair( "type",		actor.type ) );
			arr.push(",\n\t\t\t"	+ pair( "name",		actor.name ) );
			arr.push(",\n\t\t\t"	+ pair( "status",	actor.status ) );
			arr.push(",\n\t\t\t"	+ pair( "x", Math.floor( actor.x ) + 0.5 ) + ", " + pair( "y", Math.floor( actor.y ) + 0.5 ) +", " +
									  pair( "z", actor.z ) + ", dir : " + actor.dir );
			arr.push(",\n\t\t\t"	+ pair( "hp", actor.hp ) + ", " + pair( "mp", actor.mp ) );
			arr.push(",\n\t\t\t"	+ pair( "event", actor.event ) );

			arr.push("\n\t\t}");
		}
		arr.push("\n\t],\n");

		// item
		arr.push("\titem :\n\t{");
		// def
		arr.push("\n\t\tdef :\n\t\t{");
		var obj	= data.item.def;
		var i	= 0;
		for ( var key in obj )
		{
			if ( i > 0 ) { arr.push(","); }

			var item = obj[ key ];
			arr.push("\n\t\t\t" + key + " :\n\t\t\t{");

			arr.push("\n\t\t\t\t"	+ pair( "name",		item.name ) );
			arr.push(",\n\t\t\t\t"	+ pair( "msg",		item.msg ) );
			arr.push(",\n\t\t\t\t"	+ pair( "price",	item.price ) );
			arr.push(",\n\t\t\t\t"	+ pair( "desc",		item.desc ) );

			arr.push("\n\t\t\t}");
			i++;
		}
		arr.push("\n\t\t},");
		// def
		arr.push("\n\t\tbelongings :\n\t\t[");
		for ( var i = 0; i < data.item.belongings.length; i++ )
		{
			if ( i > 0 ) { arr.push(","); }
			arr.push("\n\t\t\t"	+ stringify( data.item.belongings[i] ) );
		}
		arr.push("\n\t\t],");
		// money
		arr.push("\n\t\t"	+ pair( "money",	data.item.money ) + "," );
		// shop
		arr.push("\n\t\tshop :\n\t\t{");
		var obj	= data.item.shop;
		var i	= 0;
		for ( var key in obj )
		{
			if ( i > 0 ) { arr.push(","); }

			var shop = obj[ key ];
			arr.push("\n\t\t\t" + key + " :\n\t\t\t{");

			arr.push("\n\t\t\t\titems :\n\t\t\t\t[");
			for ( var i = 0; i < shop.items.length; i++ )
			{
				if ( i > 0 ) { arr.push(","); }
				arr.push("\n\t\t\t\t\t"	+ stringify( shop.items[i] ) );
			}
			arr.push("\n\t\t\t\t]");

			arr.push("\n\t\t\t}");
			i++;
		}
		arr.push("\n\t\t}");

		arr.push("\n\t},\n");

		// resource
		arr.push("\tresource :\n\t{");
		var resarr =
		[
			{ name : "image", obj : data.resource.image },
			{ name : "sound", obj : data.resource.sound }
		];
		for ( var k = 0; k < resarr.length; k++ )
		{
			// image, sound
			if ( k > 0 ) { arr.push(","); }

			arr.push("\n\t\t" + resarr[k].name + " :\n\t\t{");
			var obj	= resarr[k].obj;
			var i	= 0;
			for ( var key in obj )
			{
				if ( i > 0 ) { arr.push(","); }

				var prop = obj[ key ];
				if ( prop instanceof Array )
				{
					arr.push("\n\t\t\t" + key + " :\n\t\t\t[");
					for ( var n = 0; n < prop.length; n++ )
					{
						if ( n > 0 ) { arr.push(","); }
						arr.push("\n\t\t\t\t{ " + pair( "name", prop[n].name ) + " }");
					}
					arr.push("\n\t\t\t]");
				}
				else
				{
					arr.push("\n\t\t\t" + key + " : { " + pair( "name", prop.name ) + " }");
				}
				i++;
			}
			arr.push("\n\t\t}");
		}
		arr.push("\n\t},\n");

		// animation
		arr.push("\tanimation :\n\t{");
		i = 0;
		for ( var key in data.animation )
		{
			if ( i > 0 ) { arr.push(","); }

			var prop = data.animation[ key ];
			arr.push("\n\t\t" + key + " :\n\t\t{");
			arr.push("\n\t\t\t" + pair( "frameCount", prop.frameCount ) + ", " + pair( "fw", prop.fw ) + ", " + pair( "fh", prop.fh ) + ",");
			arr.push("\n\t\t\t" + pair( "frames", prop.frames ) );
			arr.push("\n\t\t}");

			i++;
		}
		arr.push("\n\t}\n");

		return arr.join("");
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( EditorScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.EditorScene	= EditorScene;


// Anonymous function end
//
})( window );
