/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE	: "<p>" +
					  "曲の再生中はキオミクが顔芸します。" +
					  "</p>",
		ID_SONGS	: "songs",
		ID_PLAY		: "play",
		ID_PAUSE	: "pause"
	},
	Color :
	{
		BACKGROUND		: "rgb(0, 0, 0)"
	},
	Font :
	{
		LARGE		: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL		: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command	= scene.command;	// input
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;

		context.save();

		context.fillStyle	= Res.Color.BACKGROUND;
		context.fillRect( 0, 0, canvas.width, canvas.height );

		context.restore();
	};
})( StageTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	SLEEP	: 0,
	WAKEUP	: 1,
	NEUTRAL	: 2,
	SMILE	: 3,
	WINK_R	: 4,
	WINK_L	: 5,
	SURPRISE: 6
};

/**
 * ActorTask
 */
var ActorTask = function( scene )
{
	this.command		= new AICommand();
	this.animator		= null;
	this.trans			= null;
};
ActorTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.init = function( scene )
	{
		var image		= scene.data.resource.image;
		var animation	= scene.data.animation;
		this.interfaces	=
		[
			{ update : this.updateSleep,	animator : new ImageAnimator( image.actor.data,	animation.actorSleep ) },
			{ update : this.updateAction,	animator : new ImageAnimator( image.actor.data,	animation.actorWakeup ) },
			{ update : this.updateNeutral,	animator : new ImageAnimator( image.actor.data,	animation.actorNeutral ) },
			{ update : this.updateAction,	animator : new ImageAnimator( image.actor.data,	animation.actorSmile ) },
			{ update : this.updateAction,	animator : new ImageAnimator( image.actor.data,	animation.actorWinkR ) },
			{ update : this.updateAction,	animator : new ImageAnimator( image.actor.data,	animation.actorWinkL ) },
			{ update : this.updateAction,	animator : new ImageAnimator( image.actor.data,	animation.actorSurprise ) }
		];
		this.setStatus( ActorStatus.SLEEP );
		this.animator.start();
	};
	/**
	 * 
	 */
	pt.updateSleep = function( scene )
	{
		var upd = false;

		// if play
		if ( scene.command.tbl.play )
		{
			this.setStatus( ActorStatus.WAKEUP );
			this.animator.start();
			scene.command.tbl.play = 0;
			return true;
		}

		return upd;
	};
	/**
	 * 
	 */
	pt.updateNeutral = function( scene )
	{
		var upd = false;

		// if pause
		if ( scene.command.tbl.pause )
		{
			this.setStatus( ActorStatus.SLEEP );
			this.animator.start();
			scene.command.tbl.pause = 0;
			return true;
		}

		// update AI
		this.command.update();

		if ( this.command.tbl.smile )
		{
			this.setStatus( ActorStatus.SMILE );
			this.animator.start();
		}
		else if ( this.command.tbl.winkL )
		{
			this.setStatus( ActorStatus.WINK_L );
			this.animator.start();
		}
		else if ( this.command.tbl.winkR )
		{
			this.setStatus( ActorStatus.WINK_R );
			this.animator.start();
		}
		else if ( this.command.tbl.surprise )
		{
			this.setStatus( ActorStatus.SURPRISE );
			this.animator.start();
		}

		if ( this.status != ActorStatus.NEUTRAL )
		{
			return true;
		}

		// neutral animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			var r = Math.floor( Math.random() * 30 );
			if ( r == 1 )
			{
				this.animator.start();
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateAction = function( scene )
	{
		var upd = false;

		// if pause
		if ( scene.command.tbl.pause )
		{
			this.setStatus( ActorStatus.SLEEP );
			this.animator.start();
			scene.command.tbl.pause = 0;
			return true;
		}

		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			this.setStatus( ActorStatus.NEUTRAL );
			upd = true;
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;

		if ( this.trans == null )
		{
			this.trans		= { x : ( canvas.width / 2 ), y : ( canvas.height / 2 ) };
		}
		context.drawImage(
			this.animator.image,
			this.animator.fx,							// sx
			this.animator.fy,							// sy
			this.animator.fw,							// sw
			this.animator.fh,							// sh
			this.trans.x - ( this.animator.fw / 2 ),	// dx
			this.trans.y - ( this.animator.fh / 2 ),	// dy
			this.animator.fw,							// dw
			this.animator.fh							// dh
		);
	};
})( ActorTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "scene.actorTask.status=" + scene.actorTask.status;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );

/**
 * AICommand
 */
var AICommand = function()
{
	this.tbl.smile		= 0;
	this.tbl.winkR		= 0;
	this.tbl.winkL		= 0;
	this.tbl.surprise	= 0;
};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 20 );
		if      ( r == 0 )	{ this.tbl.smile	= 1; }
		else if ( r == 1 )	{ this.tbl.winkR	= 1; }
		else if ( r == 2 )	{ this.tbl.winkL	= 1; }
		else if ( r == 3 )	{ this.tbl.surprise	= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.play	= 0;
	this.tbl.pause	= 0;

	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 27: this.tbl.escape= value; break;	// ESC
				case 81: this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * KmpScene
 */
var KmpScene = function( app, name )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;
	this.command		= new InputCommand();

	// create task
	this.stageTask		= new StageTask( this );
	this.actorTask		= new ActorTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.stageTask.append( this.actorTask );
	this.stageTask.append( this.soundTask );
	this.stageTask.append( this.debugTask );
	// head of task list
	this.child		= this.stageTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager		= new MessageManager();
};
KmpScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
		var self	= this;
		var button;

		// play
		button	= this.app.getDomElement( Res.String.ID_PLAY );
		this.app.addSysEventHandler( button, "click", function()
		{
			try
			{
				var sound		= self.soundTask;
				var resource	= self.data.resource;
				var command		= self.command;

				var select		= self.app.getDomElement( Res.String.ID_SONGS );
				var play		= self.app.getDomElement( Res.String.ID_PLAY );
				var pause		= self.app.getDomElement( Res.String.ID_PAUSE );

				// play
				sound.prepareLoop( resource.sound.songs[ select.selectedIndex ].data )
				// set ability
				select.disabled	= true;
				play.disabled	= true;
				pause.disabled	= false;
				// set command
				command.tbl.play	= 1;
			}
			catch ( e )
			{
				self.app.kill();
				Debug.alertError( e );
			}
		});

		// pause
		button	= this.app.getDomElement( Res.String.ID_PAUSE );
		this.app.addSysEventHandler( button, "click", function()
		{
			try
			{
				var sound		= self.soundTask;
				var resource	= self.data.resource;
				var command		= self.command;

				var select		= self.app.getDomElement( Res.String.ID_SONGS );
				var play		= self.app.getDomElement( Res.String.ID_PLAY );
				var pause		= self.app.getDomElement( Res.String.ID_PAUSE );

				// pause
				sound.cancel( resource.sound.songs[ select.selectedIndex ].data )
				// set ability
				select.disabled	= false;
				play.disabled	= false;
				pause.disabled	= true;
				// set command
				command.tbl.pause	= 1;
			}
			catch ( e )
			{
				self.app.kill();
				Debug.alertError( e );
			}
		});
		button.disabled = true;
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};
	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data		= data;
		// init actor
		this.actorTask.init( this );
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( KmpScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.KmpScene	= KmpScene;

// Anonymous function end
//
})( window );
