//  background_copy.hpp: copy operations by another thread

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_IOSTREAMS_BACKGROUND_COPY_HPP
#define HAMIGAKI_IOSTREAMS_BACKGROUND_COPY_HPP

#include <boost/iostreams/detail/buffer.hpp>
#include <boost/iostreams/constants.hpp>
#include <boost/iostreams/positioning.hpp>
#include <boost/iostreams/read.hpp>
#include <boost/iostreams/traits.hpp> 
#include <boost/iostreams/write.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>
#include <boost/noncopyable.hpp>

namespace hamigaki { namespace iostreams {

namespace detail
{

class background_copy_base
{
public:
    virtual ~background_copy_base(){}

    bool done()
    {
        return do_done();
    }

    std::streampos tell()
    {
        return do_tell();
    }

    void stop()
    {
        return do_stop();
    }

private:
    virtual bool do_done() = 0;
    virtual std::streampos do_tell() = 0;
    virtual void do_stop() = 0;
};

template<typename Source, typename Sink>
class background_copy_impl
    : public background_copy_base
{
    typedef typename boost::iostreams::char_type_of<Source>::type char_type;
    typedef boost::iostreams::detail::buffer<char_type> buffer_type;

public:
    background_copy_impl(
            const Source& src, const Sink& sink, std::streamsize buffer_size)
        : src_(src), sink_(sink), buffer_size_(buffer_size)
        , done_(false), interrupted_(false)
    {
    }

    void run()
    {
        try
        {
            buffer_type buf(buffer_size_);

            total_ = 0;
            while (!interrupted())
            {
                buf.set(0, 0);
                std::streamsize read_size =
                    boost::iostreams::read(src_, buf.data(), buf.size());

                if (read_size == -1)
                    break;

                buf.set(0, read_size);
                while ((buf.ptr() != buf.eptr()) && !interrupted())
                {
                    std::streamsize amt =
                        boost::iostreams::write(
                            sink_, buf.ptr(), buf.eptr() - buf.ptr());

                    total_ += amt;
                    buf.ptr() += amt;
                }
            }
        }
        catch (...)
        {
        }

        boost::mutex::scoped_lock locking(mutex_);
        done_ = true;
    }

private:
    boost::mutex mutex_;
    Source src_;
    Sink sink_;
    std::streamsize buffer_size_;
    volatile std::streamsize total_;
    volatile bool done_;
    volatile bool interrupted_;

    bool do_done() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        return done_;
    }

    std::streampos do_tell() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        return boost::iostreams::offset_to_position(total_);
    }

    void do_stop() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        interrupted_ = true;
    }

    bool interrupted()
    {
        boost::mutex::scoped_lock locking(mutex_);
        return interrupted_;
    }
};

} // namespace detail

class background_copy : boost::noncopyable
{
public:
    template<typename Source, typename Sink>
    void start(Source src, Sink sink,
        std::streamsize buffer_size = 
            boost::iostreams::default_device_buffer_size)
    {
        typedef detail::background_copy_impl<Source,Sink> impl_type;
        impl_type* ptr = new impl_type(src, sink, buffer_size);
        pimpl_.reset(ptr);
        thread_ptr_.reset(
            new boost::thread(
                boost::bind(boost::mem_fn(&impl_type::run), ptr)));
    }

    ~background_copy()
    {
        stop();
    }

    bool done()
    {
        return pimpl_->done();
    }

    void stop()
    {
        if (pimpl_.get())
        {
            pimpl_->stop();
            thread_ptr_->join();

            thread_ptr_.reset();
            pimpl_.reset();
        }
    }

    std::streampos tell()
    {
        return pimpl_->tell();
    }

private:
    std::auto_ptr<detail::background_copy_base> pimpl_;
    std::auto_ptr<boost::thread> thread_ptr_;
};

} } // End namespaces iostreams, hamigaki.

#endif // HAMIGAKI_IOSTREAMS_BACKGROUND_COPY_HPP
