//  split.hpp: stream generator for block device

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_IOSTREAMS_SPLIT_HPP
#define HAMIGAKI_IOSTREAMS_SPLIT_HPP

#include <boost/iostreams/categories.hpp>
#include <boost/iostreams/detail/config/limits.hpp>
#include <boost/iostreams/detail/buffer.hpp>
#include <boost/iostreams/detail/ios.hpp>
#include <boost/iostreams/detail/template_params.hpp>
#include <boost/iostreams/positioning.hpp>
#include <boost/iostreams/read.hpp>
#include <boost/iostreams/traits.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/static_assert.hpp>

namespace hamigaki { namespace iostreams {

template<typename BlockSource>
class split
{
private:
    typedef boost::iostreams::stream_offset offset_type;

public:
    typedef typename boost::iostreams::
        char_type_of<BlockSource>::type char_type;

    typedef typename boost::iostreams::
        category_of<BlockSource>::type category;

    #define BOOST_PP_LOCAL_MACRO(n) \
        BOOST_IOSTREAMS_TEMPLATE_PARAMS(n, T) \
        explicit split( \
              int buffer_size BOOST_PP_COMMA_IF(n) \
              BOOST_PP_ENUM_BINARY_PARAMS(n, const T, &t) ) \
            : pimpl_(new impl(buffer_size BOOST_PP_COMMA_IF(n) \
                     BOOST_PP_ENUM_PARAMS(n, t))) \
            { } \
        /**/
    #define BOOST_PP_LOCAL_LIMITS (0, BOOST_IOSTREAMS_MAX_FORWARDING_ARITY)
    #include BOOST_PP_LOCAL_ITERATE()
    #undef BOOST_PP_LOCAL_MACRO

    std::streamsize read(char_type* s, std::streamsize n)
    {
        if (pimpl_->eof_)
            return -1;

        buffer_type& buf = pimpl_->buf_;
        char_type* next_s = s;
        char_type* end_s = s + n;

        while (true)
        {
            if (buf.ptr() != buf.eptr())
            {
                std::streamsize amt =
                    (std::min)(buf.eptr() - buf.ptr(), end_s - next_s);

                next_s = std::copy(buf.ptr(), buf.ptr() + amt, next_s);
                buf.ptr() += amt;
            }

            if (next_s == end_s)
                break;

            std::streamsize amt = fill();
            if (amt <= 0)
                break;
        }
        return boost::iostreams::detail::
            check_eof(static_cast<std::streamsize>(next_s - s));
    }

    void close()
    {
        pimpl_->close();
        pimpl_->eof_ = true;
    }

    std::streampos seek(offset_type off, BOOST_IOS::seekdir way)
    {
        buffer_type& buf = pimpl_->buf_;

        if (way == BOOST_IOS::cur)
        {
            std::streampos pos = pimpl_->seek(0, BOOST_IOS::cur);
            offset_type cur_off = boost::iostreams::position_to_offset(pos);
            cur_off += (buf.ptr() - buf.data());
            if (off == 0)
                return cur_off;

            off = cur_off + off;
            way = BOOST_IOS::beg;
        }

        // avoid ADL
        offset_type blocks = split::block_count(off);
        offset_type rems = split::block_remnants(off);

        std::streampos pos = pimpl_->seek(pimpl_->block_size()*blocks, way);
        buf.set(0, 0);
        if (rems != 0)
        {
            fill(); // TODO: error check
            buf.ptr() += rems;
        }
        return pos + rems;
    }

    BlockSource& source() { return *pimpl_; }
    const BlockSource& source() const { return *pimpl_; }

private:
    typedef boost::iostreams::detail::buffer<char_type> buffer_type;

    std::streamsize fill()
    {
        buffer_type& buf = pimpl_->buf_;
        std::size_t block_size = pimpl_->block_size();
        std::size_t size = (buf.size() / block_size) * block_size;
        if (size == 0)
        {
            size = block_size;
            buf.resize(size);
        }

        std::streamsize amt = pimpl_->read(buf.data(), size);
        if (amt == -1)
        {
            pimpl_->eof_ = true;
            return -1;
        }
        buf.set(0, amt);
        return amt;
    }

    offset_type block_count(offset_type off)
    {
        std::size_t block_size = pimpl_->block_size();
        if (off >= 0)
            return off / block_size;
        else
            return -((-off + block_size) / block_size);
    }

    offset_type block_remnants(offset_type off)
    {
        std::size_t block_size = pimpl_->block_size();
        if (off >= 0)
            return off % block_size;
        else
            return ((-off + block_size) % block_size);
    }

    struct impl : BlockSource
    {
    #define BOOST_PP_LOCAL_MACRO(n) \
        BOOST_IOSTREAMS_TEMPLATE_PARAMS(n, T) \
        impl( int buffer_size BOOST_PP_COMMA_IF(n) \
              BOOST_PP_ENUM_BINARY_PARAMS(n, const T, &t) ) \
            : BlockSource(BOOST_PP_ENUM_PARAMS(n, t)), \
              buf_(buffer_size), eof_(false) \
            { buf_.set(0, 0); } \
        /**/
    #define BOOST_PP_LOCAL_LIMITS (0, BOOST_IOSTREAMS_MAX_FORWARDING_ARITY)
    #include BOOST_PP_LOCAL_ITERATE()
    #undef BOOST_PP_LOCAL_MACRO

        buffer_type  buf_;
        bool         eof_;
    };

    boost::shared_ptr<impl> pimpl_;
};

} } // End namespaces iostreams, hamigaki.

#endif // HAMIGAKI_IOSTREAMS_SPLIT_HPP
