//  ogg_play.cpp: a simple ogg player

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <hamigaki/audio/pcm_device.hpp>
#include <hamigaki/audio/vorbis_file.hpp>
#include <hamigaki/iostreams/background_copy.hpp>
#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/copy.hpp>
#include <boost/iostreams/positioning.hpp>
#include <boost/iostreams/restrict.hpp>
#include <boost/progress.hpp>
#include <exception>
#include <iostream>

namespace audio = hamigaki::audio;
namespace io = boost::iostreams;

int main(int argc, char* argv[])
{
    try
    {
        if (argc != 2)
        {
            std::cerr
                << "Usage: ogg_play (input file)"
                << std::endl;

            return 1;
        }

        audio::vorbis_file<io::file_source> vf(
            (io::file_source(std::string(argv[1]), BOOST_IOS::binary)));

        const boost::int64_t total = vf.total();

        const audio::vorbis_info& info = vf.info();
        audio::pcm_format fmt;
        fmt.rate = info.rate;
        fmt.bits = 16;
        fmt.channels = info.channels;

        hamigaki::iostreams::background_copy bg_copy;
        bg_copy.start(
            vf,
            audio::pcm_sink(fmt, 1024));

        boost::progress_display progress(total);
        while (!bg_copy.done())
        {
            progress += static_cast<unsigned long>(
                io::position_to_offset(bg_copy.tell()) - progress.count());
        }

        // probably count < expected_count
        unsigned long rest =
            progress.expected_count() - progress.count();
        if (rest != 0)
            progress += rest;

        return 0;
    }
    catch (const std::exception& e)
    {
        std::cerr << "Error: " << e.what() << std::endl;
    }
    return 1;
}
