//  pcm_device.cpp: a simple pcm device

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <hamigaki/audio/pcm_device.hpp>
#include <boost/config.hpp>
#include <boost/iostreams/detail/ios.hpp>
#include <boost/noncopyable.hpp>
#include <stdexcept>

#if defined(BOOST_WINDOWS)
    #include <boost/ptr_container/ptr_vector.hpp>
    #include <vector>
    #include <windows.h>
    #include <mmsystem.h>
#else
    #include <sys/ioctl.h>
    #include <sys/soundcard.h>
    #include <fcntl.h>
    #include <unistd.h>
#endif

namespace hamigaki { namespace audio {

#if defined(BOOST_WINDOWS)
namespace
{

const std::size_t wave_buffer_count = 4;

class semaphore : boost::noncopyable
{
public:
    semaphore(long init_value, long max_value) :
        handle_(::CreateSemaphore(0, init_value, max_value, 0))
    {
    }

    ~semaphore()
    {
        ::CloseHandle(handle_);
    }

    long release(long n)
    {
        long prev;
        ::ReleaseSemaphore(handle_, n, &prev);
        return prev;
    }

    void wait()
    {
        ::WaitForSingleObject(handle_, INFINITE);
    }

private:
    ::HANDLE handle_;
};

class wave_buffer : boost::noncopyable
{
public:
    wave_buffer(::HWAVEOUT handle, std::size_t size) :
        handle_(handle), buffer_(size), pos_(0)
    {
        std::memset(&header_, 0, sizeof(header_));

        header_.lpData = &buffer_[0];
        header_.dwBufferLength  = size;
        header_.dwFlags = 0;

        ::MMRESULT res = ::waveOutPrepareHeader(handle_, &header_, sizeof(header_));
        if (res != MMSYSERR_NOERROR)
            throw BOOST_IOSTREAMS_FAILURE("cannot prepare wave buffer");
    }

    ~wave_buffer()
    {
        ::waveOutUnprepareHeader(handle_, &header_, sizeof(header_));
    }

    void flush()
    {
        if (pos_ != 0)
        {
            header_.dwBufferLength = pos_;
            pos_ = 0;
            ::waveOutWrite(handle_, &header_, sizeof(header_));
        }
    }

    bool write(const char* data, std::size_t size, std::size_t& written)
    {
        written = (std::min)(buffer_.size() - pos_, size);
        std::memcpy(&buffer_[pos_], data, written);
        pos_ += written;
        if (pos_ == buffer_.size())
        {
            pos_ = 0;
            header_.dwBufferLength = buffer_.size();
            ::MMRESULT res = ::waveOutWrite(handle_, &header_, sizeof(header_));
            if (res != MMSYSERR_NOERROR)
                throw BOOST_IOSTREAMS_FAILURE("wave device write error");
            return true;
        }
        return false;
    }

private:
    ::HWAVEOUT handle_;
    std::vector<char> buffer_;
    ::WAVEHDR header_;
    std::size_t pos_;
};

} // namespace

class pcm_sink::impl : boost::noncopyable
{
public:
    impl(const pcm_format& f, std::size_t buffer_size)
        : sema_(wave_buffer_count, wave_buffer_count)
        , pos_(0), wait_(true)
    {
        ::WAVEFORMATEX fmt;
        fmt.wFormatTag = WAVE_FORMAT_PCM;
        fmt.nChannels = f.channels;
        fmt.nSamplesPerSec = f.rate;
        fmt.nAvgBytesPerSec = f.rate * (f.channels * f.bits / 8);
        fmt.nBlockAlign = f.channels * f.bits / 8;
        fmt.wBitsPerSample = f.bits;
        fmt.cbSize = 0;

        ::MMRESULT res = ::waveOutOpen(
            &handle_, WAVE_MAPPER, &fmt,
            reinterpret_cast< ::DWORD_PTR>(&pcm_sink::impl::Callback),
            reinterpret_cast< ::DWORD_PTR>(&sema_),
            CALLBACK_FUNCTION);
        if (res != MMSYSERR_NOERROR)
            throw BOOST_IOSTREAMS_FAILURE("waveOutOpen failed");

        for (std::size_t i = 0; i < wave_buffer_count; ++i)
            buffers_.push_back(new wave_buffer(handle_, buffer_size));
    }

    ~impl()
    {
        close();
        ::waveOutReset(handle_);
        ::waveOutClose(handle_);
    }

    std::streamsize write(const char* s, std::streamsize n)
    {
        std::streamsize result = n;
        while (n != 0)
        {
            if (wait_)
            {
                sema_.wait();
                wait_ = false;
            }

            std::size_t written;
            if (buffers_[pos_].write(s, n, written))
            {
                wait_ = true;
                pos_ = (pos_ + 1) % buffers_.size();
                n -= written;
                s += written;
            }
            else
                break;
        }
        return result;
    }

    bool flush()
    {
        if (!wait_)
        {
            buffers_[pos_].flush();
            wait_ = true;
            pos_ = (pos_ + 1) % buffers_.size();
        }

        for (std::size_t i = 0; i < buffers_.size(); ++i)
            sema_.wait();
        sema_.release(buffers_.size());

        return true;
    }

    void close()
    {
        flush();
        ::waveOutReset(handle_);
    }

private:
    semaphore sema_;
    ::HWAVEOUT handle_;
    boost::ptr_vector<wave_buffer> buffers_;
    std::size_t pos_;
    bool wait_;

    static void CALLBACK Callback(::HWAVEOUT hwo, ::UINT uMsg,
        ::DWORD_PTR dwInstance, ::DWORD dwParam1, ::DWORD dwParam2)
    {
        semaphore& sema = *reinterpret_cast<semaphore*>(dwInstance);
        sema.release(1);
    }
};

#else // not defined(BOOST_WINDOWS)
namespace
{

class dev_dsp : boost::noncopyable
{
public:
    dev_dsp() :
        fd_(::open("/dev/dsp", O_WRONLY))
    {
        if (fd_ == -1)
            throw BOOST_IOSTREAMS_FAILURE("cannot open /dev/dsp");
    }

    ~dev_dsp()
    {
        this->close();
    }

    std::streamsize write(const char* s, std::streamsize n)
    {
        int res = ::write(fd_, s, n);
        if (res < 0)
            throw BOOST_IOSTREAMS_FAILURE("cannot write DSP");
        return res;
    }

    void close()
    {
        if (fd_ != -1)
        {
            ::close(fd_);
            fd_ = -1;
        }
    }

    bool flush()
    {
        return true;
    }

    void format(int value)
    {
        int tmp = value;
        if (::ioctl(fd_, SNDCTL_DSP_SETFMT, &tmp) == -1)
            throw BOOST_IOSTREAMS_FAILURE("cannot change DSP format");
        if (tmp != value)
            throw BOOST_IOSTREAMS_FAILURE("unsupported DSP format");
    }

    void channels(int value)
    {
        int tmp = value;
        if (::ioctl(fd_, SNDCTL_DSP_CHANNELS, &tmp) == -1)
            throw BOOST_IOSTREAMS_FAILURE("cannot change DSP channels");
        if (tmp != value)
            throw BOOST_IOSTREAMS_FAILURE("unsupported DSP channels");
    }

    void speed(int value)
    {
        int tmp = value;
        if (::ioctl(fd_, SNDCTL_DSP_SPEED, &tmp) == -1)
            throw BOOST_IOSTREAMS_FAILURE("cannot change DSP speed");
        if (tmp != value)
            throw BOOST_IOSTREAMS_FAILURE("unsupported DSP speed");
    }

private:
    int fd_;
};

} // namespace

class pcm_sink::impl : public dev_dsp
{
public:
    impl(const pcm_format& f, std::size_t buffer_size)
    {
        if (f.bits == 8)
            dev_dsp::format(AFMT_U8);
        else if (f.bits == 16)
            dev_dsp::format(AFMT_S16_LE);
        else
            throw BOOST_IOSTREAMS_FAILURE("unsupported DSP format");
        dev_dsp::channels(f.channels);
        dev_dsp::speed(f.rate);
    }
};
#endif // not defined(BOOST_WINDOWS)

pcm_sink::pcm_sink(const pcm_format& f, std::size_t buffer_size)
    : pimpl_(new impl(f, buffer_size))
{
}

std::streamsize pcm_sink::write(const char* s, std::streamsize n)
{
    return pimpl_->write(s, n);
}

bool pcm_sink::flush()
{
    return pimpl_->flush();
}

void pcm_sink::close()
{
    pimpl_->close();
}

} } // End namespaces audio, hamigaki.
