//  float.hpp: float codec

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_DETAIL_FLOAT_HPP
#define HAMIGAKI_AUDIO_DETAIL_FLOAT_HPP

#include <boost/cstdint.hpp>
#include <boost/integer.hpp>
#include <cmath>
#include <limits>

namespace hamigaki { namespace audio { namespace detail {

inline float decode_ieee754_float_impl(const char* src)
{
    int exp =
        (static_cast<int>(static_cast<unsigned char>(src[3]) & 0x7F) << 1) |
        (static_cast<int>(static_cast<unsigned char>(src[2]) & 0x80) >> 7) ;

    typedef boost::int_t<24>::fast frac_t;
    frac_t frac =
        (static_cast<frac_t>(static_cast<unsigned char>(src[2]) & 0x7F) <<16) |
        (static_cast<frac_t>(static_cast<unsigned char>(src[1]) & 0xFF) << 8) |
        (static_cast<frac_t>(static_cast<unsigned char>(src[0]) & 0xFF)     ) ;

    if ((exp == 0) && (frac == 0))
        return 0.0f;
    else if (exp == 255)
    {
        if (frac == 0)
            return std::numeric_limits<float>::infinity();
        else
            return std::numeric_limits<float>::quiet_NaN();
    }

    frac |= 0x800000;

    return std::ldexp(static_cast<float>(frac), exp-(127+23));
}

inline float decode_ieee754_float(const char* src)
{
    float tmp = decode_ieee754_float_impl(src);
    bool sign = (static_cast<unsigned char>(src[3]) & 0x80) != 0;
    return sign ? -tmp : tmp;
}

inline bool sign_bit(float f)
{
    if (f < 0.0f)
        return true;
    else if (f > 0.0f)
        return false;
    else
    {
        char buf[8];
        std::sprintf(buf, "%g", f);
        return buf[0] == '-';
    }
}

inline bool is_nan(float f)
{
    return f != f;
}

inline void encode_ieee754_float_impl(char* dst, float f)
{
    if (f == 0.0f)
    {
        std::memset(dst, 0, 4);
        return;
    }
    else if (f == std::numeric_limits<float>::infinity())
    {
        std::memcpy(dst, "\x00\x00\x80\x7F", 4);
        return;
    }
    else if (is_nan(f))
    {
        std::memcpy(dst, "\x00\x00\xC0\x7F", 4);
        return;
    }

    int exp;
    typedef boost::int_t<24>::fast frac_t;
    frac_t frac = static_cast<frac_t>(std::ldexp(std::frexp(f, &exp), 23+1));
    exp += (127-1);
    if (frac < 0.0f)
        frac = -frac;

    dst[3] = static_cast<char>(static_cast<unsigned char>((exp & 0xFE) >> 1));

    dst[2] = static_cast<char>(
        static_cast<unsigned char>((exp & 0x01) << 7) |
        static_cast<unsigned char>((frac & 0x7F0000) >> 16));

    dst[1] = static_cast<char>(
        static_cast<unsigned char>((frac & 0xFF00) >> 8));

    dst[0] = static_cast<char>(static_cast<unsigned char>((frac & 0xFF)));
}

inline char* encode_ieee754_float(char* dst, float f)
{
    encode_ieee754_float_impl(dst, std::abs(f));
    if (sign_bit(f))
        dst[3] |= 0x80;
    return dst;
}

} } } // End namespaces detail, audio, hamigaki.

#endif // HAMIGAKI_AUDIO_DETAIL_FLOAT_HPP
