//  wide_adaptor.hpp: an adaptor for making wide character stream

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_WIDE_ADAPTOR_HPP
#define HAMIGAKI_AUDIO_WIDE_ADAPTOR_HPP

#include <hamigaki/audio/detail/float.hpp>
#include <hamigaki/audio/sample_format.hpp>
#include <hamigaki/iostreams/traits.hpp>
#include <boost/iostreams/categories.hpp>
#include <boost/iostreams/operations.hpp>
#include <boost/iostreams/traits.hpp>
#include <boost/cstdint.hpp>
#include <vector>

namespace hamigaki { namespace audio {

namespace detail
{

template<sample_format_type Type>
struct cvt_int32;

template<> struct cvt_int32<uint8>
{
    static boost::int32_t decode(const char* s)
    {
        boost::int32_t tmp = static_cast<unsigned char>(*s);
        return (tmp - 128) * static_cast<boost::int32_t>(16777216);
    }

    static void encode(char* s, boost::int32_t n)
    {
        boost::uint8_t tmp = static_cast<boost::uint8_t>(
            n/static_cast<boost::int32_t>(16777216) + 128);
        *s = static_cast<char>(static_cast<unsigned char>(tmp));
    }
};

template<> struct cvt_int32<int_le16>
{
    static boost::int32_t decode(const char* s)
    {
        boost::uint16_t tmp = static_cast<unsigned char>(s[0]);
        tmp |= static_cast<boost::uint16_t>(
            static_cast<unsigned char>(s[1])) << 8;

        boost::int32_t val = (tmp & 0x8000)
            ? -static_cast<boost::int32_t>(~tmp) - 1
            : static_cast<boost::int32_t>(tmp);

        return val * static_cast<boost::int32_t>(65536);
    }

    static void encode(char* s, boost::int32_t n)
    {
        boost::int16_t val = static_cast<boost::int16_t>(
            n/static_cast<boost::int32_t>(65536));

        boost::uint32_t tmp = (val > 0)
            ? static_cast<boost::uint32_t>(val)
            : 0x10000 - static_cast<boost::uint32_t>(-val);

        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>(tmp & 0xFF));
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 8) & 0xFF));
    }
};

template<> struct cvt_int32<int_le24>
{
    static boost::int32_t decode(const char* s)
    {
        boost::uint32_t tmp = static_cast<unsigned char>(s[0]);
        tmp |= static_cast<boost::uint32_t>(
            static_cast<unsigned char>(s[1])) << 8;
        tmp |= static_cast<boost::uint32_t>(
            static_cast<unsigned char>(s[2])) << 16;

        boost::int32_t val = (tmp & 0x800000)
            ? -static_cast<boost::int32_t>(~tmp) - 1
            : static_cast<boost::int32_t>(tmp);

        return val * 256;
    }

    static void encode(char* s, boost::int32_t n)
    {
        boost::int32_t val = static_cast<boost::int32_t>(n/256);

        boost::uint32_t tmp = (val > 0)
            ? static_cast<boost::uint32_t>(val)
            : 0x1000000 - static_cast<boost::uint32_t>(-val);

        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>(tmp & 0xFF));
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 8) & 0xFF));
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 16) & 0xFF));
    }
};

template<> struct cvt_int32<int_le32>
{
    static boost::int32_t decode(const char* s)
    {
        boost::uint32_t tmp = static_cast<unsigned char>(s[0]);
        tmp |= static_cast<boost::uint32_t>(
            static_cast<unsigned char>(s[1])) << 8;
        tmp |= static_cast<boost::uint32_t>(
            static_cast<unsigned char>(s[2])) << 16;
        tmp |= static_cast<boost::uint32_t>(
            static_cast<unsigned char>(s[3])) << 24;

        boost::int32_t val = (tmp & 0x80000000)
            ? -static_cast<boost::int32_t>(~tmp) - 1
            : static_cast<boost::int32_t>(tmp);

        return val;
    }

    static void encode(char* s, boost::int32_t n)
    {
        boost::int32_t val = static_cast<boost::int32_t>(n/256);

        boost::uint32_t tmp = (val > 0)
            ? static_cast<boost::uint32_t>(val)
            : 0x1000000 - static_cast<boost::uint32_t>(-val);

        *(s++) = '\0';
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>(tmp & 0xFF));
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 8) & 0xFF));
        *(s++) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 16) & 0xFF));
    }
};

} // namespace detail

template<class CharT, class Device>
class wide_adaptor;

template<class Device>
class wide_adaptor<float, Device>
{
public:
    typedef float char_type;

    struct category :
        boost::iostreams::mode_of<Device>::type,
        boost::iostreams::device_tag,
        boost::iostreams::closable_tag,
        boost::iostreams::optimally_buffered_tag {};

    explicit wide_adaptor(const Device& dev)
        : dev_(dev)
        , buffer_(boost::iostreams::optimal_buffer_size(dev_))
        , type_(audio::sample_format_of(dev_))
    {
    }

    wide_adaptor(const Device& dev, std::streamsize buffer_size)
        : dev_(dev), buffer_(buffer_size)
        , type_(audio::sample_format_of(dev_))
    {
    }

    wide_adaptor(const Device& dev, sample_format_type type)
        : dev_(dev)
        , buffer_(boost::iostreams::optimal_buffer_size(dev_))
        , type_(type)
    {
    }

    wide_adaptor(const Device& dev,
            sample_format_type type, std::streamsize buffer_size)
        : dev_(dev), buffer_(buffer_size), type_(type)
    {
    }

    void close(BOOST_IOS::openmode which = BOOST_IOS::in | BOOST_IOS::out)
    { 
        boost::iostreams::close(dev_, which);
    }

    std::streamsize read(float* s, std::streamsize n)
    {
        std::streamsize total = 0;

        while (total != n)
        {
            if (type_ == uint8)
            {
                std::streamsize amt =
                    read_int<uint8>(s + total, n - total);
                if (amt == -1)
                    break;
                total += amt;
            }
            else if (type_ == int_le16)
            {
                std::streamsize amt =
                    read_int<int_le16>(s + total, n - total);
                if (amt == -1)
                    break;
                total += amt;
            }
            else if (type_ == int_le32)
            {
                std::streamsize amt = read_int<int_le32>(s + total, n - total);
                if (amt == -1)
                    break;
                total += amt;
            }
            else if (type_ == float_le32)
            {
                std::streamsize amt = read_float(s + total, n - total);
                if (amt == -1)
                    break;
                total += amt;
            }
            else
                throw BOOST_IOSTREAMS_FAILURE("unsupported format");
        }

        return (total != 0) ? total : -1;
    }

    std::streamsize write(const float* s, std::streamsize n)
    {
        std::streamsize total = 0;

        while (total != n)
        {
            if (type_ == uint8)
            {
                std::streamsize amt = write_int<uint8>(s + total, n - total);
                total += amt;
            }
            else if (type_ == int_le16)
            {
                std::streamsize amt =
                    write_int<int_le16>(s + total, n - total);
                total += amt;
            }
            else if (type_ == int_le32)
            {
                std::streamsize amt =
                    write_int<int_le32>(s + total, n - total);
                total += amt;
            }
            else if (type_ == float_le32)
            {
                std::streamsize amt = write_float(s + total, n - total);
                total += amt;
            }
            else
                throw BOOST_IOSTREAMS_FAILURE("unsupported format");
        }

        return (total != 0) ? total : -1;
    }

    std::streamsize optimal_buffer_size() const
    {
        return buffer_.size() / sample_size(type_);
    }

private:
    Device dev_;
    std::vector<char> buffer_;
    sample_format_type type_;

    template<sample_format_type Type>
    std::streamsize read_int(float* s, std::streamsize n)
    {
        const std::streamsize smp_sz = sample_size(Type);

        std::streamsize count =
            (std::min)(
                n,
                static_cast<std::streamsize>(buffer_.size())/smp_sz
            );

        std::streamsize amt =
            boost::iostreams::read(dev_, &buffer_[0], count*smp_sz);
        if (amt == -1)
            return -1;
        count = amt / smp_sz;

        for (std::streamsize i = 0, offset = 0;
            i < count; ++i, offset += smp_sz)
        {
            s[i] = static_cast<float>(
                detail::cvt_int32<Type>::decode(
                    &buffer_[offset]) / 256) / 8388608.f;
        }

        return count;
    }

    template<sample_format_type Type>
    std::streamsize write_int(const float* s, std::streamsize n)
    {
        const std::streamsize smp_sz = sample_size(Type);

        std::streamsize count =
            (std::min)(
                n,
                static_cast<std::streamsize>(buffer_.size())/smp_sz
            );

        for (std::streamsize i = 0, offset = 0;
            i < count; ++i, offset += smp_sz)
        {
            detail::cvt_int32<Type>::encode(&buffer_[offset],
                static_cast<boost::int32_t>(s[i]*8388608.f)*256);
        }

        boost::iostreams::write(dev_, &buffer_[0], count*smp_sz);

        return count;
    }

    std::streamsize read_float(float* s, std::streamsize n)
    {
        const std::streamsize smp_sz = 4;

        std::streamsize count =
            (std::min)(
                n,
                static_cast<std::streamsize>(buffer_.size())/smp_sz
            );

        std::streamsize amt =
            boost::iostreams::read(dev_, &buffer_[0], count*smp_sz);
        if (amt == -1)
            return -1;
        count = amt / smp_sz;

        for (std::streamsize i = 0, offset = 0;
            i < count; ++i, offset += smp_sz)
        {
            s[i] = detail::decode_ieee754_float(&buffer_[offset]);
        }

        return count;
    }

    std::streamsize write_float(const float* s, std::streamsize n)
    {
        const std::streamsize smp_sz = 4;

        std::streamsize count =
            (std::min)(
                n,
                static_cast<std::streamsize>(buffer_.size())/smp_sz
            );

        for (std::streamsize i = 0, offset = 0;
            i < count; ++i, offset += smp_sz)
        {
            detail::encode_ieee754_float(&buffer_[offset], s[i]);
        }

        boost::iostreams::write(dev_, &buffer_[0], count*smp_sz);

        return count;
    }
};

template<class CharT, class Device>
inline wide_adaptor<CharT, Device>
widen(const Device& dev)
{
    return wide_adaptor<CharT, Device>(dev);
}

template<class CharT, class Device>
inline wide_adaptor<CharT, Device>
widen(const Device& dev, std::streamsize buffer_size)
{
    return wide_adaptor<CharT, Device>(dev, buffer_size);
}

template<class CharT, class Device>
inline wide_adaptor<CharT, Device>
widen(const Device& dev, sample_format_type type)
{
    return wide_adaptor<CharT, Device>(dev, type);
}

template<class CharT, class Device>
inline wide_adaptor<CharT, Device>
widen(const Device& dev, sample_format_type type, std::streamsize buffer_size)
{
    return wide_adaptor<CharT, Device>(dev, type, buffer_size);
}

} } // End namespaces audio, hamigaki.

#endif // HAMIGAKI_AUDIO_WIDE_ADAPTOR_HPP
