//  multiplexer.hpp: stream multiplexer

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_IOSTREAMS_MULTIPLEXER_HPP
#define HAMIGAKI_IOSTREAMS_MULTIPLEXER_HPP

#include <boost/iostreams/categories.hpp>
#include <boost/iostreams/close.hpp>
#include <boost/iostreams/read.hpp>
#include <boost/iostreams/traits.hpp>
#include <boost/ptr_container/ptr_vector.hpp>
#include <boost/assert.hpp>
#include <boost/shared_ptr.hpp>
#include <algorithm>
#include <vector>

namespace hamigaki { namespace iostreams {

namespace detail
{

template<typename CharT>
class muxer_component_base
{
public:
    typedef CharT char_type;

    virtual ~muxer_component_base() {}
    virtual std::streamsize read(CharT* s, std::streamsize n) = 0;
    virtual void close() = 0;
};

template<class Source>
class muxer_component
    : public muxer_component_base<
        typename boost::iostreams::char_type_of<Source>::type
    >
{
public:
    muxer_component(const Source& src, std::streamsize elm_size)
        : src_(src), element_size_(elm_size)
    {
    }

    std::streamsize read(char_type* s, std::streamsize n) // virtual
    {
        if (n < element_size_)
            return -1;
        return boost::iostreams::read(src_, s, element_size_);
    }

    void close() // virtual
    {
        boost::iostreams::close(src_, BOOST_IOS::in);
    }

    const Source* component() const
    {
        return &src_;
    }

private:
    Source src_;
    std::streamsize element_size_;
};

template<typename CharT>
class multiplexer_impl
{
private:
    struct closer
    {
        void operator()(detail::muxer_component_base<CharT>& src) const
        {
            src.close();
        }
    };

public:
    multiplexer_impl() : block_size_(0)
    {
    }

    template<typename T>
    const T* component(int n) const
    {
        typedef detail::muxer_component<T> wrapper;

        if (const wrapper* ptr = dynamic_cast<const wrapper*>(&sources_[n]))
            return ptr->component();
        else
            return 0;
    }

    template<typename Source>
    void push(const Source& src, std::streamsize elm_size)
    {
        sources_.push_back(new detail::muxer_component<Source>(src, elm_size));
        block_size_ += elm_size;
    }

    std::streamsize read(CharT* s, std::streamsize n)
    {
        if (sources_.empty())
            return -1;

        std::streamsize total = 0;
        while (n != 0)
        {
            std::streamsize amt = read_once(s, n);
            if (amt < 0)
                break;
            s += amt;
            n -= amt;
            total += amt;
        }

        return (total != 0) ? total : -1;
    }

    void close()
    {
        std::for_each(sources_.begin(), sources_.end(), closer());
    }

    std::streamsize block_size() const
    {
        return block_size_;
    }

private:
    boost::ptr_vector<detail::muxer_component_base<CharT> > sources_;
    std::streamsize block_size_;

    std::streamsize read_once(CharT* s, std::streamsize n)
    {
        std::streamsize total = 0;
        for (std::size_t i = 0; i < sources_.size(); ++i)
        {
            std::streamsize amt = sources_[i].read(s, n);
            if (amt < 0)
                return -1;
            s += amt;
            n -= amt;
            total += amt;
        }
        return total;
    }
};

} // namespace detail

template<typename CharT>
class basic_multiplexer
{
    typedef detail::multiplexer_impl<CharT> impl_type;

public:
    typedef CharT char_type;

    struct category :
        boost::iostreams::source_tag,
        boost::iostreams::closable_tag {};

    basic_multiplexer() : pimpl_(new impl_type)
    {
    }

    template<typename T>
    const T* component(int n) const
    {
        return pimpl_->component<T>(n);
    }

    template<typename Source>
    void push(const Source& src, std::streamsize elm_size)
    {
        pimpl_->push(src, elm_size);
    }

    template<typename Source>
    void push(const Source& src)
    {
        pimpl_->push(src, src.block_size());
    }

    std::streamsize read(CharT* s, std::streamsize n)
    {
        return pimpl_->read(s, n);
    }

    void close()
    {
        pimpl_->close();
    }

    std::streamsize block_size() const
    {
        return pimpl_->block_size();
    }

private:
    boost::shared_ptr<impl_type> pimpl_;
};

} } // End namespaces iostreams, hamigaki.

#endif // HAMIGAKI_IOSTREAMS_MULTIPLEXER_HPP
