/* $Id: file.c,v 1.2 2002/12/17 08:09:51 tkubo Exp $ */

/* file.c - FILE I/O module.
 *
 * Copyright (C) 2002 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "xhardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>
#include <unistd.h>
#include "xhardmeter.h"

#define MAGIC_STRING "hardmeter data\n"
#define PROGRAM_MAGIC 'P'
#define DEFAULT_FLAG_MAGIC 'D'
#define EXPERIMENT_MAGIC 'E'
#define EVENT_MAGIC 'V'
#define FLAG_MAGIC 'F'

static char buf1[128];
static char buf2[128];

#define CHECK_MAGIC(fp, magic) do { \
   int c = fgetc(fp); \
   if (c != magic) { \
     ungetc(c, fp); \
     return 0; \
   } \
   fgetc(fp); /* discard return code */ \
} while(0)

#define GET_STRING(buf, ptr, fp) do { \
  int i; \
  if (fgets(buf, sizeof(buf), fp) == NULL) \
     return -1; \
  i = strlen(buf) - 1; \
  if (buf[i] == '\n') \
    buf[i] = '\0'; \
  ptr = buf; \
} while (0)
#define GET_STRING1(ptr, fp) GET_STRING(buf1, ptr, fp)
#define GET_STRING2(ptr, fp) GET_STRING(buf2, ptr, fp)

#define GET_FLAG_VAL(ptr, fp) do { \
  if (fgets(buf2 /* danger! */, sizeof(buf2), fp) == NULL) \
     return -1; \
  sscanf(buf2, "%llx", &(ptr)); \
} while (0)

/* members of hardmeter_source_t */
static int source_start(void **top_datap, void *userdata);
static int source_end(void *top_data);
static int source_program(void *top_data, char **name, char **path);
static int source_default_flag(void *top_data, char **name, guint64 *val);
static int source_experiment_start(void **experiment_datap, void *top_data, char **name);
static int source_experiment_end(void *experiment_data);
static int source_event_start(void **event_datap, void *experiment_data, char **name, hardmeter_event_t **base, hardmeter_ebs_t *ebs);
static int source_event_end(void *event_data);
static int source_flag(void *event_data, char **name, guint64 *val);
static int source_error_cleanup(void *top_data);

static hardmeter_source_t source = {
  source_start,
  source_end,
  source_program,
  source_default_flag,
  source_experiment_start,
  source_experiment_end,
  source_event_start,
  source_event_end,
  source_flag,
  source_error_cleanup
};
hardmeter_source_t *hardmeter_file_source = &source;

/* members of hardmeter_builder_t */
static int builder_start(void **top_datap, void *userdata);
static int builder_end(void *top_data);
static int builder_program(void *top_data, const char *name, const char *path);
static int builder_default_flag(void *top_data, const char *name, guint64 val);
static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name);
static int builder_experiment_end(void *experiment_data);
static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs);
static int builder_event_end(void *event_data);
static int builder_flag(void *event_data, const char *name, guint64 val);
static int builder_error_cleanup(void *top_data);

static hardmeter_builder_t builder = {
  builder_start,
  builder_end,
  builder_program,
  builder_default_flag,
  builder_experiment_start,
  builder_experiment_end,
  builder_event_start,
  builder_event_end,
  builder_flag,
  builder_error_cleanup
};
hardmeter_builder_t *hardmeter_file_builder = &builder;


static int source_start(void **top_datap, void *userdata)
{
  FILE *fp = fopen(userdata, "r");
  char buf[128];

  if (fp == NULL)
    return -1;
  if (fgets(buf, sizeof(buf), fp) == NULL) {
    fclose(fp);
    return -1;
  }
  if (strcmp(buf, MAGIC_STRING) != 0) {
    fclose(fp);
    return -1;
  }
  *top_datap = fp;
  return 0;
}

static int source_end(void *top_data)
{
  FILE *fp = top_data;
  fclose(fp);
  return 1;
}

static int source_program(void *top_data, char **name, char **path)
{
  FILE *fp = top_data;

  CHECK_MAGIC(fp, PROGRAM_MAGIC);
  GET_STRING1(*name, fp);
  GET_STRING2(*path, fp);
  return 1;
}
static int source_default_flag(void *top_data, char **name, guint64 *val)
{
  FILE *fp = top_data;

  CHECK_MAGIC(fp, DEFAULT_FLAG_MAGIC);
  GET_STRING1(*name, fp);
  GET_FLAG_VAL(*val, fp);
  return 1;
}
static int source_experiment_start(void **experiment_datap, void *top_data, char **name)
{
  FILE *fp = top_data;

  CHECK_MAGIC(fp, EXPERIMENT_MAGIC);
  GET_STRING1(*name, fp);
  *experiment_datap = fp;
  return 1;
}
static int source_experiment_end(void *experiment_data)
{
  return 1;
}
static int source_event_start(void **event_datap, void *experiment_data, char **name, hardmeter_event_t **base, hardmeter_ebs_t *ebs)
{
  FILE *fp = experiment_data;
  char *basename;
  hardmeter_event_t *ev;
  char buf[128];
  int enable, type, sample, buffer, max;
  guint64 interval;

  CHECK_MAGIC(fp, EVENT_MAGIC);
  GET_STRING1(*name, fp);
  GET_STRING2(basename, fp);

  *base = NULL;
  for (ev = hardmeter_event_non_retirement; ev->name != NULL; ev++) {
    if (strcmp(ev->name, basename) == 0) {
      *base = ev;
      break;
    }
  }
  if (*base == NULL) {
    for (ev = hardmeter_event_at_retirement; ev->name != NULL; ev++) {
      if (strcmp(ev->name, basename) == 0) {
	*base = ev;
	break;
      }
    }
  }
  if (*base == NULL) {
    for (ev = hardmeter_event_tagging; ev->name != NULL; ev++) {
      if (strcmp(ev->name, basename) == 0) {
	*base = ev;
	break;
      }
    }
  }
  if (*base == NULL)
    return 0;

  fgets(buf, sizeof(buf), fp);
  sscanf(buf, "%d:%d:%d:%lld:%d:%d", &enable, &type, &sample, &interval, &buffer, &max);
  ebs->enabled = enable;
  ebs->type = type;
  ebs->sample = sample;
  ebs->interval = interval;
  ebs->buffer = buffer;
  ebs->max = max;
  *event_datap = fp;
  return 1;
}
static int source_event_end(void *event_data)
{
  return 1;
}
static int source_flag(void *event_data, char **name, guint64 *val)
{
  FILE *fp = event_data;

  CHECK_MAGIC(fp, FLAG_MAGIC);
  GET_STRING1(*name, fp);
  GET_FLAG_VAL(*val, fp);
  return 1;
}
static int source_error_cleanup(void *top_data)
{
  return 1;
}

/* members of hardmeter_builder_t */
static int builder_start(void **top_datap, void *userdata)
{
  FILE *fp = fopen(userdata, "w");
  if (fp == NULL)
    return -1;
  *top_datap = fp;
  fputs(MAGIC_STRING, fp);
  return 0;
}

static int builder_end(void *top_data)
{
  FILE *fp = top_data;
  fclose(fp);
  return 0;
}
static int builder_program(void *top_data, const char *name, const char *path)
{
  FILE *fp = top_data;
  fprintf(fp, "%c\n%s\n%s\n", PROGRAM_MAGIC, name, path);
  return 0;
}
static int builder_default_flag(void *top_data, const char *name, guint64 val)
{
  FILE *fp = top_data;
  fprintf(fp, "%c\n%s\n%llx\n", DEFAULT_FLAG_MAGIC, name, val);
  return 0;
}
static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name)
{
  FILE *fp = top_data;
  fprintf(fp, "%c\n%s\n", EXPERIMENT_MAGIC, name);
  *experiment_datap = fp;
  return 0;
}
static int builder_experiment_end(void *experiment_data)
{
  return 0;
}

static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs)
{
  FILE *fp = experiment_data;
  fprintf(fp, "%c\n%s\n%s\n%d:%d:%d:%lld:%d:%d\n", EVENT_MAGIC, name, base->name, ebs->enabled, ebs->type, ebs->sample, ebs->interval, ebs->buffer, ebs->max);
  *event_datap = fp;
  return 0;
}
static int builder_event_end(void *event_data)
{
  return 0;
}
static int builder_flag(void *event_data, const char *name, guint64 val)
{
  FILE *fp = event_data;
  fprintf(fp, "%c\n%s\n%llx\n", FLAG_MAGIC, name, val);
  return 0;
}
static int builder_error_cleanup(void *top_data)
{
  return 0;
}
