/* $Id: gtk.c,v 1.4 2002/12/17 08:09:51 tkubo Exp $ */

/* gtk.c - A part of Gtk module.
 *
 * Copyright (C) 2002 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "xhardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>
#include <unistd.h>
#include "xhardmeter.h"

static char *static_char = NULL; /* ugly hack */

static gint delete_event_cb(GtkWidget *widget, GdkEventAny *event, xhardmeter_window_t *w);
static void menu_load_cb(gpointer data, guint action, GtkWidget *widget);
static void menu_save_cb(gpointer data, guint action, GtkWidget *widget);
static void menu_save_as_cb(gpointer data, guint action, GtkWidget *widget);
static void menu_export_cb(gpointer data, guint action, GtkWidget *widget);
static void menu_quit_cb(gpointer data, guint action, GtkWidget *widget);

/* members of hardmeter_source_t */
static int source_start(void **top_datap, void *userdata);
static int source_end(void *top_data);
static int source_program(void *top_data, char **name, char **path);
static int source_default_flag(void *top_data, char **name, guint64 *val);
static int source_experiment_start(void **experiment_datap, void *top_data, char **name);
static int source_experiment_end(void *experiment_data);
static int source_event_start(void **event_datap, void *experiment_data, char **name, hardmeter_event_t **base, hardmeter_ebs_t *ebs);
static int source_event_end(void *event_data);
static int source_flag(void *event_data, char **name, guint64 *val);
static int source_error_cleanup(void *top_data);

static hardmeter_source_t source = {
  source_start,
  source_end,
  source_program,
  source_default_flag,
  source_experiment_start,
  source_experiment_end,
  source_event_start,
  source_event_end,
  source_flag,
  source_error_cleanup
};
hardmeter_source_t *hardmeter_gtk_p4_source = &source;

/* members of hardmeter_builder_t */
static int builder_start(void **top_datap, void *userdata);
static int builder_end(void *top_data);
static int builder_program(void *top_data, const char *name, const char *path);
static int builder_default_flag(void *top_data, const char *name, guint64 val);
static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name);
static int builder_experiment_end(void *experiment_data);
static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs);
static int builder_event_end(void *event_data);
static int builder_flag(void *event_data, const char *name, guint64 val);
static int builder_error_cleanup(void *top_data);

static hardmeter_builder_t builder = {
  builder_start,
  builder_end,
  builder_program,
  builder_default_flag,
  builder_experiment_start,
  builder_experiment_end,
  builder_event_start,
  builder_event_end,
  builder_flag,
  builder_error_cleanup
};
hardmeter_builder_t *hardmeter_gtk_p4_builder = &builder;

void xhardmeter_free_top(xhardmeter_top_t *top)
{
  GSList *l;
  xhardmeter_flag_t *flag;

  gtk_clist_clear(top->parent->clist);
  for (l = top->parent->default_xflags; l != NULL; l = l->next) {
    flag = l->data;
    if (gtk_toggle_button_get_active(flag->button)) {
      gtk_toggle_button_set_active(flag->button, FALSE);
      flag->cb->clear(flag->widget);
    }
  }
  xhardmeter_set_menu_sensitive(top->parent, TRUE);
  while (top->xexperiments) {
    gtk_notebook_remove_page(top->parent->notebook, 1);
  }
  if (top->filename)
    g_free(top->filename);
  g_free(top);
}

void xhardmeter_free_experiment(xhardmeter_experiment_t *exp)
{
  while (exp->xevents) {
    xhardmeter_free_event(exp->xevents->data);
  }
  exp->parent->xexperiments = g_slist_remove(exp->parent->xexperiments, exp);
  g_free(exp);
}

void xhardmeter_free_event(xhardmeter_event_t *ev)
{
  ev->parent->xevents = g_slist_remove(ev->parent->xevents, ev);
  g_free(ev->ebs);
  gtk_object_unref(GTK_OBJECT(ev->widget));
  g_free(ev);
}

static int source_start(void **top_datap, void *userdata)
{
  xhardmeter_window_t *w = userdata;
  xhardmeter_top_t *top = w->xtop;
  top->work = GINT_TO_POINTER(0);
  *top_datap = w->xtop;
  return 0;
}

static int source_end(void *top_data)
{
  if (static_char)
    g_free(static_char);
  static_char = NULL;
  return 0;
}

static int source_program(void *top_data, char **name, char **path)
{
  xhardmeter_top_t *top = top_data;
  gint i = GPOINTER_TO_INT(top->work);
  if (!gtk_clist_get_text(top->parent->clist, i, 0, name)) {
    /* set data for source_default_flag. */
    top->work = top->parent->default_xflags;
    return 0;
  }
  if (!gtk_clist_get_text(top->parent->clist, i, 1, path)) {
    /* set data for source_default_flag. */
    top->work = top->parent->default_xflags;
    return 0;
  }
  top->work = GINT_TO_POINTER(i + 1);
  return 1;
}

static int source_default_flag(void *top_data, char **name, guint64 *val)
{
  xhardmeter_top_t *top = top_data;
  xhardmeter_flag_t *flag;
  GSList *l = top->work;
  for (l = top->work; l != NULL; l = l->next) {
    flag = l->data;
    if (gtk_toggle_button_get_active(flag->button)) {
      *name = (char *)flag->event_property->name;
      flag->cb->get(flag->widget, val);
      top->work = l->next;
      return 1;
    }
  }
  /* set data for source_experiment_start. */
  top->work = top->xexperiments;
  return 0;
}
static int source_experiment_start(void **experiment_datap, void *top_data, char **name)
{
  xhardmeter_top_t *top = top_data;
  xhardmeter_experiment_t *exp;
  GSList *l = top->work;
  if (l != NULL) {
    exp = l->data;
    /* set data for source_event_start. */
    top->work = exp->xevents;
    *experiment_datap = l;
    *name = exp->label->label;
    return 1;
  }
  return 0;
}
static int source_experiment_end(void *experiment_data)
{
  GSList *l = experiment_data;
  xhardmeter_experiment_t *exp = l->data;
  exp->parent->work = l->next;
  return 0;
}
static int source_event_start(void **event_datap, void *experiment_data, char **name, hardmeter_event_t **base, hardmeter_ebs_t *ebs)
{
  xhardmeter_experiment_t *exp = ((GSList *)experiment_data)->data;
  xhardmeter_event_t *ev;
  GSList *l = exp->parent->work;

  if (l != NULL) {
    ev = l->data;
    /* set data for source_flag. */
    exp->parent->work = ev->xflags;
    *event_datap = l;
    if (static_char)
      g_free(static_char);
    gtk_object_get(GTK_OBJECT(ev->button), "label", &static_char, NULL);
    *name = static_char;
    *base = ev->event;
    if (ev->event->tag_base != NULL) {
      xhardmeter_get_ebs(ev->ebs, ebs);
    } else {
      memset(ebs, 0, sizeof(hardmeter_ebs_t));
    }
    return 1;
  }
  return 0;
}

static int source_event_end(void *event_data)
{
  GSList *l = event_data;
  xhardmeter_event_t *ev = l->data;
  ev->parent->parent->work = l->next;
  return 0;
}

static int source_flag(void *event_data, char **name, guint64 *val)
{
  xhardmeter_event_t *ev = ((GSList *)event_data)->data;
  GSList *l;
  xhardmeter_flag_t *flag;

  for (l = ev->parent->parent->work; l != NULL; l = l->next) {
    flag = l->data;
    if (gtk_toggle_button_get_active(flag->button)) {
      *name = (char *)flag->event_property->name;
      flag->cb->get(flag->widget, val);
      ev->parent->parent->work = l->next;
      return 1;
    }
  }
  return 0;
}

static int source_error_cleanup(void *top_data)
{
  return 0;
}

static int builder_start(void **top_datap, void *userdata)
{
  xhardmeter_window_t *w = userdata;
  xhardmeter_top_t *top;

  if (w->xtop != NULL)
    xhardmeter_free_top(w->xtop);
  top = g_new0(xhardmeter_top_t, 1);
  top->parent = w;
  w->xtop = top;
  *top_datap = top;
  return 0;
}

static int builder_end(void *top_data)
{
  return 0;
}

static int builder_program(void *top_data, const char *name, const char *path)
{
  xhardmeter_top_t *top = top_data;
  gchar *text[2];
  text[0] = (gchar *)name;
  text[1] = (gchar *)path;
  gtk_clist_append(top->parent->clist, text);
  return 0;
}

static int builder_default_flag(void *top_data, const char *name, guint64 val)
{
  xhardmeter_top_t *top = top_data;
  xhardmeter_flag_t *flag;
  GSList *l;

  for (l = top->parent->default_xflags; l != NULL; l = l->next) {
    flag = l->data;
    if (strcmp(flag->event_property->name, name) == 0) {
      gtk_toggle_button_set_active(flag->button, TRUE);
      flag->cb->set(flag->widget, val);
      return 0;
    }
  }
  /* TODO: print 'no such flag' */
  return 1;
}

static int builder_experiment_start(void **experiment_datap, void *top_data, const char *name)
{
  *experiment_datap = xhardmeter_experiment_new(top_data, name);
  return 0;
}

static int builder_experiment_end(void *experiment_data)
{
  return 0;
}

static int builder_event_start(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs)
{
  xhardmeter_event_t *ev;
  ev = xhardmeter_event_new(experiment_data, name, base);
  if (ev->ebs)
    xhardmeter_set_ebs(ev->ebs, ebs);
  *event_datap = ev;
  return 0;
}

static int builder_event_end(void *event_data)
{
  return 0;
}

static int builder_flag(void *event_data, const char *name, guint64 val)
{
  xhardmeter_event_t *ev = event_data;
  xhardmeter_flag_t *flag;
  GSList *l;

  for (l = ev->xflags; l != NULL; l = l->next) {
    flag = l->data;
    if (strcmp(flag->event_property->name, name) == 0) {
      gtk_toggle_button_set_active(flag->button, TRUE);
      flag->cb->set(flag->widget, val);
      return 0;
    }
  }
  /* TODO: print 'no such flag' */
  return 1;
}

static int builder_error_cleanup(void *top_data)
{
  return 0;
}

static gint delete_event_cb(GtkWidget *widget, GdkEventAny *event, xhardmeter_window_t *w)
{
  xhardmeter_top_t *top = w->xtop;
  if (top != NULL && top->is_modified) {
    if (!xhardmeter_ask(w->window, "Data is modified. Quit anyway?"))
      return TRUE;
  }
  gtk_main_quit();
  return TRUE;
}

static void menu_load_cb(gpointer data, guint action, GtkWidget *widget)
{
  xhardmeter_window_t *w = (xhardmeter_window_t *)data;
  const char *filename = xhardmeter_xfileselection(w->window);

  if (filename == NULL)
    return;
  hardmeter_build(hardmeter_file_source, (void *)filename,
		  hardmeter_gtk_p4_builder, w);
  w->xtop->filename = g_strdup(filename);
  return;
}

static void menu_save_cb(gpointer data, guint action, GtkWidget *widget)
{
  xhardmeter_window_t *w = (xhardmeter_window_t *)data;
  const char *filename = w->xtop->filename;
  struct stat sbuf;

  if (filename == NULL) {
    filename = xhardmeter_xfileselection(w->window);
    if (filename == NULL)
      return;
    if (stat(filename, &sbuf) == 0) {
      if (!xhardmeter_ask(w->window, "Overwrite?"))
	return;
    }
    w->xtop->filename = g_strdup(filename);
  }
  hardmeter_build(hardmeter_gtk_p4_source, w,
		  hardmeter_file_builder, (void *)filename);
  return;
}

static void menu_save_as_cb(gpointer data, guint action, GtkWidget *widget)
{
  xhardmeter_window_t *w = (xhardmeter_window_t *)data;
  const char *filename = xhardmeter_xfileselection(w->window);
  struct stat sbuf;

  if (filename == NULL)
    return;
  if (stat(filename, &sbuf) == 0) {
    if (!xhardmeter_ask(w->window, "Overwrite?"))
      return;
  }
  hardmeter_build(hardmeter_gtk_p4_source, w,
		  hardmeter_file_builder, (void *)filename);
  if (w->xtop->filename)
    g_free(w->xtop->filename);
  w->xtop->filename = g_strdup(filename);
  return;
}

static void menu_quit_cb(gpointer data, guint action, GtkWidget *widget)
{
  xhardmeter_window_t *w = (xhardmeter_window_t *)data;
  xhardmeter_top_t *top = w->xtop;
  if (top != NULL && top->is_modified) {
    if (!xhardmeter_ask(w->window, "Data is modified. Quit anyway?"))
      return;
  }
  gtk_main_quit();
}

static void menu_export_cb(gpointer data, guint action, GtkWidget *widget)
{
  xhardmeter_window_t *w = (xhardmeter_window_t *)data;
  const char *filename = xhardmeter_xfileselection(w->window);
  struct stat sbuf;

  if (filename == NULL)
    return;
  if (stat(filename, &sbuf) == 0) {
    if (!xhardmeter_ask(w->window, "Overwrite?"))
      return;
  }
  hardmeter_build(hardmeter_gtk_p4_source, w,
		  hardmeter_abyss_output_builder, (void *)filename);
  return;
}

static GtkItemFactoryEntry menu_items[] = {
  { "/_File",         NULL,         NULL, 0, "<Branch>" },
  { "/File/_Load",    "<control>L", menu_load_cb, 0, NULL },
  { "/File/_Save",    "<control>S", menu_save_cb, 0, NULL },
  { "/File/Save _As", NULL,         menu_save_as_cb, 0, NULL },
  { "/File/_Export",  NULL,         menu_export_cb, 0, NULL },
  { "/File/sep1",     NULL,         NULL, 0, "<Separator>" },
  { "/File/Quit",     "<control>Q", menu_quit_cb, 0, NULL },
  { "/_Program",      NULL,         NULL, 0, "<Branch>" },
  { "/Program/_Add",  NULL,         xhardmeter_program_add_cb, 0, NULL },
  { "/Program/_Edit", NULL,         xhardmeter_program_edit_cb, 0, NULL },
  { "/Program/_Delete",  NULL,      xhardmeter_program_delete_cb, 0, NULL },
  { "/_Experiment",   NULL,         NULL, 0, "<Branch>" },
  { "/Experiment/_Add", NULL,       xhardmeter_experiment_add_cb, 0, NULL },
  { "/Experiment/_Rename", NULL,    xhardmeter_experiment_rename_cb, 1, NULL },
  { "/Experiment/_Delete", NULL,    xhardmeter_experiment_delete_cb, 2, NULL },
  { "/E_vent",        NULL,         NULL, 0, "<Branch>" },
  { "/Event/_Add", NULL,            NULL, 3, NULL },
  { "/Event/_Rename", NULL,         xhardmeter_event_rename_cb, 4, NULL },
  { "/Event/_Delete", NULL,         xhardmeter_event_delete_cb, 5, NULL },
};

static void xmenu_set_submenu(GtkMenu *menu, const char *label, hardmeter_event_t *event, xhardmeter_window_t *w)
{
  GtkWidget *menuitem;
  GtkWidget *submenu;
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new();
  menuitem = gtk_menu_item_new_with_label(label);
  gtk_menu_append(menu, menuitem);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
  for (; event->name != NULL; event++) {
    menuitem = gtk_menu_item_new_with_label(event->name);
    gtk_object_set_data(GTK_OBJECT(menuitem), "event", event);
    gtk_signal_connect(GTK_OBJECT(menuitem), "activate", xhardmeter_event_add_cb, w);
    gtk_tooltips_set_tip(tooltips, menuitem, event->comment, NULL);
    gtk_menu_append(GTK_MENU(submenu), menuitem);
  }
}

xhardmeter_window_t *xhardmeter_window_new(void)
{
  xhardmeter_window_t *w = g_new0(xhardmeter_window_t, 1);
  GtkWidget *vbox;
  GtkWidget *menubar;
  GtkWidget *menuitem;
  GtkWidget *submenu;
  GtkWidget *vpaned;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

  /* create top level window. */
  w->window = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
  gtk_window_set_default_size(w->window, 700, 400);
  gtk_signal_connect(GTK_OBJECT(w->window), "delete_event",
		     GTK_SIGNAL_FUNC(delete_event_cb), w);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(w->window), vbox);

  /* add menubar. */
  accel_group = gtk_accel_group_new ();
  w->item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", 
				      accel_group);
  gtk_item_factory_create_items(w->item_factory, nmenu_items, menu_items, w);
  gtk_window_add_accel_group(GTK_WINDOW(w->window), accel_group);
  menubar = gtk_item_factory_get_widget(w->item_factory, "<main>");
  gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 2);

  menuitem = gtk_item_factory_get_item_by_action(w->item_factory, 3);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
  xmenu_set_submenu(GTK_MENU(submenu), "Non retirement", hardmeter_event_non_retirement, w);
  xmenu_set_submenu(GTK_MENU(submenu), "At retirement", hardmeter_event_at_retirement, w);
  xmenu_set_submenu(GTK_MENU(submenu), "Tagging", hardmeter_event_tagging, w);

  /* create frame */
  vpaned = gtk_vpaned_new();
  gtk_box_pack_start(GTK_BOX(vbox), vpaned, TRUE, TRUE, 0);
  gtk_paned_add1(GTK_PANED(vpaned), xhardmeter_program_new(w));
  gtk_paned_add2(GTK_PANED(vpaned), xhardmeter_experiment_frame_new(w));
  gtk_widget_show_all(GTK_WIDGET(w->window));

  xhardmeter_set_menu_sensitive(w, TRUE);
  return w;
}

void xhardmeter_set_menu_sensitive(xhardmeter_window_t *w, gboolean is_default_menu)
{
  GtkWidget *widget;
  int i;
  for (i = 1; i <= 5; i++) {
    widget = gtk_item_factory_get_widget_by_action(w->item_factory, i);
    gtk_widget_set_sensitive(widget, !is_default_menu);
  }
}

