/* $Id: xexperiment.c,v 1.6 2002/12/17 08:09:51 tkubo Exp $ */

/* xexperiment.c - A part of Gtk module.
 *
 * Copyright (C) 2002 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "xhardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <gtk/gtk.h>
#include "xhardmeter.h"

struct xioq_data {
  GtkToggleButton *button[5];
  GtkLabel *label;
};

struct menu_options {
  const char *label;
  int value;
};

static void notebook_switch_page_cb(GtkNotebook *notebook,
				    GtkNotebookPage *page,
				    guint page_num,
				    xhardmeter_window_t *w);
static void event_button_set_active(xhardmeter_event_t *ev);

/* properties */
static GSList *xflags_create(hardmeter_event_t *event, GtkWidget **widget, xhardmeter_ebs_t **ebs_widget);
static GtkWidget *xflags_create_option_menu(struct menu_options *options, GSList **group);
static xhardmeter_ebs_t *xflags_setup_ebs(GtkTable *table);
static GSList *xflags_setup_one_event(GtkTable *table, hardmeter_event_t *event);
static void xproperties_propname_toggled_cb(GtkToggleButton *button, GtkWidget *widget);
static void xproperties_setclr_toggled_cb(GtkToggleButton *button);
static void xproperties_ioq_toggled_cb(GtkToggleButton *button);

static GtkWidget *flag_bool_new(void)
{
  GtkWidget *widget = gtk_check_button_new_with_label("set");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
  gtk_signal_connect(GTK_OBJECT(widget), "toggled", xproperties_setclr_toggled_cb, NULL);
  return widget;
}

static void flag_bool_set(GtkWidget *w, guint64 val)
{
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val ? TRUE : FALSE);
}

static void flag_bool_get(GtkWidget *w, guint64 *val)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))) {
    *val = 1;
  } else {
    *val = 0;
  }
}

static void flag_bool_clear(GtkWidget *w)
{
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
}

static xhardmeter_flag_cb_t flag_bool_cb = {
  flag_bool_new,
  flag_bool_set,
  flag_bool_get,
  flag_bool_clear
};

static GtkWidget *flag_threshold_new(void)
{
  GtkObject *adj = gtk_adjustment_new(0.0, 0.0, 15.0, 1.0, 1.0, 0.0);
  GtkWidget *widget = gtk_hscale_new(GTK_ADJUSTMENT(adj));

  gtk_scale_set_digits(GTK_SCALE(widget), 0);
  gtk_scale_set_value_pos(GTK_SCALE(widget), GTK_POS_RIGHT);
  return widget;
}

static void flag_threshold_set(GtkWidget *w, guint64 val)
{
  GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
  gtk_adjustment_set_value(adj, (gfloat)val);
}

static void flag_threshold_get(GtkWidget *w, guint64 *val)
{
  GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
  *val = (guint64)(adj->value);
}

static void flag_threshold_clear(GtkWidget *w)
{
  GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
  gtk_adjustment_set_value(adj, 0.0);
}

static xhardmeter_flag_cb_t flag_threshold_cb = {
  flag_threshold_new,
  flag_threshold_set,
  flag_threshold_get,
  flag_threshold_clear
};

struct counter_data {
  guint64 val;
  GtkWidget *base2;
  GtkWidget *base16;
};

static void flag_counter_set_real(struct counter_data *data, guint64 val)
{
  char buf[45];
  int i, j;

  if ((val >> 40) != ((guint64)0) && (val >> 40) != ((guint64)0xFFFFFF)) {
    gtk_label_set_text(GTK_LABEL(data->base2), "overflow");
    gtk_label_set_text(GTK_LABEL(data->base16), "overflow");
    return;
  }

  for (i = j = 0; i < 40; i++) {
    if ((i != 0) && (i % 8 == 0))
      buf[j++] = ' ';
    if ((((guint64)1) << (39 - i)) & val)
      buf[j++] = '1';
    else
      buf[j++] = '0';
  }
  buf[j] = '\0';
  gtk_label_set_text(GTK_LABEL(data->base2), buf);
  sprintf(buf, "%010llx", val & 0xFFFFFFFFFF);
  gtk_label_set_text(GTK_LABEL(data->base16), buf);
  data->val = val;
}

static void flag_counter_change_cb(GtkEntry *entry, struct counter_data *data)
{
  gint64 val;

  val = strtoll(gtk_entry_get_text(GTK_ENTRY(entry)), NULL, 0);
  flag_counter_set_real(data, (guint64)val);
}

static GtkWidget *flag_counter_new(void)
{
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  struct counter_data *data = g_new0(struct counter_data, 1);
  GtkWidget *entry = gtk_entry_new();
  data->val = 0;
  data->base2 = gtk_label_new("00000000 00000000 00000000 00000000 00000000");
  data->base16 = gtk_label_new("0000000000");

  gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), data->base2, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), data->base16, FALSE, FALSE, 0);

  gtk_signal_connect(GTK_OBJECT(entry), "changed", flag_counter_change_cb, data);

  gtk_object_set_data_full(GTK_OBJECT(vbox), "counter_data", data, g_free);
  return vbox;
}

static void flag_counter_set(GtkWidget *w, guint64 val)
{
  struct counter_data *data = gtk_object_get_data(GTK_OBJECT(w), "counter_data");
  flag_counter_set_real(data, val);
}

static void flag_counter_get(GtkWidget *w, guint64 *val)
{
  struct counter_data *data = gtk_object_get_data(GTK_OBJECT(w), "counter_data");

  *val = data->val;
}

static void flag_counter_clear(GtkWidget *w)
{
  flag_counter_set(w, 0);
}

static xhardmeter_flag_cb_t flag_counter_cb = {
  flag_counter_new,
  flag_counter_set,
  flag_counter_get,
  flag_counter_clear
};

static GtkWidget *flag_ioq_new(void)
{
  GtkWidget *widget = gtk_hbox_new(FALSE, 0);
  GtkWidget *checkbutton;
  struct xioq_data *ioqdata = g_new(struct xioq_data, 1);
  int i;

  gtk_object_set_data_full(GTK_OBJECT(widget), "xioq_data", ioqdata, (GtkDestroyNotify)g_free);
  for (i = 0; i < 5; i++) {
    checkbutton = gtk_check_button_new();
    ioqdata->button[i] = GTK_TOGGLE_BUTTON(checkbutton);
    gtk_box_pack_start(GTK_BOX(widget), checkbutton, FALSE, FALSE, 0);
    gtk_object_set_data(GTK_OBJECT(checkbutton), "xioq_data", ioqdata);
    gtk_signal_connect(GTK_OBJECT(checkbutton), "toggled", xproperties_ioq_toggled_cb, NULL);
  }
  ioqdata->label = GTK_LABEL(gtk_label_new("00000"));
  gtk_object_set_data(GTK_OBJECT(ioqdata->label), "xioq_data", ioqdata);
  gtk_box_pack_start(GTK_BOX(widget), GTK_WIDGET(ioqdata->label), FALSE, FALSE, 0);

  return widget;
}

static void flag_ioq_set(GtkWidget *w, guint64 val)
{
  struct xioq_data *ioqdata = gtk_object_get_data(GTK_OBJECT(w), "xioq_data");
  int i;
  for (i = 4; i >= 0; i--) {
    gtk_toggle_button_set_active(ioqdata->button[i], val & (1 << i) ? TRUE : FALSE);
  }
}

static void flag_ioq_get(GtkWidget *w, guint64 *val)
{
  struct xioq_data *ioqdata = gtk_object_get_data(GTK_OBJECT(w), "xioq_data");
  int i;
  *val = 0;
  for (i = 4; i >= 0; i--) {
    if (gtk_toggle_button_get_active(ioqdata->button[i]))
      *val += 1 << i;
  }
}

static void flag_ioq_clear(GtkWidget *w)
{
  struct xioq_data *ioqdata = gtk_object_get_data(GTK_OBJECT(w), "xioq_data");
  int i;
  for (i = 4; i >= 0; i--) {
    gtk_toggle_button_set_active(ioqdata->button[i], FALSE);
  }
}

static xhardmeter_flag_cb_t flag_ioq_cb = {
  flag_ioq_new,
  flag_ioq_set,
  flag_ioq_get,
  flag_ioq_clear
};

static void notebook_switch_page_cb(GtkNotebook *notebook,
				    GtkNotebookPage *page,
				    guint page_num,
				    xhardmeter_window_t *w)
{
  xhardmeter_set_menu_sensitive(w, page_num == 0);
  if (w->xtop)
    w->xtop->current_xexperiment = gtk_object_get_data(GTK_OBJECT(page->child), "experiment");
}

static void event_button_set_active(xhardmeter_event_t *ev)
{
  xhardmeter_event_t *old_ev = ev->parent->current_xevent;
  GtkContainer *c = ev->parent->property_frame;
  GSList *l;

  if (!gtk_toggle_button_get_active(ev->button))
    return;
  for (l = ev->parent->xevents; l != NULL; l = l->next) {
    xhardmeter_event_t *e = l->data;
    if (ev != e && gtk_toggle_button_get_active(e->button))
      gtk_toggle_button_set_active(e->button, FALSE);
  }
  if (old_ev)
    gtk_container_remove(c, GTK_WIDGET(old_ev->widget));
  gtk_container_add(c, ev->widget);
  ev->parent->current_xevent = ev;
}

/*
 * implement properties
 */
static GSList *xflags_create(hardmeter_event_t *event, GtkWidget **widget, xhardmeter_ebs_t **ebs_widget)
{
  GtkWidget *scrolledwindow;
  GtkWidget *packer;
  GtkTable *table;
  GSList *list = NULL;
  int i;

  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_placement(GTK_SCROLLED_WINDOW(scrolledwindow),
				    GTK_CORNER_TOP_RIGHT);

  packer = gtk_packer_new();
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwindow), packer);

  table = (GtkTable *)gtk_table_new(0, 3, FALSE);
  gtk_packer_add_defaults(GTK_PACKER(packer), GTK_WIDGET(table), GTK_SIDE_TOP, GTK_ANCHOR_NE, 0);

  if (ebs_widget)
    *ebs_widget = NULL;
  if (event != NULL) {
    if (event->tag_base != NULL) {
      *ebs_widget = xflags_setup_ebs(table);
      list = g_slist_concat(list, xflags_setup_one_event(table, event->tag_base));
    }
    if (event->properties != NULL) {
      list = g_slist_concat(list, xflags_setup_one_event(table, event));
    }
  }
  for (i = 0; hardmeter_registers[i].name != NULL; i++) {
    list = g_slist_concat(list, xflags_setup_one_event(table, &hardmeter_registers[i]));
  }
  *widget = scrolledwindow;
  return list;
}

static GtkWidget *xflags_create_option_menu(struct menu_options *options, GSList **group)
{
  GtkWidget *option_menu = gtk_option_menu_new();
  GtkWidget *menu = gtk_menu_new();
  GtkWidget *menuitem;
  *group = NULL;
  while (options->label != NULL) {
    menuitem = gtk_radio_menu_item_new_with_label(*group, options->label);
    gtk_object_set_data(GTK_OBJECT(menuitem), "value", GINT_TO_POINTER(options->value));
    *group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menuitem));
    gtk_menu_append(GTK_MENU(menu), menuitem);
    options++;
  }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
  return option_menu;
}

static xhardmeter_ebs_t *xflags_setup_ebs(GtkTable *table)
{
  xhardmeter_ebs_t *ebs = g_new0(xhardmeter_ebs_t, 1);
  int row_position;
  GSList *group;
  struct menu_options type_options[] = {
    {"Precise", 1},
    {"Imprecise", 0},
    {NULL, 0}
  };
  struct menu_options sample_options[] = {
    {"EIP only", 0},
    {"All", 1},
    {NULL, 0}
  };
  GtkWidget *w;

  /* count the number of properties and resize table. */
  row_position = table->nrows;
  gtk_table_resize(table, row_position + 6, table->ncols);
  gtk_table_attach_defaults(table, gtk_hseparator_new(), 0, table->ncols, row_position, row_position + 1);
  row_position ++;

  gtk_table_attach_defaults(table, gtk_label_new("EBS"),
			    0, 1, row_position, row_position + 1);

  gtk_table_attach_defaults(table, gtk_label_new("TYPE"),
			    1, 2, row_position, row_position + 1);
  w = xflags_create_option_menu(type_options, &group);
  gtk_table_attach_defaults(table, w,
			    2, 3, row_position, row_position + 1);
  ebs->type = group;

  gtk_table_attach_defaults(table, gtk_label_new("Interval"),
			    1, 2, row_position + 1, row_position + 2);
  w = gtk_entry_new_with_max_length(32);
  gtk_table_attach_defaults(table, w,
			    2, 3, row_position + 1, row_position + 2);
  ebs->interval = GTK_ENTRY(w);

  gtk_table_attach_defaults(table, gtk_label_new("Buffer"),
			    1, 2, row_position + 2, row_position + 3);
  w = gtk_entry_new_with_max_length(32);
  gtk_table_attach_defaults(table, w,
			    2, 3, row_position + 2, row_position + 3);
  ebs->buffer = GTK_ENTRY(w);

  gtk_table_attach_defaults(table, gtk_label_new("Sample"),
			    1, 2, row_position + 3, row_position + 4);
  w = xflags_create_option_menu(sample_options, &group);
  gtk_table_attach_defaults(table, w,
			    2, 3, row_position + 3, row_position + 4);
  ebs->sample = group;

  gtk_table_attach_defaults(table, gtk_label_new("Max"),
			    1, 2, row_position + 4, row_position + 5);
  w = gtk_entry_new_with_max_length(32);
  gtk_table_attach_defaults(table, w,
			    2, 3, row_position + 4, row_position + 5);
  ebs->max = GTK_ENTRY(w);

  return ebs;
}

static GSList *xflags_setup_one_event(GtkTable *table, hardmeter_event_t *event)
{
  static GtkTooltips *tooltips = NULL;
  xhardmeter_flag_t *flag;
  GSList *list = NULL;
  GtkWidget *label;
  GtkWidget *togglebutton;
  GtkWidget *hbox;
  hardmeter_event_property_t *prop;
  int row_position;
  int i;

  if (tooltips == NULL)
    tooltips = gtk_tooltips_new();

  /* count the number of properties and resize table. */
  for (prop = event->properties, i = 0; prop->name != NULL; prop++, i++);
  row_position = table->nrows;
  gtk_table_resize(table, row_position + i + 1, table->ncols);
  gtk_table_attach_defaults(table, gtk_hseparator_new(), 0, table->ncols, row_position, row_position + 1);
  row_position ++;

  /* set event */
  label = xhardmeter_label_with_tooltips(event->name, event->comment);

  gtk_table_attach(table, label, 0, 1, row_position, row_position + 1, 0, 0, 5, 0);
  for (prop = event->properties, i = 0; prop->name != NULL; prop++, i++) {
    flag = g_new0(xhardmeter_flag_t, 1);
    flag->event_property = prop;
    switch (prop->type) {
    case PROP_BOOL:
      flag->cb = &flag_bool_cb;
      break;
    case PROP_THRESHOLD:
      flag->cb = &flag_threshold_cb;
      break;
    case PROP_COUNTER:
      flag->cb = &flag_counter_cb;
      break;
    case PROP_IOQ:
      flag->cb = &flag_ioq_cb;
      break;
    default:
      fprintf(stderr, "%s:%d: invalid type %d\n", __FILE__, __LINE__, prop->type);
      exit(1);
    }
    togglebutton = gtk_toggle_button_new_with_label(prop->name);
    flag->button = GTK_TOGGLE_BUTTON(togglebutton);
    flag->widget = flag->cb->new();
    list = g_slist_append(list, flag);

    if (prop->comment != NULL)
      gtk_tooltips_set_tip(tooltips, togglebutton, prop->comment, NULL);
    gtk_table_attach_defaults(table, togglebutton, 1, 2, row_position + i, row_position + i + 1);
    gtk_widget_set_sensitive(flag->widget, FALSE);
    gtk_signal_connect(GTK_OBJECT(togglebutton), "toggled", xproperties_propname_toggled_cb, flag->widget);
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), flag->widget, FALSE, FALSE, 0);
    gtk_table_attach(table, hbox, 2, 3, row_position + i, row_position + i + 1, GTK_FILL | GTK_EXPAND, 0, 5, 0);
  }
  return list;
}

static void xproperties_propname_toggled_cb(GtkToggleButton *button, GtkWidget *widget)
{
  gint active = gtk_toggle_button_get_active(button);
  gtk_widget_set_sensitive(widget, active);
}

static void xproperties_setclr_toggled_cb(GtkToggleButton *button)
{
  gint active = gtk_toggle_button_get_active(button);
  gtk_object_set(GTK_OBJECT(button), "label", active ? "set" : "clr", NULL);
}

static void xproperties_ioq_toggled_cb(GtkToggleButton *button)
{
  struct xioq_data *ioqdata = gtk_object_get_data(GTK_OBJECT(button), "xioq_data");
  char buf[6];
  int i;
  for (i = 0; i < 5; i++) {
    if (gtk_toggle_button_get_active(ioqdata->button[i])) {
      buf[i] = '1';
    } else {
      buf[i] = '0';
    }
  }
  buf[5] = '\0';
  gtk_label_set_text(ioqdata->label, buf);
}

/*
 * exported functions
 */
GtkWidget *xhardmeter_experiment_frame_new(xhardmeter_window_t *w)
{
  GtkWidget *frame;
  GtkWidget *notebook;
  GtkWidget *flags;

  frame = gtk_frame_new("Experiments");

  notebook = gtk_notebook_new();
  w->notebook = GTK_NOTEBOOK(notebook);
  gtk_signal_connect(GTK_OBJECT(notebook), "switch_page", notebook_switch_page_cb, w);
  gtk_container_add(GTK_CONTAINER(frame), notebook);

  w->default_xflags = xflags_create(NULL, &flags, NULL);
  gtk_notebook_append_page(w->notebook, flags, gtk_label_new("default"));

  return frame;
}

xhardmeter_experiment_t *xhardmeter_experiment_new(xhardmeter_top_t *t, const char *name)
{
  GtkWidget *tab_label;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *vbox;
  xhardmeter_experiment_t *exp;

  tab_label = gtk_label_new(name);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_notebook_append_page(t->parent->notebook, hbox, tab_label);

  frame = gtk_frame_new("Events");
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, TRUE, 0);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  frame = gtk_frame_new("Properties");
  gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);

  exp = g_new0(xhardmeter_experiment_t, 1);
  exp->parent = t;
  exp->label = GTK_LABEL(tab_label);
  exp->event_box = GTK_BOX(vbox);
  exp->property_frame = GTK_CONTAINER(frame);
  t->xexperiments = g_slist_append(t->xexperiments, exp);
  gtk_object_set_data_full(GTK_OBJECT(hbox), "experiment", exp,
			   (GtkDestroyNotify)xhardmeter_free_experiment);

  gtk_widget_show_all(hbox);
  gtk_widget_show(tab_label);
  return exp;
}

xhardmeter_event_t *xhardmeter_event_new(xhardmeter_experiment_t *exp, const char *name, hardmeter_event_t *event)
{
  static GtkTooltips *tooltips = NULL;
  xhardmeter_event_t *ev;
  GtkWidget *widget;
  xhardmeter_ebs_t *ebs;

  if (tooltips == NULL)
    tooltips = gtk_tooltips_new();

  ev = g_new0(xhardmeter_event_t, 1);
  ev->parent = exp;
  ev->event = event;
  ev->xflags = xflags_create(event, &widget, &ebs);
  ev->ebs = ebs;
  ev->button = GTK_TOGGLE_BUTTON(gtk_toggle_button_new_with_label(name));
  ev->widget = widget;
  exp->xevents = g_slist_append(exp->xevents, ev);
  gtk_object_ref(GTK_OBJECT(widget));
  gtk_object_sink(GTK_OBJECT(widget));

  gtk_signal_connect_object(GTK_OBJECT(ev->button), "clicked",
			    GTK_SIGNAL_FUNC(event_button_set_active), (void *)ev);
  gtk_object_weakref(GTK_OBJECT(ev->button), (GtkDestroyNotify)xhardmeter_free_event, ev);
  gtk_box_pack_start(exp->event_box, GTK_WIDGET(ev->button), FALSE, TRUE, 0);

  gtk_toggle_button_set_active(ev->button, TRUE);
  gtk_widget_show_all(GTK_WIDGET(ev->button));
  gtk_widget_show_all(GTK_WIDGET(ev->widget));
  return ev;
}

void xhardmeter_experiment_add_cb(xhardmeter_window_t *w)
{
  char buf[32];
  static int num = 1;

  sprintf(buf, "exp%d", num++);
  xhardmeter_experiment_new(w->xtop, buf);
}

void xhardmeter_experiment_rename_cb(xhardmeter_window_t *w)
{
  GtkNotebookPage *page = w->notebook->cur_page;
  gchar *old = NULL;
  gchar *new = NULL;

  if (page == NULL)
    return;

  gtk_label_get(GTK_LABEL(page->tab_label), &old);
  new = xhardmeter_text_dialog(w->window, "Rename", old);
  if (new != NULL) {
    gtk_label_set_text(GTK_LABEL(page->tab_label), new);
    g_free(new);
    w->xtop->is_modified = TRUE;
  }
}

void xhardmeter_experiment_delete_cb(xhardmeter_window_t *w)
{
  gint num = gtk_notebook_get_current_page(w->notebook);
  gtk_notebook_remove_page(w->notebook, num);
}

void xhardmeter_event_add_cb(GtkWidget *menuitem, xhardmeter_window_t *w)
{
  hardmeter_event_t *event = gtk_object_get_data(GTK_OBJECT(menuitem), "event");
  char buf[128];
  static int num = 1;

  snprintf(buf, sizeof buf, "%s_%d", event->name, num++);

  xhardmeter_event_new(w->xtop->current_xexperiment, buf, event);
}

void xhardmeter_event_rename_cb(xhardmeter_window_t *w)
{
  xhardmeter_event_t *ev = w->xtop->current_xexperiment->current_xevent;
  gchar *old;
  gchar *new;

  if (ev == NULL)
    return;
  gtk_object_get(GTK_OBJECT(ev->button), "label", &old, NULL);
  new = xhardmeter_text_dialog(w->window, "Rename", old);
  if (new != NULL) {
    gtk_object_set(GTK_OBJECT(ev->button), "label", new);
    g_free(new);
  }
  g_free(old);
}

void xhardmeter_event_delete_cb(xhardmeter_window_t *w)
{
  xhardmeter_event_t *ev = w->xtop->current_xexperiment->current_xevent;

  if (ev == NULL)
    return;
  gtk_container_remove(ev->parent->property_frame, GTK_WIDGET(ev->widget));
  w->xtop->current_xexperiment->current_xevent = NULL;
  gtk_widget_destroy(GTK_WIDGET(ev->button));
}

void xhardmeter_get_ebs(xhardmeter_ebs_t *xebs, hardmeter_ebs_t *ebs)
{
  GSList *l;

  ebs->enabled = 1;

  for (l = xebs->type; l != NULL; l = l->next) {
    if (GTK_CHECK_MENU_ITEM(l->data)->active)
      ebs->type = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(l->data), "value"));
  }

  for (l = xebs->sample; l != NULL; l = l->next) {
    if (GTK_CHECK_MENU_ITEM(l->data)->active)
      ebs->sample = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(l->data), "value"));
  }

  ebs->interval = strtoll(gtk_entry_get_text(xebs->interval), NULL, 10);
  ebs->buffer = strtol(gtk_entry_get_text(xebs->buffer), NULL, 10);
  ebs->max = strtol(gtk_entry_get_text(xebs->max), NULL, 10);
}

void xhardmeter_set_ebs(xhardmeter_ebs_t *xebs, hardmeter_ebs_t *ebs)
{
  GSList *l;
  char buf[128];

  for (l = xebs->type; l != NULL; l = l->next) {
    if (ebs->type == GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(l->data), "value")))
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(l->data), TRUE);
  }

  for (l = xebs->sample; l != NULL; l = l->next) {
    if (ebs->sample == GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(l->data), "value")))
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(l->data), TRUE);
  }

  snprintf(buf, sizeof(buf), "%lld", ebs->interval);
  gtk_entry_set_text(xebs->interval, buf);

  snprintf(buf, sizeof(buf), "%d", ebs->buffer);
  gtk_entry_set_text(xebs->buffer, buf);

  snprintf(buf, sizeof(buf), "%d", ebs->max);
  gtk_entry_set_text(xebs->max, buf);
}
