#ifndef _XHARDMETER_H_
#define _XHARDMETER_H_ 1

#include <gtk/gtk.h>

enum prop_type {
  PROP_BOOL,
  PROP_THRESHOLD,
  PROP_COUNTER,
  PROP_IOQ,
};

struct hardmeter_event_property {
  const char *name;
  enum prop_type type;
  const char *comment;
};
typedef struct hardmeter_event_property hardmeter_event_property_t;

struct hardmeter_event {
  const char *name;
  hardmeter_event_property_t *properties;
  struct hardmeter_event *tag_base;
  const char *comment;
};
typedef struct hardmeter_event hardmeter_event_t;

struct hardmeter_ebs {
  unsigned int enabled:1;
  unsigned int type:1; /* 0 - impresize ebs, 1 - presize ebs */
  unsigned int sample:1; /* 0 - EIP only, 1 - all registers */
  gint64 interval;
  int buffer;
  int max;
};
typedef struct hardmeter_ebs hardmeter_ebs_t;

struct hardmeter_source {
  int (*start)(void **top_datap, void *userdata);
  int (*end)(void *top_data);
  int (*program)(void *top_data, char **name, char **path);
  int (*default_flag)(void *top_data, char **name, guint64 *val);
  int (*experiment_start)(void **experiment_datap, void *top_data, char **name);
  int (*experiment_end)(void *experiment_data);
  int (*event_start)(void **event_datap, void *experiment_data, char **name, hardmeter_event_t **base, hardmeter_ebs_t *ebs);
  int (*event_end)(void *event_data);
  int (*flag)(void *event_data, char **name, guint64 *val);
  int (*error_cleanup)(void *top_data);
};
typedef struct hardmeter_source hardmeter_source_t;

struct hardmeter_builder {
  int (*start)(void **top_datap, void *userdata);
  int (*end)(void *top_data);
  int (*program)(void *top_data, const char *name, const char *path);
  int (*default_flag)(void *top_data, const char *name, guint64 val);
  int (*experiment_start)(void **experiment_datap, void *top_data, const char *name);
  int (*experiment_end)(void *experiment_data);
  int (*event_start)(void **event_datap, void *experiment_data, const char *name, hardmeter_event_t *base, hardmeter_ebs_t *ebs);
  int (*event_end)(void *event_data);
  int (*flag)(void *event_data, const char *name, guint64 val);
  int (*error_cleanup)(void *top_data);
};
typedef struct hardmeter_builder hardmeter_builder_t;

/* event.c */
extern hardmeter_event_t hardmeter_registers[];
extern hardmeter_event_t hardmeter_event_non_retirement[];
extern hardmeter_event_t hardmeter_event_at_retirement[];
extern hardmeter_event_t hardmeter_event_tagging[];

/* mediator.c */
extern hardmeter_source_t *hardmeter_empty_source;
int hardmeter_build(hardmeter_source_t *source,
		    void *source_data,
		    hardmeter_builder_t *builder,
		    void *build_data);
int hardmeter_error(const char *format, ...)  G_GNUC_PRINTF (1, 2);

/* abyss.c */
extern hardmeter_builder_t *hardmeter_abyss_output_builder;

/* file.c */
extern hardmeter_builder_t *hardmeter_file_builder;
extern hardmeter_source_t *hardmeter_file_source;

/* perfctr.c .. plan */
extern hardmeter_builder_t *hardmeter_perfctr_p4_builder; /* plan */

/* gtk.c */
typedef struct xhardmeter_window xhardmeter_window_t;
typedef struct xhardmeter_top xhardmeter_top_t;
typedef struct xhardmeter_program xhardmeter_program_t;
typedef struct xhardmeter_experiment xhardmeter_experiment_t;
typedef struct xhardmeter_event xhardmeter_event_t;
typedef struct xhardmeter_flag xhardmeter_flag_t;
typedef struct xhardmeter_flag_cb xhardmeter_flag_cb_t;
typedef struct xhardmeter_ebs xhardmeter_ebs_t;

struct xhardmeter_window {
  xhardmeter_top_t *xtop;
  /* widgets */
  GtkWindow *window;
  GtkItemFactory *item_factory;
  GtkCList *clist; /* program */
  GSList *default_xflags;
  GtkNotebook *notebook; /* experiments */
};

struct xhardmeter_top {
  gboolean is_modified; /* not used yet. */
  xhardmeter_window_t *parent;
  GSList *xexperiments;
  xhardmeter_experiment_t *current_xexperiment;
  char *filename;
  void *work; /* for source */
};

struct xhardmeter_experiment {
  xhardmeter_top_t *parent;
  GSList *xevents;
  xhardmeter_event_t *current_xevent;
  /* widgets */
  GtkLabel *label;
  GtkBox *event_box;
  GtkContainer *property_frame;
};

struct xhardmeter_event {
  xhardmeter_experiment_t *parent;
  hardmeter_event_t *event;
  GSList *xflags;
  xhardmeter_ebs_t *ebs;
  /* widgets */
  GtkToggleButton *button;
  GtkWidget *widget;
};

struct xhardmeter_flag {
  hardmeter_event_property_t *event_property;
  xhardmeter_flag_cb_t *cb;
  /* widgets */
  GtkToggleButton *button;
  GtkWidget *widget;
};

struct xhardmeter_flag_cb {
  GtkWidget *(*new)(void);
  void (*set)(GtkWidget *w, guint64 val);
  void (*get)(GtkWidget *w, guint64 *val);
  void (*clear)(GtkWidget *w);
};

struct xhardmeter_ebs {
  GSList *type;
  GSList *sample;
  GtkEntry *interval;
  GtkEntry *buffer;
  GtkEntry *max;
};

extern hardmeter_source_t *hardmeter_gtk_p4_source;
extern hardmeter_builder_t *hardmeter_gtk_p4_builder;
xhardmeter_window_t *xhardmeter_window_new(void);
void xhardmeter_set_menu_sensitive(xhardmeter_window_t *w, gboolean is_default_menu);

void xhardmeter_free_top(xhardmeter_top_t *top);
void xhardmeter_free_experiment(xhardmeter_experiment_t *exp);
void xhardmeter_free_event(xhardmeter_event_t *event);

/* xprogram.c */
GtkWidget *xhardmeter_program_new(xhardmeter_window_t *w);
void xhardmeter_program_add_cb(xhardmeter_window_t *w);
void xhardmeter_program_edit_cb(xhardmeter_window_t *w);
void xhardmeter_program_delete_cb(xhardmeter_window_t *w);

/* xexperiment.c */
GtkWidget *xhardmeter_experiment_frame_new(xhardmeter_window_t *w);
xhardmeter_experiment_t *xhardmeter_experiment_new(xhardmeter_top_t *t, const char *name);
xhardmeter_event_t *xhardmeter_event_new(xhardmeter_experiment_t *exp, const char *name, hardmeter_event_t *event);
void xhardmeter_get_ebs(xhardmeter_ebs_t *xebs, hardmeter_ebs_t *ebs);
void xhardmeter_set_ebs(xhardmeter_ebs_t *xebs, hardmeter_ebs_t *ebs);

void xhardmeter_experiment_add_cb(xhardmeter_window_t *w);
void xhardmeter_experiment_rename_cb(xhardmeter_window_t *w);
void xhardmeter_experiment_delete_cb(xhardmeter_window_t *w);

void xhardmeter_event_add_cb(GtkWidget *menuitem, xhardmeter_window_t *w);
void xhardmeter_event_rename_cb(xhardmeter_window_t *w);
void xhardmeter_event_delete_cb(xhardmeter_window_t *w);

/* xutil.c */
const char *xhardmeter_xfileselection(GtkWindow *window);
char *xhardmeter_text_dialog(GtkWindow *window, const char *title, const char *text);
GtkWidget *xhardmeter_label_with_tooltips(const char *label, const char *tip);
int xhardmeter_ask(GtkWindow *window, const char *msg);
#endif
