#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <limits.h>
#include <string.h>
#include "hardmeter.h"

#define IS_SPACE(c) ((c) == ' ' || (c) == '\t')

#define CHECK_ROOM(n, m) if ((n) + (m) >= PATH_MAX) return NULL
static char *expand_filename(const char *src)
{
	static char buf[PATH_MAX];
	int i, j;
	time_t t;
	struct tm tm;
	char pidbuf[32];
	int n;
	time(&t);
	localtime_r(&t, &tm);
	for (i = j = 0; src[i] != '\0' && j < PATH_MAX - 1; i++) {
		if (src[i] != '%') {
			buf[j++] = src[i];
			continue;
		}
		i++;
		switch (src[i]) {
		case 'Y': /*  year (1970...) */
			CHECK_ROOM(j, 4);
			n = tm.tm_year + 1900;
			buf[j++] = '0' + (n / 1000); n %= 1000;
			buf[j++] = '0' + (n / 100); n %= 100;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'y': /* year (00..99) */
			CHECK_ROOM(j, 2);
			n = tm.tm_year % 100;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'm': /* month (01..12) */
			CHECK_ROOM(j, 2);
			n = tm.tm_mon + 1;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'd': /* day of month (01..31) */
			CHECK_ROOM(j, 2);
			n = tm.tm_mday;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'H': /* hour (00..23) */
			CHECK_ROOM(j, 2);
			n = tm.tm_hour;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'M': /* minute (00..59) */
			CHECK_ROOM(j, 2);
			n = tm.tm_min;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'S': /* second (00..60) */
			CHECK_ROOM(j, 2);
			n = tm.tm_sec;
			buf[j++] = '0' + (n / 10); n %= 10;
			buf[j++] = '0' + n;
			break;
		case 'P': /* process id */
			snprintf(pidbuf, sizeof(pidbuf), "%d", getpid());
			n = strlen(pidbuf);
			CHECK_ROOM(j, n);
			memcpy(buf + j, pidbuf, n);
			j += n;
			break;
		default:
			CHECK_ROOM(j, 2);
			buf[j++] = '%';
			buf[j++] = src[i];
		}
	}
	buf[j] = '\0';
	return buf;
}

static char *get_rc(void)
{
	static char buf[PATH_MAX];
	const char *home = getenv("HOME");
	if (home == NULL)
		return NULL;
	snprintf(buf, sizeof(buf), "%s/.hardmeterrc", home);
	return buf;
}

int hardmeter_parse_rc(struct hardmeter_option_t *opt, char **filenamep, const char *name, const char **err)
{
	static char *rcfile = NULL;
	const char *errmsg = NULL;
	FILE *fp;
	char buf[BUFSIZ];
	int namelen = strlen(name);
	char *key, *val;
	char *p;

	if (rcfile == NULL) {
		rcfile = get_rc();
		if (rcfile == NULL) {
			errmsg = "cannot get environemnt variable HOME.";
			goto error;
		}
	}
	fp = fopen(rcfile, "r");
	if (fp == NULL) {
		errmsg = "cannot open .hardmeterrc file.";
		goto error;
	}
	opt->interval = HARDMETER_DEFAULT_INTERVAL;
	opt->count = HARDMETER_DEFAULT_COUNT;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		for (p = buf; IS_SPACE(*p); p++);
		if (*p == '#' || *p == '\n')
			continue;
		if (strncmp(p, name, namelen) == 0 && p[namelen] == '.') {
			key = p + namelen + 1;
			while (IS_SPACE(*key))
				key++;
			/* delete comment and trailing spaces. */
			p = strchr(key, '#');
			if (p == NULL) {
				p = key + strlen(key);
			} else {
				*p = '\0';
			}
			for (p--; IS_SPACE(*p) || *p == '\n' || *p == '\r'; p--) {
				*p = '\0';
			}
			val = strpbrk(key, " \t");
			if (val == NULL)
				continue;
			*(val++) = '\0';
			while (IS_SPACE(*val))
				val++;
			if (strcmp(key, "user") == 0) {
				opt->user = atoi(val);
			} else if (strcmp(key, "kernel") == 0) {
				opt->kernel = atoi(val);
			} else if (strcmp(key, "interval") == 0) {
				opt->interval = atoi(val);
			} else if (strcmp(key, "count") == 0) {
				opt->count = atoi(val);
			} else if (strcmp(key, "event") == 0) {
				opt->template = hardmeter_search_template(val, NULL);
			} else if (strcmp(key, "dumpfile") == 0) {
				*filenamep = expand_filename(val);
			}
		}
	}
	if (opt->user != 0 && opt->kernel != 0) {
		errmsg = "both user and kernel was not set.";
		goto error;
	}
	if (opt->interval <= 0) {
		errmsg = "interval was not positive.";
		goto error;
	}
	if (opt->count <= 0) {
		errmsg = "count was not positive.";
		goto error;
	}
	return 0;
error:
	if (err != NULL)
		*err = errmsg;
	return -1;
}
