#!/usr/bin/env ruby
# $Id: hatena_diary_writer.rb,v 1.8 2007/12/27 22:23:58 samurai20000 Exp $

require 'forwardable'
require 'kconv'
require 'logger'
require 'mechanize'
require 'optparse'
require 'pathname'
require 'time'
require 'yaml'

class HatenaDiaryWriter
  def initialize(id, passwd, group)
    @id = id
    @passwd = passwd
    @group = group
    @agent = WWW::Mechanize.new
  end
  
  def login
    diary_page = @agent.get(hatena_url)
    login_link = diary_page.links.text(convert_char_code('ログイン'))
    login_page = @agent.get(login_link.href)
    login_form = login_page.forms.first
    login_form['name'] = @id
    login_form['password'] = @passwd
    @agent.submit(login_form)
  end
  
  def edit(date, content)
    edit_page = @agent.get(hatena_url + 'edit')
    edit_form = edit_page.forms.name('edit').first
    edit_form['year'], edit_form['month'], edit_form['day'] = parse_date(date)
    edit_form['body'] = convert_char_code(content)
    ok_button = edit_form.buttons.name('edit')
    @agent.submit(edit_form, ok_button)
  end

  def img_upload(date, img)
    upload_page =  @agent.get(hatena_url + 'edit?date=' + date)
    upload_form = upload_page.forms.name('edit').first
    upload_form['year'], upload_form['month'], upload_form['day'] = parse_date(date)
    ok_button = upload_form.buttons.name('edit')
    delete_image = upload_form.checkboxes.name('deleteimage')
    
    if delete_image.empty?
      upload_form.file_uploads.first.file_name = img
      @agent.submit(upload_form, ok_button)
    else
      delete_image.check
      @agent.submit(upload_form, ok_button)
      img_upload(date, img)
    end
  end
  
  private
  
  def hatena_url
    if @group.empty?
      "http://d.hatena.ne.jp/#{@id}/"
    else
      "http://#{@group}.g.hatena.ne.jp/#{@id}/"
    end
  end
  
  def convert_char_code(text)
    if @group.empty?
      text.toeuc
    else
      text.toutf8
    end
  end

  def parse_date(date)
    date.unpack("a4a2a2")
  end
end

class DiaryWriterConfig
  def DiaryWriterConfig.load(file)
    c = new()
    c.instance_eval File.read(file)
    c
  end

  def initialize
    @id = '(user name)'
    @passwd = '(password)'
    @group = ''
    @diary_dir = '.'
    @upload_table_file = 'touch.txt'
  end
  
  attr_accessor :id, :passwd, :group, :diary_dir, :upload_table_file
end

class DiaryInvalidFileException < Exception; end

class DiaryFile
  extend Forwardable
  
  def initialize(path)
    raise DiaryInvalidFileException unless path.file?
    
    @path = path
    
    unless /(\d\d\d\d)[^\d]?(\d{1,2})[^d]?(\d{1,2})\.(txt|jpg|jpeg)\z/i =~ @path
      raise DiaryInvalidFileException
    end

    @is_image = 'txt'.casecmp($4) != 0

    year,month,day = $1, $2, $3
    month.sub!(/\d+/){ sprintf("%02d", $&.to_i)}
    day.sub!(/\d+/){ sprintf("%02d", $&.to_i)}
    @date = year + month + day

    @mtime = @path.mtime
  end
  attr_reader :date, :mtime
  
  def_delegator(:@path, :read)
  
  def name
    @path.to_s
  end
  
  def image?
    @is_image
  end
end

class DiaryUploadTable
  def initialize(file)
    @path = Pathname.new(file)
    @touch = {}
    @touch = YAML.load(@path.read) if @path.file?
    @touch = {} unless @touch.class == Hash
  end
  
  def uploaded?(dfile)
    old = @touch[key(dfile)]
    return false unless old
    return dfile.mtime <= Time.parse(old)
  end
  
  def update(dfile)
    @touch[key(dfile)] = dfile.mtime.to_s
  end
  
  def store
    @path.open('w') do |file|
      YAML.dump(@touch, file)
    end
  end
  
  private
  
  def key(dfile)
    dfile.date + (dfile.image? ? '.img' : '.txt')
  end
end
