/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmppoller;

import java.net.InetAddress;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.ejb.EJBException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;

import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.sharedtable.bean.DataTable;
import com.clustercontrol.sharedtable.bean.ValueObject;
import com.clustercontrol.snmppoller.bean.PollingConfig;
import com.clustercontrol.snmppoller.job.CheckJob;
import com.clustercontrol.snmppoller.job.PollingThread;
import com.clustercontrol.snmppoller.job.SnmpPollingJob;
import com.clustercontrol.snmppoller.util.JNDIConnectionManager;

/**
 * SNMPݡ󥰤ǼΡñ̤ݻ륯饹
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class SnmpSharedTable {
	protected static Log m_log = LogFactory.getLog( SnmpSharedTable.class );
	
	private final String m_checkJobGroup = "SNMP_CHECK_JOB";  // åѥ֥롼פ̾
	private String m_quartzJndiName = null;  // QuartzlookupݤJNDI̾
	
	private int m_defaultPort = -1;
	private int m_defaultVersion = 1;
	private String m_defaultCommunity = "public";
	private int m_defaultRetries = -1;
	private int m_defaultTimeout = 1000;  // milliseconds
	
	private String m_jndiName;  // JNDIϿݤ̾
	private String m_groupName; // QuartzϿݤ̾
	
	private String m_checkJobName; // åѥ֤̾
	
	private String m_checkInterval = "34 */10 * * * ? *"; // å֤εư
	private List m_oidList;  // OIDΥå
	
	// ͱͽ()
	// å˼ֳ֤ꤳδְʾ廲ȤƤʤơ֥Ϻ
	private int m_gracePeriod = 5;
	
	// 塼󥰤¹Ԥ뤫ݤΥե饰
	private boolean m_scheduleFlag = false;
	
	// Υơ֥ưƤݡ顼(ʣ)ǺǤֳ֤Τ
	// QuartzΥ֤ȤϿƤΤǺǾμֳ֤Τ
	private int m_minInterval = Integer.MAX_VALUE;
	
	private HashMap<String, DataTable> m_dataTable;   // ΡΥơ֥Υޥå
	private HashMap<String, PollingConfig>  m_pollerConf;  // ΡΥޡơ֥
	
	public SnmpSharedTable(String jndiName, String groupName){
		this.m_jndiName = jndiName;
		this.m_groupName = groupName;
		this.m_dataTable = new HashMap<String, DataTable>();
		this.m_pollerConf = new HashMap<String, PollingConfig>();
		this.m_checkJobName = groupName;
	}
	
	/**
	 * å֤QuartzϿޤ
	 */
	public void scheduleCheckJob(){
		// ˥֤
		JobDetail job = new JobDetail(m_checkJobName, m_checkJobGroup, CheckJob.class);
		//ִλ˺ʤ褦ˤ
		job.setDurability(true);
		
		//JobDetail˸ƤӽФ᥽åɤΰ
		job.getJobDataMap().put("jndiName", m_jndiName);
		
		scheduleJob(m_checkJobName, m_checkJobGroup, job, m_checkInterval);
	}
	
	/**
	 * å֤ޤ
	 */
	public void deleteCheckJob(){
		m_log.debug("deleteCheckJob()  JobName : " + m_checkJobName + "  JobGroup : " + m_checkJobGroup);
		
		// ֥塼
		Scheduler scheduler = getScheduler();
		
		// ϿƤ른֤
		try {
			scheduler.deleteJob(m_checkJobName, m_checkJobGroup);
		} catch (SchedulerException e) {
			m_log.error(e);
		}
	}
	
	/**
	 * ְֳʾ廲ȤƤʤơ֥Ĵ١ݡ顼μѹޤ
	 *
	 */
	public void checkAll() {
		m_log.debug("checkAll()  Group : " + m_groupName + "  ȤƤʤơ֥õޤ");
		
		// ƤΥΡɤФƥåԤ
		Set<String> facilityIdSet = m_dataTable.keySet();		
		
		// ǸƤӽФschedule()m_dataTableǤ뤿ᡢ
		// ŪString˥ԡƤ롼פ
		String[] facilityIds = 
			(String[])facilityIdSet.toArray(new String[facilityIdSet.size()]);
		
		for(int i=0; i<facilityIds.length; i++){
			check(facilityIds[i]);
		}
		
		// Quartzѹ
		if(m_scheduleFlag){
			schedule();
		}
		
		m_log.debug("Group : " + m_groupName + "  ȤƤʤơ֥Υåλ");
	}
	
	/**
	 * ְֳʾ廲ȤƤʤơ֥Ĵ١ݡ顼μѹޤ
	 *
	 */
	private void check(String facilityId) {
		m_log.debug("check()  Group : " + m_groupName + "  " + facilityId
				+ "  ȤƤʤơ֥õޤ");
		
		synchronized (m_pollerConf) {			
			// ֳ֤ѹԤݤ򼨤ե饰
			boolean changeFlag = false;
			
			// ߻ޤ
			long now = System.currentTimeMillis();
			
			// ǽ狼ηв֤򻻽
			int elapsedTime;
			try {
				elapsedTime = (int)((now - getLastReferenceTime(facilityId))/1000);
				
				// ֳ֤
				int interval = this.getInterval(facilityId);
				
				// ǥХå
				m_log.debug("в " + elapsedTime + "s   ֳ " + interval + "s");
				
				// в֤⾮ʼֳ֤Ϻ
				while(interval != 0 && (interval + m_gracePeriod) < elapsedTime){
					// ǥХå
					m_log.debug("Group : " + m_groupName + "  " + facilityId + "   " + interval);
					
					// ֳ֤
					removePollingInterval(facilityId, interval);
					changeFlag = true;
					
					// ֳ֤ټ
					interval = this.getInterval(facilityId);
				}
				
				if(interval == 0){
					// ֳ֤ƺ줿ϼߤ
					this.deleteTable(facilityId);
				} else if(changeFlag){
					// ֳ֤ѹQuartzμѹ
					m_log.debug("Quartzμѹ  " + facilityId);
					m_scheduleFlag = true;
				}
				
			} catch (FacilityNotFoundExciption e) {
				m_log.error(e);
			}
		}
		
		m_log.debug("Group : " + m_groupName + "  " + facilityId
				+ "  ȤƤʤơ֥Υåλ");
	}
	
	/**
	 * ˥ơ֥롣
	 */
	private void createTable(String facilityId){
		m_log.debug("createTable()  Group : " + m_groupName + "  " + facilityId);
		
		PollingConfig pConfig = new PollingConfig(
				getIpAddress(facilityId),
				this.m_defaultPort,
				this.m_defaultVersion,
				this.m_defaultCommunity,
				this.m_defaultRetries,
				this.m_defaultTimeout
		);
		
		synchronized (m_pollerConf) {
			// ͳǼѤΥơ֥
			this.m_pollerConf.put(facilityId, pConfig);
			this.m_dataTable.put(facilityId, new DataTable());
		}
		
		// åɤ򤿤Ƥƥݡ󥰤¹Ԥ
		PollingThread pt = new PollingThread(
				pConfig.getSnmpConfig(),
				this.m_oidList,
				this.m_jndiName,
				facilityId
		);
		Thread th = new Thread(pt);
		th.start();
	}
	
	/**
	 * ơ֥
	 * ơ֥ͤǼ뤿ưƤݡ顼ߤ롣
	 * @param facilityId
	 */
	public void deleteTable(String facilityId){
		m_log.debug("deleteTable()  Group : " + facilityId);
		
		synchronized (m_pollerConf) {
			this.m_dataTable.remove(facilityId);
			this.m_pollerConf.remove(facilityId);
			m_scheduleFlag = true;
		}
	}
	
	/**
	 * ƤΥơ֥
	 * ơ֥ͤǼ뤿ưƤݡ顼ߤ롣
	 */
	public void deleteTableAll(){
		m_log.debug("deleteTableAll()  Group : "  + m_groupName);
		
		Iterator itr = getTargetNodeList().iterator();
		
		synchronized (m_pollerConf) {
			while(itr.hasNext()){
				String facilityId = (String)itr.next();
				
				// ơ֥
				this.m_dataTable.remove(facilityId);
				this.m_pollerConf.remove(facilityId);
			}
		}
		
		// ϿƤ른֤
		try {
			// ֥塼
			Scheduler scheduler = getScheduler();
			
			// Υ롼̾ϿƤ른֤ƺ
			String[] jobNames = scheduler.getJobNames(m_groupName);
			
			for(int i=0; i<jobNames.length; i++){
				// ǥХå
				m_log.debug("deleteJob  JobName : " + jobNames[i] + "  JobGroup : " + m_groupName);
				
				scheduler.deleteJob(jobNames[i], m_groupName);
			}
		} catch (SchedulerException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * ߤΥݥȥΥΡɾơ֥ȿǤ롣
	 */
	public void refreshTable(){
		m_log.debug("refreshTable()  Group : " + m_groupName + "  ݥȥIPɥ쥹ΰۤʤơ֥õޤ");
		
		// ƤΥΡɤФƥåԤ
		Set<String> facilityIdSet = m_dataTable.keySet();		
		
		// ǸƤӽФschedule()m_dataTableǤ뤿ᡢ
		// ŪString˥ԡƤ롼פ
		String[] facilityIds = 
			(String[])facilityIdSet.toArray(new String[facilityIdSet.size()]);
		
		for(int i=0; i<facilityIds.length; i++){
			String facilityId = facilityIds[i];
			
			// ơ֥ϿƤIPɥ쥹ޤ
			PollingConfig config = getConfig(facilityId);
			String tableIpString = config.getSnmpConfig().getAddress().getHostAddress();
			
			// ݥȥϿƤIPɥ쥹ޤ
			InetAddress repIpAddress = getIpAddress(facilityId);

			if(m_log.isDebugEnabled()){
				m_log.debug("Table IP Address : " + tableIpString);
				m_log.debug("Repository IP Address : " + repIpAddress.getHostAddress());
			}
			
			// ơ֥ϿƤIPɥ쥹ȥݥȥƤۤʤ
			if(!tableIpString.equals(repIpAddress.getHostAddress())){
				m_log.debug("Change IP Address " + facilityId + " : "
						+ tableIpString + " -> " + repIpAddress.getHostAddress());
				
				// ơ֥Ƥ
				try {
					getTable(facilityId).setValueAll(new DataTable());
				} catch (FacilityNotFoundExciption e) {
					m_log.debug(e);
				}

				// IPɥ쥹
				config.getSnmpConfig().setAddress(repIpAddress);
				
				m_log.debug("Quartzμѹ  " + facilityId);
				m_scheduleFlag = true;
			}
		}
		
		// Quartzѹ
		if(m_scheduleFlag){
			schedule();
		}
		
		m_log.debug("Group : " + m_groupName + "  ݥȥIPɥ쥹ΰۤʤơ֥Υåλ");
	}
	
	/**
	 * ݡ󥰤Υ塼󥰤ꤹ
	 */
	public void schedule() {
		m_log.debug("schedule()  Group : "  + m_groupName);
		
		if(m_scheduleFlag == false){
			return;
		}
		
		synchronized (m_pollerConf) {
			m_log.debug("塼ѹ  Group : "  + m_groupName);
			
			// ǽƤμֳ֤Ǽ
			int[] intervalList = PollingConfig.getAvailableIntervals();
			
			// ǾΤ˽ͤȤƺͤꤹ
			m_minInterval = Integer.MAX_VALUE;
			
			// Quartzˤϡֳ˥֤Ͽ
			// ƥ֤˼оݤΥΡ(ʣ)IPɥ쥹°ꤹ
			for(int i=0; i<intervalList.length; i++){
				int interval = intervalList[i];
				
				// ֤̾
				String jobName = "Int" + interval + "Sec";
				
				// ˥֤
				JobDetail job = new JobDetail(jobName, m_groupName, SnmpPollingJob.class);
				//ִλ˺ʤ褦ˤ
				job.setDurability(true);
				
				// ݻƤơ֥ƤΥݡ󥰾ФƼֳ֤å
				// μֳ֤ΤΤꥹȤȤQuartz
				Iterator itr = m_pollerConf.keySet().iterator();
				
				HashMap<String, SnmpConfig> confMap = new HashMap<String, SnmpConfig>();
				while(itr.hasNext()){
					String facilityId = (String)itr.next();
					PollingConfig conf = (PollingConfig)m_pollerConf.get(facilityId);
					// åоݤΥեƥ(Ρ)κǾֳ꤬μֳ֤Ȱפ
					// μֳ֤ǤμоݥΡɤȤơQuartϿ롣
					if(conf.getMinInterval() == interval){  
						confMap.put(facilityId, conf.getSnmpConfig());
					}
				}
				
				// оݤΥΡɤҤȤĤʤϡ֤򥹥塼󥰤ʤ
				if(confMap.size() == 0){
					// ֥塼
					Scheduler scheduler = getScheduler();
					
					// ϿƤ른֤
					try {
						scheduler.deleteJob(jobName, m_groupName);
					} catch (SchedulerException e) {
						e.printStackTrace();
					}
				} else {
					// оݥΡɤμ򥸥֤Ͽ塼󥰤
					
					m_minInterval = Math.min(m_minInterval, interval);
					
					job.getJobDataMap().put("jndiName", m_jndiName);  // ơ֥lookup뤿JNDI̾
					job.getJobDataMap().put("interval", interval);    // ֳ
					job.getJobDataMap().put("oidList", m_oidList);     // OIDΥꥹ
					// ΥꥹȤ򥸥֤Ͽ
					job.getJobDataMap().put("snmpConfMap", confMap);
					
					//塼
					String cronString = PollingConfig.parseCronExpression(interval);
					scheduleJob(jobName, m_groupName, job, cronString);
				}
			}
		}
		
		m_scheduleFlag = false;
	}
	
	private void scheduleJob(String jobName, String groupName, JobDetail job, String cronString) {
		m_log.debug("scheduleJob()  Group : " + m_groupName + " JobName : " + jobName + " " +  groupName + " : " + cronString);
		
		//CronTrigger
		CronTrigger cronTrigger = new CronTrigger(jobName, groupName);
		
		try {
			cronTrigger.setCronExpression(cronString);
		} catch (ParseException e) {
			// 顼
			m_log.error(e);
		}
		
		// Scheduler˥֤Ͽ
		try {
			// ֥塼
			Scheduler scheduler = getScheduler();
			
			// ϿƤ른֤(ϿƤʤϲ⤪ʤ)
			scheduler.deleteJob(jobName, groupName);
			
			// ֤Ͽ
			scheduler.scheduleJob(job, cronTrigger);
		} catch (SchedulerException e) {
			// 顼
			m_log.error(e);
		}
	}
	
	private Scheduler getScheduler() {
		Scheduler scheduler = null;
		try {
			//QuartzScheduleråå
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Object obj = iniCtx.lookup(m_quartzJndiName);
			scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);
		} catch (NamingException e) {
			// 顼
			m_log.error(e);
		}
		return scheduler;
	}
	
	/**
	 * եƥIDΥΡɤIPɥ쥹ޤ
	 * 
	 * @param facilityId оݤΥեƥID
	 */
	private InetAddress getIpAddress(String facilityId) {
		// IPɥ쥹
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList<String> attributes = new ArrayList<String>();
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			HashMap nodeInfo = bean.getNodeDetail(facilityId, attributes);
			bean.remove();
			
			String ipAddressStr = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			return InetAddress.getByName(ipAddressStr);
		} 
		catch (Exception e) {			
			String message= "Couldn't get an IP Address : " + facilityId;
			throw new EJBException(message, e) ;
		}
	}
	
	public PollingConfig getConfig(String facilityId) {
		PollingConfig conf = (PollingConfig)m_pollerConf.get(facilityId);
		
		if(conf == null){
			// 顼
			m_log.debug("ϿƤʤեƥIDǼμ¹Ԥޤ : " + facilityId);
		}
		
		return conf;
	}
	
	public void setPollingInterval(String facilityId, int interval) {
		synchronized (m_pollerConf) {
			// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
			if(this.m_dataTable.containsKey(facilityId) == false){
				this.createTable(facilityId);
			}
			
			// оݥΡɤ
			PollingConfig conf = this.getConfig(facilityId);
			
			// ֳ֤
			// ǥݡ󥰤ѹɬפʾˤQuartzΥ塼ѹ
			if(conf.addInterval(interval)){  // ֳ֤ɲä뤳ȤǼֳ֤ѹtrue֤
				m_scheduleFlag = true;
				
				// ꤵƤǾμֳ֤⾮ʼֳ֤ꤵ줿ϥꥹ塼
				if(interval < m_minInterval){
					schedule();
				}
			}
		}
	}
	
	private void removePollingInterval(String facilityId, int interval) throws FacilityNotFoundExciption{
		synchronized (m_pollerConf) {
			// оݥΡɤ
			PollingConfig conf = this.getConfig(facilityId);
			
			// ֳ֤
			conf.removeInterval(interval);
			m_scheduleFlag = true;
		}
	}
	
	/**
	 * ݡ顼μֳ֤֤ޤ
	 * @param facilityId ΡɤΥեƥID
	 * @return ֳ
	 * @throws FacilityNotFoundExciption 
	 */
	public int getInterval(String facilityId) throws FacilityNotFoundExciption{
		return this.getConfig(facilityId).getMinInterval();
	}
	
	/**
	 * ΥեƥIDΥơ֥ФǽȻ֤ޤ
	 * @param facilityId եƥID
	 * @return ǽȻ
	 * @throws FacilityNotFoundExciption 
	 */
	public long getLastReferenceTime(String facilityId) throws FacilityNotFoundExciption{
		return getTable(facilityId).getLastReference();
	}
	
	private DataTable getTable(String facilityId) throws FacilityNotFoundExciption{
		DataTable table = (DataTable)m_dataTable.get(facilityId);
		
		if(table == null){
			// 顼
			m_log.debug("ϿƤʤեƥIDǥơ֥뤬Ȥޤ : " + facilityId);
			throw new FacilityNotFoundExciption();
		}
		
		return table;
	}
	
	/**
	 * եƥIDΥΡɤMIBͤͭơ֥˳Ǽޤ
	 * @param facilityId ΡɤΥեƥID
	 * @param oid ͤOID
	 * @param date 
	 * @param value 
	 * @throws FacilityNotFoundExciption 
	 */
	public void putValue(String facilityId, String oid, long date, Object value) throws FacilityNotFoundExciption {
		m_log.debug("putValue()   Group : " + m_groupName + "  " + facilityId + " : " + value);
		
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);			
		}
		
		getTable(facilityId).putValue(oid, date, value);
	}
	
	/**
	 * եƥIDΥΡɤζͭơ֥ʣμͤǳǼޤ
	 * ƱOIDΤΤͤ񤭤ޤ
	 * 
	 * @param facilityId եƥID
	 * @param table ͤǼƤơ֥
	 * @throws FacilityNotFoundExciption 
	 */
	public void putValueAll(String facilityId, DataTable table) throws FacilityNotFoundExciption {
		m_log.debug("putValueAll()   Group : " + m_groupName + "  " + facilityId);
		
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		getTable(facilityId).putValueAll(table);
	}
	
	/**
	 * եƥIDΥΡɤζͭơ֥Υơ֥֤ޤ
	 * μͤϥꥢޤ
	 * 
	 * @param facilityId եƥID
	 * @param table ͤǼƤơ֥
	 * @throws FacilityNotFoundExciption 
	 * @throws FacilityNotFoundExciption 
	 */
	public void setValueAll(String facilityId, DataTable table) throws FacilityNotFoundExciption {
		m_log.debug("setValueAll()   Group : " + m_groupName + "  " + facilityId);
		
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		getTable(facilityId).setValueAll(table);
	}
	
	/**
	 * եƥλOIDμ֤ͤޤ
	 * Ʊˡꤵ줿䤤碌ֳ֤OIDͤ褦˥ݡ顼ꤷޤ
	 * 
	 * @param facilityId եƥID
	 * @param interval 䤤碌ֳ()
	 * @param oid SNMPOID
	 * @return OIDμ
	 * @throws FacilityNotFoundExciption 
	 */
	public ValueObject getValue(String facilityId, int interval, String oid) {
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// ꤷݡ顼Υ塼ѹѹ
		setPollingInterval(facilityId, interval);
		try {
			return getTable(facilityId).getValue(oid);
		} catch (FacilityNotFoundExciption e) {
			m_log.debug(e);
		}
		return null;
	}
	
	/**
	 * եƥλOID۲MIBĥ꡼μ֤ͤޤ
	 * Ʊˡꤵ줿䤤碌ֳ֤OIDͤ褦˥ݡ顼ꤷޤ
	 * 
	 * @param facilityId եƥID
	 * @param interval 䤤碌ֳ()
	 * @param oid SNMPOID
	 * @return OIDʲΥĥ꡼˴ޤޤͤΥå(ʤnull֤)
	 * @throws FacilityNotFoundExciption 
	 */
	public Set getValueSet(String facilityId, int interval, String oid) {
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// ꤷݡ顼Υ塼ѹѹ
		setPollingInterval(facilityId, interval);
		
		// ͤǼ륻å
		HashSet set = new HashSet();
		
		DataTable dt;
		try {
			dt = getTable(facilityId).getValueAll();
			
			// ȤƤեOIDΥեOIDʸƬʬ
			// OIDǤϡͤΥåȤ˳Ǽ
			Iterator itr = dt.keySet().iterator();
			while(itr.hasNext()){
				String fullOid = (String)itr.next();
				
				// ⤷Ƭʬפʤ餽Υĥ꡼λ¹Ƚ
				if(fullOid.startsWith(oid)){
					set.add(getTable(facilityId).getValue(fullOid));
				}
			}
		} catch (FacilityNotFoundExciption e) {
			m_log.debug(e);
		}
		// ֥ĥ꡼˴ޤޤͤʤnull֤
		if(set.size() == 0){
			return null;
		} else {
			return set;
		}
	}
	
	/**
	 * եƥΥΡɤФμֳ֤SNMPݡ󥰤Ԥ褦
	 * ݡ顼ѹޤ
	 * @param facilityId оݥΡɤΥեƥID
	 * @param interval ֳ()
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String facilityId, int interval) {
		// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
		if(this.m_dataTable.containsKey(facilityId) == false){
			this.createTable(facilityId);
		}
		
		// 򹹿
		this.setPollingInterval(facilityId, interval);
	}
	
	/**
	 * եƥΥΡɤФμֳ֤SNMPݡ󥰤Ԥ褦
	 * ݡ顼ѹޤ
	 * @param facilityIds оݥΡɤΥեƥIDΥꥹ
	 * @param interval ֳ()
	 * @param waitTime ݡưξԤ
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String[] facilityIds, int interval, long waitTime) {
		for(int i=0; i<facilityIds.length; i++){
			String facilityId = facilityIds[i];
			
			// ΡɤΥեƥIDбơ֥뤬ʤϿ˺
			if(this.m_dataTable.containsKey(facilityId) == false){
				this.createTable(facilityId);
			}
			
			// 򹹿
			this.setPollingInterval(facilityId, interval);
		}
		
		// QuartzΥ֥塼
		this.m_scheduleFlag = true;
		this.schedule();
		
		// ݡ󥰷̤֤äƤΤλԤ
		try {
			Thread.sleep(waitTime);
		} catch (InterruptedException e) {
			m_log.error(e);
		}
	}

	/**
	 * եƥΥΡɤФμֳ֤SNMPݡ󥰤Ԥ褦
	 * ݡ顼ѹޤ
	 * @param facilityIds оݥΡɤΥեƥIDΥꥹ
	 * @param interval ֳ()
	 * @throws FacilityNotFoundExciption 
	 */
	public void setPollingSchedule(String[] facilityIds, int interval) {
		setPollingSchedule(facilityIds, interval, 0);
	}

	
	public String getCheckInterval() {
		return m_checkInterval;
	}
	
	public void setCheckInterval(String interval) {
		m_checkInterval = interval;
	}
	
	public List getOidList() {
		return m_oidList;
	}
	
	public void setOidList(List oidList) {
		m_oidList = oidList;
	}
	
	public List getTargetNodeList(){
		TreeSet set = new TreeSet(this.m_dataTable.keySet());
		return new ArrayList(set);
	}
	
	public int getDefaultPort() {
		return m_defaultPort;
	}
	
	public void setDefaultPort(int port) {
		m_defaultPort = port;
	}
	
	public String getDefaultCommunity() {
		return m_defaultCommunity;
	}
	
	public void setDefaultCommunity(String community) {
		m_defaultCommunity = community;
	}
	
	public int getDefaultVersion() {
		return m_defaultVersion;
	}
	
	public void setDefaultVersion(int version) {
		m_defaultVersion = version;
	}
	
	public int getDefaultRetries() {
		return m_defaultRetries;
	}
	
	public void setDefaultRetries(int retries) {
		m_defaultRetries = retries;
	}
	
	public int getDefaultTimeout() {
		return m_defaultTimeout;
	}
	
	public void setDdefaultTimeout(int timeout) {
		m_defaultTimeout = timeout;
	}
	
	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}
	
	public void setQuartzJndiName(String jndiName) {
		m_quartzJndiName = jndiName;
	}
}
