/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.dao;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.CommunicationException;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.RoleAttributeConstant;
import com.clustercontrol.accesscontrol.bean.UserAttributeConstant;
import com.clustercontrol.accesscontrol.ejb.entity.RoleBean;
import com.clustercontrol.accesscontrol.ejb.entity.RolePK;
import com.clustercontrol.accesscontrol.util.LdapConnectionManager;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * RoleのDAOインターフェースを実装するクラスです。
 *
 * @version 2.2.0
 * @since 2.0.0
 * 
 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean
 * @see com.clustercontrol.accesscontrol.dao.RoleDAO
 */
public class RoleDAOImpl implements RoleDAO {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( RoleDAOImpl.class );
	/** オブジェクトクラス属性名 */
	protected static final String OBJECTCLASS = "objectClass";
	/** ベースDN */
	protected static final String BASE_DN = "ou=roles";
	/** Roleのオブジェクトクラス属性値 */
	protected static final String OBJECT_CLASS_ROLE = "ccRole";

	public void init() {
		
	}

	/** 
	 * 引数で指定されたプライマリキーでLDAPを検索し、取得した内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#load(com.clustercontrol.accesscontrol.ejb.entity.RolePK, com.clustercontrol.accesscontrol.ejb.entity.RoleBean)
	 */
	public void load(RolePK pk, RoleBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		try {
			clearBean(ejb);
			
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			//属性を取得
			javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
			
			//dn取得
			ejb.setDn(pk.getDn());
			//cn取得
			ejb.setCn((String)attrs.get(UserAttributeConstant.CN).get(0));
			//description取得
			if(attrs.get(RoleAttributeConstant.DESCRIPTION) != null){
				ejb.setDescription((String)attrs.get(RoleAttributeConstant.DESCRIPTION).get(0));
			}
			//ccCreateTimestamp取得
			if(attrs.get(RoleAttributeConstant.CREATETIMESTAMP) != null){
				ejb.setCreateTimestamp(stringToDate((String)attrs.get(RoleAttributeConstant.CREATETIMESTAMP).get(0)));
			}
			//ccCreatorsName取得
			if(attrs.get(RoleAttributeConstant.CREATORSNAME) != null){
				ejb.setCreatorsName((String)attrs.get(RoleAttributeConstant.CREATORSNAME).get(0));
			}
			//ccModifiersName取得
			if(attrs.get(RoleAttributeConstant.MODIFIERSNAME) != null){
				ejb.setModifiersName((String)attrs.get(RoleAttributeConstant.MODIFIERSNAME).get(0));
			}
			//ccModifyTimestamp取得
			if(attrs.get(RoleAttributeConstant.MODIFYTIMESTAMP) != null){
				ejb.setModifyTimestamp(stringToDate((String)attrs.get(RoleAttributeConstant.MODIFYTIMESTAMP).get(0)));
			}
			//member取得
			if(attrs.get(RoleAttributeConstant.MEMBER) != null){
				javax.naming.NamingEnumeration hosts = attrs.get(RoleAttributeConstant.MEMBER).getAll();
				ArrayList<String> memberList = new ArrayList<String>();
				while ( hosts.hasMoreElements() ) {
					String member = (String)hosts.nextElement();
					memberList.add(member);
				}
				ejb.setMember(memberList);
				hosts.close();
			}
		}
		catch ( javax.naming.NamingException ex ) {
		    if(ex instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
			throw new EJBException(ex);
		}
		finally{
			try {
				if(ctx != null){
					ctx.close();
				}
			} catch (NamingException e) {
			}
		}
	}

	/**
	 * 引数で指定されたEntity Beanの内容でLDAPを更新します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#store(com.clustercontrol.accesscontrol.ejb.entity.RoleBean)
	 */
	public void store(RoleBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			//属性を取得
			javax.naming.directory.Attributes attrs = ctx.getAttributes(ejb.getDn());
			
			javax.naming.directory.Attributes addAttrs = 
				new javax.naming.directory.BasicAttributes();
			javax.naming.directory.Attributes replaceAttrs = 
				new javax.naming.directory.BasicAttributes();
			javax.naming.directory.Attributes removeAttrs = 
				new javax.naming.directory.BasicAttributes();
			
			//description設定
			if(attrs.get(RoleAttributeConstant.DESCRIPTION) == null && ejb.getDescription() != null){
				addAttrs.put(RoleAttributeConstant.DESCRIPTION, ejb.getDescription());
			}
			else if(attrs.get(RoleAttributeConstant.DESCRIPTION) != null && ejb.getDescription() == null){
				removeAttrs.put(RoleAttributeConstant.DESCRIPTION, null);
			}
			else if(attrs.get(RoleAttributeConstant.DESCRIPTION) != null && ejb.getDescription() != null){
				replaceAttrs.put(RoleAttributeConstant.DESCRIPTION, ejb.getDescription());
			}
			//ccCreateTimestamp設定
			if(attrs.get(RoleAttributeConstant.CREATETIMESTAMP) == null && ejb.getCreateTimestamp() != null){
				addAttrs.put(RoleAttributeConstant.CREATETIMESTAMP, dateToString(ejb.getCreateTimestamp()));
			}
			else if(attrs.get(RoleAttributeConstant.CREATETIMESTAMP) != null && ejb.getCreateTimestamp() == null){
				removeAttrs.put(RoleAttributeConstant.CREATETIMESTAMP, null);
			}
			else if(attrs.get(RoleAttributeConstant.CREATETIMESTAMP) != null && ejb.getCreateTimestamp() != null){
				replaceAttrs.put(RoleAttributeConstant.CREATETIMESTAMP, dateToString(ejb.getCreateTimestamp()));
			}
			//ccCreatorsName設定
			if(attrs.get(RoleAttributeConstant.CREATORSNAME) == null && ejb.getCreatorsName() != null){
				addAttrs.put(RoleAttributeConstant.CREATORSNAME, ejb.getCreatorsName());
			}
			else if(attrs.get(RoleAttributeConstant.CREATORSNAME) != null && ejb.getCreatorsName() == null){
				removeAttrs.put(RoleAttributeConstant.CREATORSNAME, null);
			}
			else if(attrs.get(RoleAttributeConstant.CREATORSNAME) != null && ejb.getCreatorsName() != null){
				replaceAttrs.put(RoleAttributeConstant.CREATORSNAME, ejb.getCreatorsName());
			}
			//ccModifiersName設定
			if(attrs.get(RoleAttributeConstant.MODIFIERSNAME) == null && ejb.getModifiersName() != null){
				addAttrs.put(RoleAttributeConstant.MODIFIERSNAME, ejb.getModifiersName());
			}
			else if(attrs.get(RoleAttributeConstant.MODIFIERSNAME) != null && ejb.getModifiersName() == null){
				removeAttrs.put(RoleAttributeConstant.MODIFIERSNAME, null);
			}
			else if(attrs.get(RoleAttributeConstant.MODIFIERSNAME) != null && ejb.getModifiersName() != null){
				replaceAttrs.put(RoleAttributeConstant.MODIFIERSNAME, ejb.getModifiersName());
			}
			//ccModifyTimestamp設定
			if(attrs.get(RoleAttributeConstant.MODIFYTIMESTAMP) == null && ejb.getModifyTimestamp() != null){
				addAttrs.put(RoleAttributeConstant.MODIFYTIMESTAMP, dateToString(ejb.getModifyTimestamp()));
			}
			else if(attrs.get(RoleAttributeConstant.MODIFYTIMESTAMP) != null && ejb.getModifyTimestamp() == null){
				removeAttrs.put(RoleAttributeConstant.MODIFYTIMESTAMP, null);
			}
			else if(attrs.get(RoleAttributeConstant.MODIFYTIMESTAMP) != null && ejb.getModifyTimestamp() != null){
				replaceAttrs.put(RoleAttributeConstant.MODIFYTIMESTAMP, dateToString(ejb.getModifyTimestamp()));
			}
			//member設定
			if(attrs.get(RoleAttributeConstant.MEMBER) == null && ejb.getMember() != null){
				javax.naming.directory.Attribute attr =
					new javax.naming.directory.BasicAttribute(RoleAttributeConstant.MEMBER);
				ArrayList list = ejb.getMember();
				for(int i = 0; i < list.size(); i++) {
					String member = (String)list.get(i);
					attr.add(i, member);
				}
				if(attr.size() > 0){
					addAttrs.put(attr);
				}
			}
			else if(attrs.get(RoleAttributeConstant.MEMBER) != null && ejb.getMember() == null){
				removeAttrs.put(RoleAttributeConstant.MEMBER, null);
			}
			else if(attrs.get(RoleAttributeConstant.MEMBER) != null && ejb.getMember() != null){
				javax.naming.directory.Attribute attr =
					new javax.naming.directory.BasicAttribute(RoleAttributeConstant.MEMBER);
				ArrayList list = ejb.getMember();
				for(int i = 0; i < list.size(); i++) {
					String member = (String)list.get(i);
					attr.add(i, member);
				}
				replaceAttrs.put(attr);
			}
			
			//属性を変更する（追加）
			ctx.modifyAttributes(
					ejb.getDn(), 
					javax.naming.directory.DirContext.ADD_ATTRIBUTE, 
					addAttrs);
			//属性を変更する（置換）
			ctx.modifyAttributes(
					ejb.getDn(), 
					javax.naming.directory.DirContext.REPLACE_ATTRIBUTE, 
					replaceAttrs);
			//属性を変更する（削除）
			ctx.modifyAttributes(
					ejb.getDn(), 
					javax.naming.directory.DirContext.REMOVE_ATTRIBUTE, 
					removeAttrs);
		}
		catch ( javax.naming.NamingException ex ) {
		    if(ex instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
			throw new EJBException(ex);
		}
		finally{
			try {
				if(ctx != null){
					ctx.close();
				}
			} catch (NamingException e) {
			}
		}
	}

	/**
	 * 引数で指定されたプライマリキーでLDAPから削除します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#remove(com.clustercontrol.accesscontrol.ejb.entity.RolePK)
	 */
	public void remove(RolePK pk) throws RemoveException, EJBException {
		javax.naming.directory.DirContext ctx = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			//エントリを削除
			ctx.destroySubcontext(pk.getDn());
		}
		catch ( javax.naming.NamingException ex ) {
		    if(ex instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {pk.getDn()};
            apllog.put("SYS", "005", args);
			throw new EJBException(ex);
		}
		finally{
			try {
				if(ctx != null){
					ctx.close();
				}
			} catch (NamingException e) {
			}
		}
	}

	/**
	 * 引数で指定されたEntity Beanの内容をLDAPに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#create(com.clustercontrol.accesscontrol.ejb.entity.RoleBean)
	 */
	public RolePK create(RoleBean ejb) throws CreateException, EJBException {
		RolePK pk = null;
		javax.naming.directory.DirContext ctx = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			// Attributes for new entry
			javax.naming.directory.Attributes attrs =
				new javax.naming.directory.BasicAttributes();
			
			//objectClassを設定
			javax.naming.directory.Attribute attr =
				new javax.naming.directory.BasicAttribute(OBJECTCLASS);
			attr.add(0, OBJECT_CLASS_ROLE);
			attrs.put(attr);
			
			//cnを設定
			attrs.put(RoleAttributeConstant.CN, ejb.getCn());
			// dn文字列作成
			StringBuffer dn = new StringBuffer();
			dn.append(RoleAttributeConstant.CN);
			dn.append("=");
			dn.append(ejb.getCn());
			dn.append(",");
			dn.append(BASE_DN);
			
			//dnを設定
			ejb.setDn(dn.toString());

			//エントリを作成
			ctx.createSubcontext(ejb.getDn(), attrs);

			pk = new RolePK(ejb.getDn());
		}
		catch ( javax.naming.NamingException ex ) {
            AplLogger apllog = new AplLogger("ACC", "acc");
            String[] args = {ejb.getDn()};
            apllog.put("SYS", "004", args);
		    if(ex instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
		    else if(ex instanceof NameAlreadyBoundException){
		    	throw new DuplicateKeyException(ex.getMessage());
            }
			throw new CreateException(ex.getMessage());
		}
		finally{
			try {
				if(ctx != null){
					ctx.close();
				}
			} catch (NamingException e) {
			}
		}
		return pk;
	}

	/**
	 * 全件取得する検索を行います。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean#ejbFindAll()
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		ArrayList<RolePK> ret = new ArrayList<RolePK>();

		m_log.debug("findAll()");
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_ROLE);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			results = ctx.search(BASE_DN, search.toString(), constraints);
			
			if (results == null) {
				m_log.debug("findAll() : No Search Result");
				return ret;
			}

			//検索結果を取得
			while ( results.hasMoreElements() ) {
				javax.naming.directory.SearchResult aResult = 
					(javax.naming.directory.SearchResult)results.nextElement();
				RolePK pk = new RolePK(aResult.getName() + "," + BASE_DN);
				ret.add(pk);
			}
			
		} catch (NamingException e) {
		    if(e instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
		    m_log.debug("findAll() : " + e.getMessage());
		}
		finally{
			try {
				if(results != null)
					results.close();
				if(ctx != null)
					ctx.close();
			} catch (NamingException e) {
			}
		}


		return ret;
	}

	/**
	 * 1件取得する検索を行います。
	 * 引数で指定されたプライマリキーで、LDAPを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean#ejbFindByPrimaryKey(RolePK)
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#findByPrimaryKey(com.clustercontrol.accesscontrol.ejb.entity.RolePK)
	 */
	public RolePK findByPrimaryKey(RolePK pk) throws FinderException {
		javax.naming.directory.DirContext ctx = null;
		try {
			m_log.debug("findByPrimaryKey() : dn = " + pk.getDn());
			
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			//属性を取得
			javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
			attrs.hashCode();
			
			return pk;
		}
		catch ( javax.naming.NamingException ex ) {
		    if(ex instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
		    m_log.debug("findByPrimaryKey() : " + ex.getMessage());
		    String msg = "RoleDAOImpl.findByPrimaryKey() find error : dn=" + pk.getDn() + " not found.";
			throw new FinderException(msg);
		}
		finally{
			try {
				if(ctx != null){
					ctx.close();
				}
			} catch (NamingException e) {
			}
		}
	}

	/**
	 * 1件取得する検索を行います。
	 * 引数で指定されたCNで、LDAPを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean#ejbFindByCn(String)
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#findByCn(java.lang.String)
	 */
	public RolePK findByCn(String cn) throws FinderException {
		m_log.debug("findByCn() : cn = " + cn);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// 名前付きオブジェクトを検索
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_ROLE);
		search.append(")(");
		search.append(RoleAttributeConstant.CN);
		search.append("=");
		search.append(cn);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		RolePK pk = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			results = ctx.search(BASE_DN, search.toString(), constraints);
			
			if (results == null) {
				String msg = "RoleDAOImpl.findByCn() find error : cn=" + cn + " not found.";
				throw new FinderException(msg);
			}
			
			javax.naming.directory.SearchResult aResult = null;
			javax.naming.directory.Attributes attrs = null;
			while ( results.hasMoreElements() ) {
				aResult = (javax.naming.directory.SearchResult)results.nextElement();
				attrs = aResult.getAttributes();
				break;
			}
			
			if (attrs == null) {
				String msg = "RoleDAOImpl.findByCn() find error : cn=" + cn + " not found.";
				throw new FinderException(msg);
			}
			
			pk = new RolePK(aResult.getName() + "," + BASE_DN);
			
		} catch (NamingException e) {
		    if(e instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
//		    m_log.debug("findByCn() : " + e.getMessage());
		    String msg = "RoleDAOImpl.findByCn() find error : cn=" + cn + " not found.";
			throw new FinderException(msg);
		}
		finally{
			try {
				if(results != null)
					results.close();
				if(ctx != null)
					ctx.close();
			} catch (NamingException e) {
			}
		}

		return pk;
	}
	
	/**
	 * 複数件取得する検索を行います。
	 * 引数で指定されたdnがmemberに含まれるRoleを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean#ejbFindByMember(String)
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#findByMember(java.lang.String)
	 */
	public Collection findByMember(String dn) throws FinderException {
		ArrayList<RolePK> ret = new ArrayList<RolePK>();

		m_log.debug("findByMember() : dn = " + dn);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_ROLE);
		search.append(")(");
		search.append(RoleAttributeConstant.MEMBER);
		search.append("=");
		search.append(dn);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			results = ctx.search(BASE_DN, search.toString(), constraints);
			
			if (results == null) {
				m_log.debug("findByMember() : No Search Result : dn = " + dn);
				return ret;
			}

			//検索結果を取得
			while ( results.hasMoreElements() ) {
				javax.naming.directory.SearchResult aResult = 
					(javax.naming.directory.SearchResult)results.nextElement();
				RolePK pk = new RolePK(aResult.getName() + "," + BASE_DN);
				ret.add(pk);
			}
			
		} catch (NamingException e) {
		    if(e instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
		    m_log.debug("findByMember() : " + e.getMessage());
		}
		finally{
			try {
				if(results != null)
					results.close();
				if(ctx != null)
					ctx.close();
			} catch (NamingException e) {
			}
		}

		return ret;
	}
	
	/**
	 * 1件取得する検索を行います。
	 * 引数で指定されたcnが一致し、memberにdnが含まれてるRoleを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.RoleBean#ejbFindByCnAndMenber(String, String)
	 * @see com.clustercontrol.accesscontrol.dao.RoleDAO#findByCnAndMenber(java.lang.String, java.lang.String)
	 */
	public RolePK findByCnAndMenber(String cn, String dn) throws FinderException {
		m_log.debug("findByCnAndMenber() : cn = " + cn + ", dn = " + dn);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// 名前付きオブジェクトを検索
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_ROLE);
		search.append(")(");
		search.append(RoleAttributeConstant.CN);
		search.append("=");
		search.append(cn);
		search.append(")(");
		search.append(RoleAttributeConstant.MEMBER);
		search.append("=");
		search.append(dn);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		RolePK pk = null;
		try {
			// LDAP用のコンテキストの作成
			ctx = LdapConnectionManager.getConnectionManager().getDirContext();
			
			results = ctx.search(BASE_DN, search.toString(), constraints);
			
			if (results == null) {
				String msg = "RoleDAOImpl.findByCnAndMenber() find error : cn=" + cn + ", dn = " + dn + " not found.";
				throw new FinderException(msg);
			}
			
			javax.naming.directory.SearchResult aResult = null;
			javax.naming.directory.Attributes attrs = null;
			while ( results.hasMoreElements() ) {
				aResult = (javax.naming.directory.SearchResult)results.nextElement();
				attrs = aResult.getAttributes();
				break;
			}
			
			if (attrs == null) {
				String msg = "RoleDAOImpl.findByCnAndMenber() find error : cn=" + cn + ", dn = " + dn + " not found.";
				throw new FinderException(msg);
			}
			
			pk = new RolePK(aResult.getName() + "," + BASE_DN);
			
		} catch (NamingException e) {
		    if(e instanceof CommunicationException){
		        LdapConnectionManager.getConnectionManager().setDirContext(null);
		    }
//		    m_log.debug("findByCnAndMenber() : " + e.getMessage());
			String msg = "RoleDAOImpl.findByCnAndMenber() find error : cn=" + cn + ", dn = " + dn + " not found.";
			throw new FinderException(msg);
		}
		finally{
			try {
				if(results != null)
					results.close();
				if(ctx != null)
					ctx.close();
			} catch (NamingException e) {
			}
		}

		return pk;
	}
	
	/**
	 * StringをDateに変換する
	 * 
	 * @param value String
	 * @return Date
	 */
	public Date stringToDate(String value) {
		Date ret = null;
		
		SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss.SSS");

		try {
			ret = formatter.parse(value);
		} catch (ParseException e) {
		}

		return ret;
	}
	
	/**
	 * DateをStringに変換する
	 * 
	 * @param value Date
	 * @return String
	 */
	public String dateToString(Date value) {
		String ret = null;
		
		SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss.SSSZ");

		ret = formatter.format(value);


		return ret;
	}
	
	/**
	 * 引数で渡されたBeanの値をクリアします。
	 * 
	 * @param ejb Role{@link com.clustercontrol.accesscontrol.ejb.entity.RoleBean}
	 */
	protected void clearBean(RoleBean ejb){
	    ejb.setDn(null);
	    ejb.setCn(null);
	    ejb.setMember(null);
	    ejb.setDescription(null);
	    ejb.setCreateTimestamp(null);
	    ejb.setCreatorsName(null);
	    ejb.setModifiersName(null);
	    ejb.setModifyTimestamp(null);
	}
}
