/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.composite;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.util.Messages;

/**
 * スコープ付き監視基本情報コンポジットクラスです。
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「監視項目ID」 テキストボックス（親）</dd>
 *  <dd>「説明」 テキストボックス（親）</dd>
 *  <dd>「スコープ」 テキストボックス</dd>
 *  <dd>「参照」 ボタン</dd>
 * </dl>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class MonitorBasicScopeComposite extends MonitorBasicComposite {
	
	/** スコープ テキストボックス。 */
	protected Text m_textScope = null;
	
	/** 参照 ボタン。 */
	protected Button m_buttonScope = null;
	
	/** 選択されたスコープのファシリティID。 */
	protected String m_facilityId = null;
	
	
	/**
	 * インスタンスを返します。
	 * <p>
	 * 親クラスのコンストラクタを呼び出し、監視基本情報のコンポジットを配置します。
	 * 初期処理を呼び出し、スコープのコンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see com.clustercontrol.monitor.run.composite.MonitorBasicComposite#MonitorBasicComposite(Composite, int)
	 * @see #initialize()
	 */
	public MonitorBasicScopeComposite(Composite parent, int style) {
		super(parent, style);
		
		this.initialize();
	}
	
	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);
		
		/*
		 * スコープ
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("scope") + " : ");
		// テキスト
		this.m_textScope = new Text(this, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 6;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textScope.setLayoutData(gridData);
		// 参照ボタン
		m_buttonScope = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_buttonScope.setLayoutData(gridData);
		m_buttonScope.setText(Messages.getString("refer"));
		m_buttonScope.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				// シェルを取得
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
				
				ScopeTreeDialog dialog = new ScopeTreeDialog(shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem item = dialog.getSelectItem();
					FacilityInfo info = item.getData();
					m_facilityId = info.getFacilityId();
					if (info.getType() == FacilityConstant.TYPE_NODE) {
						m_textScope.setText(info.getFacilityName());
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						m_textScope.setText(path.getPath(item));
					}
				}
			}
		});
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
	}
	
	/**
	 * 引数で指定された監視情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 * 
	 * @see com.clustercontrol.monitor.run.composite.MonitorBasicComposite#setInputData(MonitorInfo)
	 */
	public void setInputData(MonitorInfo info) {
		
		super.setInputData(info);
		
		if(info != null){
			if (info.getScope() != null) {
				this.m_textScope.setText(info.getScope());
			}
			if (info.getFacilityId() != null) {
				this.m_facilityId = info.getFacilityId();
			}
			
		}
	}
	
	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.monitor.run.composite.MonitorBasicComposite#createInputData(MonitorInfo)
	 * @see #setValidateResult(String, String)
	 */
	public ValidateResult createInputData(MonitorInfo info) {
		
		if(info != null){
			
			ValidateResult validateResult = super.createInputData(info);
			
			if(validateResult != null){
				return validateResult;
			}
			else{
				if (this.m_textScope.getText() != null
						&& !"".equals((this.m_textScope.getText()).trim())) {
					info.setFacilityId(this.m_facilityId);
				}
				else{
					return setValidateResult(Messages.getString("message.hinemos.1"), Messages.getString("message.hinemos.3"));
				}
			}
		}
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
		super.setEnabled(enabled);
		this.m_textScope.setEnabled(enabled);
		this.m_buttonScope.setEnabled(enabled);
	}
	
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
}