/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.util;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Hashtable;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.WorkbenchWindow;
import org.eclipse.ui.internal.util.StatusLineContributionItem;
import org.jboss.naming.NamingContextFactory;
import org.jboss.security.auth.callback.UsernamePasswordHandler;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.accesscontrol.dialog.LoginDialog;
import com.clustercontrol.accesscontrol.ejb.session.AccessCheck;
import com.clustercontrol.accesscontrol.ejb.session.AccessCheckHome;

/**
 * ログインマネージャクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class LoginManager {
    public static final String KEY_EJB_URL = "ejbUrl";
    public static final String VALUE_EJB_URL = "jnp://localhost:1099";
    
    public static final String USER_OBJECT_ID = "com.clustercontrol.accesscontrol.user";
    
    public static final String ACTION_SET_ID    = "com.clustercontrol.accesscontrol.ActionSet";
    public static final String ACTION_ID_LOGIN  = "com.clustercontrol.accesscontrol.etc.action.LoginAction";
    public static final String ACTION_ID_LOGOUT = "com.clustercontrol.accesscontrol.etc.action.LogoutAction";
    
    private static LoginManager m_instance = null;
    private NamingContext m_namingcontext = null;
    private LoginContext m_loginContext = null;
    private String m_url = null;
    private String m_uid = null;
    private String m_password = null;

    /**
     * このオブジェクトを取得します。 シングルトン
     * 
     * @return LoginManager ログインマネージャ
     * 
	 * @version 2.0.0
	 * @since 2.0.0
     */
    public static LoginManager getContextManager() {
        if (m_instance == null) {
            m_instance = new LoginManager();
        }
        return m_instance;
    }

    /**
     * コンストラクタ
     * 
   	 * @version 2.0.0
	 * @since 2.0.0
     */
    private LoginManager() {
        try {
           Configuration.setConfiguration(new HinemosClientConfig());
        } catch(Exception e) {
        }
    }
    
    /**
     * ログイン
     * 
     * @param uid
     * @param password
     * @throws NamingException
     * @throws LoginException
     * 
	 * @version 2.0.0
	 * @since 2.0.0
     */
    public synchronized void login(String uid, String password) throws Exception {
    	//ログアウト
    	logout();
    	
    	m_uid = uid;
    	m_password = password;
    	
    	//NamingContext取得
    	m_namingcontext = getContext();
    	
    	try {
        	//ログインチェック
        	AccessCheckHome home = (AccessCheckHome)m_namingcontext.lookup(AccessCheckHome.JNDI_NAME);
        	
			AccessCheck accessCheck = home.create();
			
			accessCheck.checkLogin();
			
			// ステータスバーへユーザIDを登録
			IWorkbench workbench = PlatformUI.getWorkbench();
			WorkbenchWindow workbenchWindow = (WorkbenchWindow)workbench.getActiveWorkbenchWindow();
			IActionBars bars = workbenchWindow.getActionBars();
			IStatusLineManager lineManager = bars.getStatusLineManager();
			
			StatusLineContributionItem statusLine = new StatusLineContributionItem(USER_OBJECT_ID);
			statusLine.setText(Messages.getString("hinemos.user") + " : " + m_uid);
			lineManager.add(statusLine);
			lineManager.update(true);
			
		} catch (Exception e) {
			logout();
			throw e;
		}
    }
    
    /**
     * ログアウト
     * 
     * @throws NamingException
     * 
	 * @version 2.0.0
	 * @since 2.0.0
     */
    public synchronized void logout() throws NamingException {
    	if(m_loginContext instanceof LoginContext){
    		try {
				m_loginContext.logout();
				
				m_uid = null;
				
				// ステータスバーからユーザIDを削除
				IWorkbench workbench = PlatformUI.getWorkbench();
				WorkbenchWindow workbenchWindow = (WorkbenchWindow)workbench.getActiveWorkbenchWindow();
				IActionBars bars = workbenchWindow.getActionBars();
				
				IStatusLineManager lineManager = bars.getStatusLineManager();
				IContributionItem item = lineManager.find(USER_OBJECT_ID);
				lineManager.remove(item);
				lineManager.update(true);
				
			} catch (LoginException e) {
			}
    	}
		m_loginContext = null;
		m_namingcontext = null;
    }
    
    /**
     * ログインチェック
     * 
     * @return
     * 
	 * @version 2.0.0
	 * @since 2.0.0
     */
    public boolean isLogin() {
    	if(m_loginContext instanceof LoginContext){
    		return true;
    	}
    	else{
    		return false;
    	}
    }
    
    /**
     * RepositoryCollectorを取得します。
     * 
     * @return RepositoryCollector
     * 
     * @version 2.0.0
	 * @since 2.0.0
     */
    public synchronized NamingContext getNamingContext() throws NamingException {
    	if(!isLogin()){
    		
			Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
			
    		//ログインダイアログ表示
            LoginDialog dialog = new LoginDialog(shell);
            
            //ダイアログ表示
            if (dialog.open() == IDialogConstants.OK_ID) {
            	try {
    				login(dialog.getUserid(), dialog.getPassword());
    	
    	            // ログイン成功ダイアログを生成
                	MessageDialog.openInformation(
                			null, 
                			Messages.getString("successful"), 
                			Messages.getString("message.accesscontrol.5"));
    			} catch (Exception e) {
                    // ログイン失敗ダイアログを生成
                	MessageDialog.openError(
                			null, 
                			Messages.getString("failed"), 
                			Messages.getString("message.accesscontrol.6"));
    			}
            }
    	}
    	return m_namingcontext;
    }

    /**
     * NamingContextを取得します。
     * 
     * @return
     * @throws NamingException
     * @throws LoginException
     * 
 	 * @version 2.0.0
	 * @since 2.0.0
     */
    @SuppressWarnings("unchecked")
	private NamingContext getContext() throws NamingException, LoginException {
        //接続先URL取得
        m_url = getUrl();

        Hashtable props = new Hashtable();
        props.put(InitialContext.PROVIDER_URL, m_url);
        
    	UsernamePasswordHandler handler = new UsernamePasswordHandler(m_uid, m_password.toCharArray());
		m_loginContext = new LoginContext("hinemosClient", handler);
		m_loginContext.login();
		
        NamingContextFactory ncf = new NamingContextFactory();
        NamingContext namingContext = (NamingContext) ncf
                .getInitialContext(props);
        
        return namingContext;
    }
    
    /**
     * 接続先URLを取得します。
     * 
     * @return String 接続先URL
  	 * @version 2.0.0
	 * @since 2.0.0
     */
    private String getUrl() {

        //リソースストアから接続先URLを取得
        IPreferenceStore store = ClusterControlPlugin.getDefault()
                .getPreferenceStore();
        String url = store.getString(KEY_EJB_URL);
        if (url.compareTo("") == 0) {
            url = VALUE_EJB_URL;
        }

        return url;
    }
    
    /**
	 * HinemosClient用のコンフィギュレーションクラス
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
     */
    class HinemosClientConfig extends Configuration
    {
    	/* (non-Javadoc)
    	 * @see javax.security.auth.login.Configuration#refresh()
    	 */
    	public void refresh() {
    		
    	}
    	
    	/* (non-Javadoc)
    	 * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
    	 */
    	public AppConfigurationEntry[] getAppConfigurationEntry(String name) {
    		AppConfigurationEntry[] entry = null;
    		try {
    			Class[] parameterTypes = {};
    			Method m = getClass().getDeclaredMethod(name, parameterTypes);
    			Object[] args = {};
    			entry = (AppConfigurationEntry[]) m.invoke(this, args);
    		} catch(Exception e) {
    		}
    		return entry;
    	}
    	
    	/**
    	 * Hinemos Client Configuration
    	 * @return
    	 */
    	@SuppressWarnings("unchecked")
    	AppConfigurationEntry[] hinemosClient() {
    		String name = "org.jboss.security.ClientLoginModule";
    		HashMap options = new HashMap();
    		options.put("multi-threaded", "true");
    		options.put("restore-login-identity", "true");
    		AppConfigurationEntry ace = new AppConfigurationEntry(name,
    				AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
    		AppConfigurationEntry[] entry = {ace};
    		return entry;
    	}
    }
}