/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logtransfer.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.ejb.entity.LogTransferFileInfoLocal;
import com.clustercontrol.logtransfer.ejb.entity.LogTransferInfoLocal;
import com.clustercontrol.logtransfer.ejb.entity.LogTransferInfoUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ログ転送設定情報検索クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class SelectLogTransfer {
	
	protected static Log m_log = LogFactory.getLog( SelectLogTransfer.class );
	
	/**
	 * ログ転送設定情報を取得
	 * 
	 * @return notifyInfo
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public LogTransferInfo getLogTransfer(String transferId) throws CreateException, FinderException, NamingException {
		
		LogTransferInfo bean = null;
		try 
		{
			// ログ転送設定情報を取得
			LogTransferInfoLocal logTransfer = LogTransferInfoUtil.getLocalHome().findByPrimaryKey(transferId);
			
			ArrayList<LogTransferFileInfo> logTransferFiles = new ArrayList<LogTransferFileInfo>();
			Collection ct = logTransfer.getLogTransferFileInfo();
			
			// スコープの取得
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repository.getFacilityPath(logTransfer.getFacilityId(), null);
			
			Iterator itr = ct.iterator();
			LogTransferFileInfoLocal local = null;
			while(itr.hasNext()){
				local = (LogTransferFileInfoLocal)itr.next();
				LogTransferFileInfo event = new LogTransferFileInfo(
						transferId,
						local.getFilePath(),
						local.getRunInterval().intValue(),
						local.getExistenceFlg().intValue() ,
						local.getValidFlg().intValue());
				logTransferFiles.add(event);
			}
			
			bean = new LogTransferInfo(
					logTransfer.getTransferId(),
					logTransfer.getDescription(),
					facilityPath,
					logTransfer.getFacilityId(),
					logTransferFiles,
					logTransfer.getValidFlg().intValue(),
					logTransfer.getRegDate(),
					logTransfer.getUpdateDate(),
					logTransfer.getRegUser(),
					logTransfer.getUpdateUser()
			);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			String[] args = { transferId };
			apllog.put("SYS", "004", args);
			m_log.debug("getLogTransfer():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			String[] args = { transferId };
			apllog.put("SYS", "004", args);
			m_log.debug("getLogTransfer():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			String[] args = { transferId };
			apllog.put("SYS", "004", args);
			m_log.debug("getLogTransfer():" + e.getMessage());
			throw e;
		}
		return bean;
	}
	
	/**
	 * ログ転送設定一覧を取得
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getLogTransferList() throws CreateException, FinderException, NamingException {
		
		ArrayList list = null;
		try 
		{
			// ログ転送設定情報一覧を取得
			Collection ct = LogTransferInfoUtil.getLocalHome().findAllOrderByTransferId();
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			apllog.put("SYS", "005");
			m_log.debug("getLogTransferList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			apllog.put("SYS", "005");
			m_log.debug("getLogTransferList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRANSFER", "logTransfer");
			apllog.put("SYS", "005");
			m_log.debug("getLogTransferList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 転送対象ログファイル情報マップを取得
	 * 
	 * @return 転送対象ログファイル情報マップ（キー：転送設定ID）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, ArrayList<LogTransferFileInfo>> getLogTransferFileInfoMap() throws FinderException, NamingException {
		
		HashMap<String, ArrayList<LogTransferFileInfo>> map = new HashMap<String, ArrayList<LogTransferFileInfo>>();
		try 
		{
			// ログ転送設定情報一覧を取得
			Collection ct = LogTransferInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				LogTransferInfoLocal logTransfer = (LogTransferInfoLocal)itr.next();
				
				ArrayList<LogTransferFileInfo> list = new ArrayList<LogTransferFileInfo>();
				Collection ctFile = logTransfer.getLogTransferFileInfo();
				
				Iterator itrFile = ctFile.iterator();
				LogTransferFileInfoLocal local = null;
				String transferId = logTransfer.getTransferId();
				while(itrFile.hasNext()){
					local = (LogTransferFileInfoLocal)itrFile.next();
					
					int validFlg = ValidConstant.TYPE_INVALID;
					if(logTransfer.getValidFlg() == ValidConstant.TYPE_VALID){
						validFlg = local.getValidFlg().intValue();
					}
					
					LogTransferFileInfo event = new LogTransferFileInfo(
							transferId,
							local.getFilePath(),
							local.getRunInterval().intValue(),
							local.getExistenceFlg().intValue(),
							validFlg);
					list.add(event);
				}
				
				map.put(transferId, list);
			}
			
		} catch (FinderException e) {
			m_log.debug("getLogTransferFileInfoMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getLogTransferFileInfoMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
	
	/**
	 * ファシリティID情報マップを取得
	 * 
	 * @return ファシリティID情報マップ（キー：転送設定ID）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public HashMap<String, ArrayList<String>> getFacilityIdsMap() throws CreateException, FinderException, NamingException {
		
		HashMap<String, ArrayList<String>> map = new HashMap<String, ArrayList<String>>();
		try 
		{
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			// ログ転送設定情報一覧を取得
			Collection ct = LogTransferInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				LogTransferInfoLocal logTransfer = (LogTransferInfoLocal)itr.next();
				
				// ログ転送対象のファシリティID一覧取得
				String facilityId = logTransfer.getFacilityId();
				ArrayList<String> facilityIdList = null;
				if(repository.isNode(facilityId)){
					// ノードの場合
					facilityIdList = new ArrayList<String>();
					facilityIdList.add(facilityId);
				}
				else{
					// スコープの場合
					facilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
				}
				
				map.put(logTransfer.getTransferId(), facilityIdList);
			}
		} catch (CreateException e) {
			m_log.debug("getFacilityIdsMap():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getFacilityIdsMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getFacilityIdsMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
	
	/**
	 * ログ転送設定情報をObjectの2次元配列に格納
	 * 
	 * @param ct
	 * @return Objectの2次元配列
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			LogTransferInfoLocal logTransfer = (LogTransferInfoLocal)itr.next();
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(logTransfer.getFacilityId(), null);
			
			ArrayList info = new ArrayList();
			info.add(logTransfer.getTransferId());
			info.add(logTransfer.getDescription());
			info.add(facilityPath);
			info.add(logTransfer.getValidFlg().intValue());
			info.add(logTransfer.getRegUser());
			info.add(logTransfer.getRegDate() == null ? null:new Date(logTransfer.getRegDate().getTime()));
			info.add(logTransfer.getUpdateUser());
			info.add(logTransfer.getUpdateDate() == null ? null:new Date(logTransfer.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
}
