/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */
package com.clustercontrol.monitor.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.monitor.ejb.entity.EventLogBean;
import com.clustercontrol.monitor.ejb.entity.EventLogPK;

/**
 * イベント情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean
 * @see com.clustercontrol.monitor.dao.EventLogDAO
 */
public class EventLogDAOImpl implements EventLogDAO {

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(EventLogDAOImpl.class);

	public void init() {
	}

	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#load(com.clustercontrol.monitor.ejb.entity.EventLogPK, com.clustercontrol.monitor.ejb.entity.EventLogBean)
	 */
	public void load(EventLogPK pk, EventLogBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			String sql = "SELECT * FROM cc_event_log WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setPluginId(res.getString("plugin_id"));
				ejb.setMonitorId(res.getString("monitor_id"));
				ejb.setFacilityId(res.getString("facility_id"));
				ejb.setOutputDate(res.getTimestamp("output_date"));

				ejb.setGenerationDate(res.getTimestamp("generation_date"));
				ejb.setScopeText(res.getString("scope_text"));
				ejb.setApplication(res.getString("application"));
				ejb.setMessageId(res.getString("message_id"));
				ejb.setMessage(res.getString("message"));
				ejb.setMessageOrg(res.getString("message_org"));
				ejb.setPriority(res.getInt("priority"));
				ejb.setConfirmFlg(res.getInt("confirm_flg"));
				ejb.setConfirmDate(res.getTimestamp("confirm_date"));
				ejb.setDuplicationCount(res.getInt("duplication_count"));
				ejb.setInhibitedFlg(res.getInt("inhibited_flg"));

			} else {
				String msg = "EventLog data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#store(com.clustercontrol.monitor.ejb.entity.EventLogBean)
	 */
	public void store(EventLogBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_event_log SET ");
			sql.append("generation_date = ?, ");
			sql.append("scope_text = ?, ");
			sql.append("application = ?, ");
			sql.append("message_id = ?, ");
			sql.append("message = ?, ");
			sql.append("message_org = ?, ");
			sql.append("priority = ?, ");
			sql.append("confirm_flg = ?, ");
			sql.append("confirm_date = ?, ");
			sql.append("duplication_count = ?, ");
			sql.append("inhibited_flg = ?");
			sql.append("WHERE plugin_id = ? AND ");
			sql.append("monitor_id = ? AND ");
			sql.append("facility_id = ? AND ");
			sql.append("output_date = ?");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setTimestamp(1, ejb.getGenerationDate());
			stmt.setString(2, ejb.getScopeText());
			stmt.setString(3, ejb.getApplication());
			stmt.setString(4, ejb.getMessageId());
			stmt.setString(5, ejb.getMessage());
			stmt.setString(6, ejb.getMessageOrg());
			stmt.setInt(7, ejb.getPriority());
			stmt.setInt(8, ejb.getConfirmFlg());
			stmt.setTimestamp(9, ejb.getConfirmDate());
			stmt.setInt(10, ejb.getDuplicationCount());
			stmt.setInt(11, ejb.getInhibitedFlg());
			
			stmt.setString(12, ejb.getPluginId());
			stmt.setString(13, ejb.getMonitorId());
			stmt.setString(14, ejb.getFacilityId());
			stmt.setTimestamp(15, ejb.getOutputDate());

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
						+ ejb.getOutputDate() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getPluginId() + ", "
					+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
					+ ejb.getOutputDate() + " SQLException");
			e.printStackTrace();
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
						+ ejb.getOutputDate() + " SQLException");
				e1.printStackTrace();
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#remove(com.clustercontrol.monitor.ejb.entity.EventLogPK)
	 */
	public void remove(EventLogPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_event_log WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#create(com.clustercontrol.monitor.ejb.entity.EventLogBean)
	 */
	public EventLogPK create(EventLogBean ejb) throws CreateException,
			EJBException {
		m_log.debug("create() start : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate());

		Connection conn = null;
		PreparedStatement stmt = null;
		EventLogPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_event_log (");
			sql.append("plugin_id, ");
			sql.append("monitor_id, ");
			sql.append("facility_id, ");
			sql.append("output_date, ");
			sql.append("generation_date, ");
			sql.append("scope_text, ");
			sql.append("application, ");
			sql.append("message_id, ");
			sql.append("message, ");
			sql.append("message_org, ");
			sql.append("priority, ");
			sql.append("confirm_flg, ");
			sql.append("confirm_date, ");
			sql.append("duplication_count, ");
			sql.append("inhibited_flg )");
			sql.append("values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getPluginId());
			stmt.setString(2, ejb.getMonitorId());
			stmt.setString(3, ejb.getFacilityId());
			stmt.setTimestamp(4, ejb.getOutputDate());
			stmt.setTimestamp(5, ejb.getGenerationDate());
			stmt.setString(6, ejb.getScopeText());
			stmt.setString(7, ejb.getApplication());
			stmt.setString(8, ejb.getMessageId());
			stmt.setString(9, ejb.getMessage());
			stmt.setString(10, ejb.getMessageOrg());
			stmt.setInt(11, ejb.getPriority());
			stmt.setInt(12, ejb.getConfirmFlg());
			stmt.setTimestamp(13, ejb.getConfirmDate());
			stmt.setInt(14, ejb.getDuplicationCount());
			stmt.setInt(15, ejb.getInhibitedFlg());
			
			int row =	stmt.executeUpdate();
			pk = new EventLogPK(
					ejb.getMonitorId(),
					ejb.getPluginId(),
					ejb.getFacilityId(),
					ejb.getOutputDate()
					);
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindAll()
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<EventLogPK> ret = new ArrayList<EventLogPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_event_log";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				EventLogPK pk= new EventLogPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @param pk プライマリキー
	 * @return プライマリキー
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindByPrimaryKey(EventLogPK)
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findByPrimaryKey(EventLogPK)
	 */
	public EventLogPK findByPrimaryKey(EventLogPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			// SQL文の定義
			String sql = "SELECT * FROM cc_event_log WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 重複カウントが最小 かつ 出力日時が最新のデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param monitorId 監視項目ID
	 * @param pluginId プラグインID
	 * @param facilityId ファシリティID
	 * @param generationDate 出力日時
	 * @param inhibitedFlg 抑制フラグ
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindByGenerationDateOrder(String, String, String, Timestamp, Integer)
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findByGenerationDateOrder(String, String, String, Timestamp, Integer)
	 */
	public Collection findByGenerationDateOrder(String monitorId, String pluginId, String facilityId, Timestamp generationDate, Integer inhibitedFlg) throws FinderException {
		m_log.debug("findByGenerationDateOrder() start : ");
		
		ArrayList<EventLogPK> ret = new ArrayList<EventLogPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT * FROM cc_event_log ");
			sql.append("WHERE monitor_id = ? ");
			sql.append("AND plugin_id = ? ");
			sql.append("AND facility_id = ? ");
			sql.append("AND generation_date <= ? ");
			sql.append("AND inhibited_flg = ? ");
			sql.append("ORDER BY generation_date DESC, duplication_count LIMIT 1");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, monitorId);
			stmt.setString(2, pluginId);
			stmt.setString(3, facilityId);
			stmt.setTimestamp(4, generationDate);
			stmt.setInt(5, inhibitedFlg);

			res = stmt.executeQuery();
			
			while(res.next()) {
				EventLogPK pk= new EventLogPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByGenerationDateOrder() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByGenerationDateOrder() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByGenerationDateOrder() end : ");
		return ret;
	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 受信日時順にデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param monitorId 監視項目ID
	 * @param pluginId プラグインID
	 * @param facilityId ファシリティID
	 * @param generationDate 出力日時
	 * @param messageOrg オリジナルメッセージ
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 *
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindByMessageOrg(String, String, String, Timestamp, String)
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findByMessageOrg(String, String, String, Timestamp, String)
	 */
	public Collection findByMessageOrg(String monitorId, String pluginId, String facilityId, Timestamp generationDate, String messageOrg) throws FinderException {
		m_log.debug("selectByMessageOrg() start : ");
		
		// Where句の生成
		StringBuffer sqlWhere = new StringBuffer();

		sqlWhere.append(" plugin_id = ?");
		sqlWhere.append(" AND monitor_id = ?");
		sqlWhere.append(" AND facility_id = ?");
		sqlWhere.append(" AND generation_date = ?");
		
		if(messageOrg != null){
			sqlWhere.append(" AND message_org = ?");
		}
		else{
			sqlWhere.append(" AND message_org is null ");
		}
		
		ArrayList<EventLogPK> ret = new ArrayList<EventLogPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT * FROM cc_event_log ");
			sql.append("WHERE " + sqlWhere.toString());
			sql.append(" ORDER BY output_date");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, pluginId);
			stmt.setString(2, monitorId);
			stmt.setString(3, facilityId);
			stmt.setTimestamp(4, generationDate);
			
			if(messageOrg != null){
				stmt.setString(5, messageOrg);
			}
			res = stmt.executeQuery();
			
			while(res.next()) {
				EventLogPK pk= new EventLogPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("selectByMessageOrg() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("selectByMessageOrg() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("selectByMessageOrg() end : ");
		return ret;

	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 *
	 * @param facilityId ファシリティIDの配列
	 * @param priority 重要度
	 * @param outputFromDate 受信日時（開始）
	 * @param outputToDate 受信日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @param confirmFlg 確認フラグ（未／済）
	 * @param orderByFlg 受信日時の順序（true：昇順、false：降順）
	 * @param limit 問い合わせ限度数
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 *
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 */
	@SuppressWarnings("unchecked")
	public Collection findEvent(
			String[] facilityId, 
			Integer priority, 
			Timestamp outputFromDate, 
			Timestamp outputToDate, 
			Timestamp generationFromDate, 
			Timestamp generationToDate, 
			String application, 
			String message, 
			Integer confirmFlg, 
			boolean orderByFlg,
			Integer limit) throws FinderException {
		m_log.debug("selectEvent() start : ");
		
		// Where句の生成
		StringBuffer sqlWhere = new StringBuffer();
		boolean whereFlg = false;
		
		// ファシリティID設定
		if(facilityId != null && facilityId.length>0) {
			sqlWhere.append("facility_id IN (");
			for(int index=0; index<facilityId.length; index++){
				if(index != 0)
					sqlWhere.append(",");
				sqlWhere.append("?");
			}
			sqlWhere.append(")");
			whereFlg = true;
		}
		
		// 重要度設定
		if(priority != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " priority = ?");
			whereFlg = true;
		}
		// 受信日時（自）設定
		if(outputFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date >= ?");
			whereFlg = true;
		}
		// 受信日時（至）設定
		if(outputToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date <= ?");
			whereFlg = true;
		}
		// 出力日時（自）設定
		if(generationFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date >= ?");
			whereFlg = true;
		}
		// 出力日時（至）設定
		if(generationToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date <= ?");
			whereFlg = true;
		}
		// アプリケーション設定
		if(application != null && !"".equals(application)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " application like ?");
			whereFlg = true;
		}
		// メッセージ設定
		if(message != null && !"".equals(message)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " message like ?");
			whereFlg = true;
		}
		// 確認有無
		if(confirmFlg != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " confirm_flg = ?");
			whereFlg = true;
		}
		
		ArrayList<EventLogPK> ret = new ArrayList<EventLogPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT * FROM cc_event_log ");
			if(sqlWhere.length() > 0){
				sql.append("WHERE " + sqlWhere.toString());
			}
			if(orderByFlg){
				sql.append(" ORDER BY output_date");
			}
			else{
				sql.append(" ORDER BY output_date DESC");
			}
			if(limit != null){
				sql.append(" LIMIT " + limit.toString());
			}
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			int count = 1;
			if(sqlWhere.length() > 0){
				// ファシリティID設定
				if(facilityId != null && facilityId.length>0) {
					for(int index=0; index<facilityId.length; index++){
						stmt.setString(count, facilityId[index]);
						count++;
					}
				}
				// 重要度設定
				if(priority != null) {
					stmt.setInt(count, priority);
					count++;
				}
				// 受信日時（自）設定
				if(outputFromDate != null) {
					stmt.setTimestamp(count, outputFromDate);
					count++;
				}
				// 受信日時（至）設定
				if(outputToDate != null) {
					stmt.setTimestamp(count, outputToDate);
					count++;
				}
				// 出力日時（自）設定
				if(generationFromDate != null) {
					stmt.setTimestamp(count, generationFromDate);
					count++;
				}
				// 出力日時（至）設定
				if(generationToDate != null) {
					stmt.setTimestamp(count, generationToDate);
					count++;
				}
				// アプリケーション設定
				if(application != null && !"".equals(application)) {
					stmt.setString(count, "%" + application + "%");
					count++;
				}
				// メッセージ設定
				if(message != null && !"".equals(message)) {
					stmt.setString(count, "%" + message + "%");
					count++;
				}
				// 確認有無
				if(confirmFlg != null) {
					stmt.setInt(count, confirmFlg);
					count++;
				}
			}
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				EventLogPK pk= new EventLogPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("selectEvent() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("selectEvent() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("selectEvent() end : ");
		return ret;
	}
	
	/**
	 * 引数で指定された値でSELECT文を発行し、取得した件数を返します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param priority 重要度
	 * @param outputFromDate 受信日時（開始）
	 * @param outputToDate 受信日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @param confirmFlg 確認フラグ（未／済）
	 * @return 件数
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeCountEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 */
	@SuppressWarnings("unchecked")
	public int countEvent(
			String[] facilityId, 
			Integer priority, 
			Timestamp outputFromDate, 
			Timestamp outputToDate, 
			Timestamp generationFromDate, 
			Timestamp generationToDate, 
			String application, 
			String message, 
			Integer confirmFlg) throws SQLException {
		m_log.debug("countEvent() start : ");
		
		// Where句の生成
		StringBuffer sqlWhere = new StringBuffer();
		boolean whereFlg = false;
		int count = 0;
		
		// ファシリティID設定
		if(facilityId != null && facilityId.length>0) {
			sqlWhere.append("facility_id IN (");
			for(int index=0; index<facilityId.length; index++){
				if(index != 0)
					sqlWhere.append(",");
				sqlWhere.append("?");
			}
			sqlWhere.append(")");
			whereFlg = true;
		}
		
		// 重要度設定
		if(priority != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " priority = ?");
			whereFlg = true;
		}
		// 受信日時（自）設定
		if(outputFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date >= ?");
			whereFlg = true;
		}
		// 受信日時（至）設定
		if(outputToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date <= ?");
			whereFlg = true;
		}
		// 出力日時（自）設定
		if(generationFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date >= ?");
			whereFlg = true;
		}
		// 出力日時（至）設定
		if(generationToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date <= ?");
			whereFlg = true;
		}
		// アプリケーション設定
		if(application != null && !"".equals(application)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " application like ?");
			whereFlg = true;
		}
		// メッセージ設定
		if(message != null && !"".equals(message)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " message like ?");
			whereFlg = true;
		}
		// 確認有無
		if(confirmFlg != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " confirm_flg = ?");
			whereFlg = true;
		}

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT COUNT(*) FROM cc_event_log ");
			if(sqlWhere.length() > 0){
				sql.append("WHERE " + sqlWhere.toString());
			}
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			int i = 1;
			if(sqlWhere.length() > 0){
				// ファシリティID設定
				if(facilityId != null && facilityId.length>0) {
					for(int index=0; index<facilityId.length; index++){
						stmt.setString(i, facilityId[index]);
						i++;
					}
				}
				// 重要度設定
				if(priority != null) {
					stmt.setInt(i, priority);
					i++;
				}
				// 受信日時（自）設定
				if(outputFromDate != null) {
					stmt.setTimestamp(i, outputFromDate);
					i++;
				}
				// 受信日時（至）設定
				if(outputToDate != null) {
					stmt.setTimestamp(i, outputToDate);
					i++;
				}
				// 出力日時（自）設定
				if(generationFromDate != null) {
					stmt.setTimestamp(i, generationFromDate);
					i++;
				}
				// 出力日時（至）設定
				if(generationToDate != null) {
					stmt.setTimestamp(i, generationToDate);
					i++;
				}
				// アプリケーション設定
				if(application != null && !"".equals(application)) {
					stmt.setString(i, "%" + application + "%");
					i++;
				}
				// メッセージ設定
				if(message != null && !"".equals(message)) {
					stmt.setString(i, "%" + message + "%");
					i++;
				}
				// 確認有無
				if(confirmFlg != null) {
					stmt.setInt(i, confirmFlg);
					i++;
				}
			}
			
			res = stmt.executeQuery();
			
			if(res.next()) {
				count = res.getInt(1);
			}
			
		} catch (SQLException e) {
			m_log.error("countEvent() error :  SQLException");
			throw e;
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("countEvent() error :  SQLException");
				throw e1;
			}
		}
		m_log.debug("countEvent() end : ");
		return count;
	}

	/**
	 * 引数で指定された値でUPDATE文を発行し、更新した件数を返します。<BR>
	 * 確認フラグを更新し、確認済みの場合は確認済み日時を更新します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param priority 重要度
	 * @param outputFromDate 受信日時（開始）
	 * @param outputToDate 受信日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @param beforConfirmFlg 更新前確認フラグ（未／済）
	 * @param afterConfirmFlg 更新後確認フラグ（未／済）
	 * @return 更新数
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeBatchConfirm(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, Integer)
	 */
	public int confirmEvent(String[] facilityId, Integer priority,
			Timestamp outputFromDate, Timestamp outputToDate,
			Timestamp generationFromDate, Timestamp generationToDate,
			String application, String message, Integer beforConfirmFlg,
			Integer afterConfirmFlg)
			throws SQLException {
		m_log.debug("confirmEvent() start : ");
		
		// Where句の生成
		StringBuffer sqlWhere = new StringBuffer();
		boolean whereFlg = false;
		
		// ファシリティID設定
		if(facilityId != null && facilityId.length>0) {
			sqlWhere.append("facility_id IN (");
			for(int index=0; index<facilityId.length; index++){
				if(index != 0)
					sqlWhere.append(",");
				sqlWhere.append("?");
			}
			sqlWhere.append(")");
			whereFlg = true;
		}
		
		// 重要度設定
		if(priority != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " priority = ?");
			whereFlg = true;
		}
		// 受信日時（自）設定
		if(outputFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date >= ?");
			whereFlg = true;
		}
		// 受信日時（至）設定
		if(outputToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " output_date <= ?");
			whereFlg = true;
		}
		// 出力日時（自）設定
		if(generationFromDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date >= ?");
			whereFlg = true;
		}
		// 出力日時（至）設定
		if(generationToDate != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " generation_date <= ?");
			whereFlg = true;
		}
		// アプリケーション設定
		if(application != null && !"".equals(application)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " application like ?");
			whereFlg = true;
		}
		// メッセージ設定
		if(message != null && !"".equals(message)) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " message like ?");
			whereFlg = true;
		}
		// 確認有無
		if(beforConfirmFlg != null) {
			sqlWhere.append(this.checkAndLogicalOperator(whereFlg) + " confirm_flg = ?");
			whereFlg = true;
		}
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		int row;  // update された行の数
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			// 確認フラグと確認時刻を更新する
			sql.append("UPDATE cc_event_log SET confirm_flg = ?, confirm_date = ? ");
			if(sqlWhere.length() > 0){
				sql.append("WHERE " + sqlWhere.toString());
			}
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			int count = 1;
			
			// 確認フラグの更新
			stmt.setInt(count++, afterConfirmFlg);
			
			// 確認時刻の更新
			stmt.setTimestamp(count++, new Timestamp(System.currentTimeMillis()));
			
			if(sqlWhere.length() > 0){
				// ファシリティID設定
				if(facilityId != null && facilityId.length>0) {
					for(int index=0; index<facilityId.length; index++){
						stmt.setString(count, facilityId[index]);
						count++;
					}
				}
				// 重要度設定
				if(priority != null) {
					stmt.setInt(count, priority);
					count++;
				}
				// 受信日時（自）設定
				if(outputFromDate != null) {
					stmt.setTimestamp(count, outputFromDate);
					count++;
				}
				// 受信日時（至）設定
				if(outputToDate != null) {
					stmt.setTimestamp(count, outputToDate);
					count++;
				}
				// 出力日時（自）設定
				if(generationFromDate != null) {
					stmt.setTimestamp(count, generationFromDate);
					count++;
				}
				// 出力日時（至）設定
				if(generationToDate != null) {
					stmt.setTimestamp(count, generationToDate);
					count++;
				}
				// アプリケーション設定
				if(application != null && !"".equals(application)) {
					stmt.setString(count, "%" + application + "%");
					count++;
				}
				// メッセージ設定
				if(message != null && !"".equals(message)) {
					stmt.setString(count, "%" + message + "%");
					count++;
				}
				// 確認有無
				if(beforConfirmFlg != null) {
					stmt.setInt(count, beforConfirmFlg);
					count++;
				}
			}
			
			// クエリを実行
			row = stmt.executeUpdate();
		} catch (SQLException e) {
			m_log.error("confirmEvent() error :  SQLException");
			m_log.error(e.getMessage());
			throw e;
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("confirmEvent() error :  SQLException");
				m_log.error(e1.getMessage());
				throw e1;
			}
		}
		m_log.debug("confirmEvent() end : ");
		return row;
	}

	/**
	 * 引数で指定された値でSELECT文を発行します。<BR>
	 * 重要度が最高 かつ 受信日時が最新のデータを検索します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @param facilityId ファシリティIDの配列
	 * @param priority 重要度
	 * @param outputFromDate 受信日時（開始）
	 * @param outputToDate 受信日時（終了）
	 * @param generationFromDate 出力日時（開始）
	 * @param generationToDate 出力日時（終了）
	 * @param application アプリケーション
	 * @param message メッセージ
	 * @param confirmFlg 確認フラグ（未／済）
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.bean.PriorityConstant
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindHighPriorityEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 * @see com.clustercontrol.monitor.dao.EventLogDAO#findHighPriorityEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 */
	public Collection findHighPriorityEvent(String[] facilityId, Integer priority, Timestamp outputFromDate, Timestamp outputToDate, Timestamp generationFromDate, Timestamp generationToDate, String application, String message, Integer confirmFlg) throws FinderException {
		m_log.debug("selectHighPriorityEvent() start : ");
		
		ArrayList<EventLogPK> ret = new ArrayList<EventLogPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			StringBuffer sql = new StringBuffer();
			sql.append("SELECT * FROM cc_event_log ");
			sql.append("WHERE ");
			
			// ファシリティID設定
			sql.append("facility_id IN (");
			for(int index=0; index<facilityId.length; index++){
				if(index != 0)
					sql.append(",");
				sql.append("?");
			}
			sql.append(")");
			
			// 重要度設定
			if(priority != null) {
				sql.append(" and priority = ?");
			}
			// 受信日時（自）設定
			if(outputFromDate != null) {
				sql.append(" and output_date >= ?");
			}
			// 受信日時（至）設定
			if(outputToDate != null) {
				sql.append("and output_date <= ?");
			}
			// 出力日時（自）設定
			if(generationFromDate != null) {
				sql.append(" and generation_date >= ?");
			}
			// 出力日時（至）設定
			if(generationToDate != null) {
				sql.append(" and generation_date <= ?");
			}
			// アプリケーション設定
			if(application != null && !"".equals(application)) {
				sql.append(" and application like ?");
			}
			// メッセージ設定
			if(message != null && !"".equals(message)) {
				sql.append(" and message like ?");
			}
			// 確認有無
			if(confirmFlg != null) {
				sql.append(" and confirm_flg = ?");
			}
			
			sql.append(" ORDER BY priority, output_date DESC LIMIT 1");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			int count = 1;
			// ファシリティID設定
			if(facilityId != null && facilityId.length>0) {
				for(int index=0; index<facilityId.length; index++){
					stmt.setString(count, facilityId[index]);
					count++;
				}
			}
			// 重要度設定
			if(priority != null) {
				stmt.setInt(count, priority);
				count++;
			}
			// 受信日時（自）設定
			if(outputFromDate != null) {
				stmt.setTimestamp(count, outputFromDate);
				count++;
			}
			// 受信日時（至）設定
			if(outputToDate != null) {
				stmt.setTimestamp(count, outputToDate);
				count++;
			}
			// 出力日時（自）設定
			if(generationFromDate != null) {
				stmt.setTimestamp(count, generationFromDate);
				count++;
			}
			// 出力日時（至）設定
			if(generationToDate != null) {
				stmt.setTimestamp(count, generationToDate);
				count++;
			}
			// アプリケーション設定
			if(application != null && !"".equals(application)) {
				stmt.setString(count, "%" + application + "%");
				count++;
			}
			// メッセージ設定
			if(message != null && !"".equals(message)) {
				stmt.setString(count, "%" + message + "%");
				count++;
			}
			// 確認有無
			if(confirmFlg != null) {
				stmt.setInt(count, confirmFlg);
				count++;
			}
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				EventLogPK pk= new EventLogPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("selectHighPriorityEvent() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("selectHighPriorityEvent() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("selectHighPriorityEvent() start : ");
		return ret;
	}
	
	/**
	 * AND論理演算子
	 * 
	 * @param flag
	 * @return 論理演算子
	 */
	private String checkAndLogicalOperator(boolean flag) {
		
		if(flag){
			return " and ";
		}
		else{
			return "";
		}
	}
}
