/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.quartz.Trigger;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.QuartzConstant;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;

/**
 * スケジュールを登録するクラスです。
 *
 * @version 2.1.0
 * @since 2.0.0
 */
public class ModifySchedule {

	/** ログ出力のインスタンス。 */
    protected static Log m_log = LogFactory.getLog( ModifySchedule.class );

    /**
	 * Quartzに監視情報のスケジュールと有効/無効を登録します。
	 * <p>
	 * <ol>
	 * <li>監視対象IDと監視項目IDを指定し、監視を実行するメソッドのジョブを作成します。</li>
	 * <li>呼びだすメソッドの引数として、下記項目をセットします。</li>
	 * <dl>
	 *  <dt>引数</dt>
	 *  <dd>監視対象ID</dd>
	 *  <dd>監視項目ID</dd>
	 *  <dd>監視種別</dd>
	 * </dl>
	 * <li>監視情報の有効/無効を、セットします。</li>
	 * <li>Quartzにジョブとスケジュールを登録します。</li>
	 * </ol>
	 *  
     * @param info 監視情報
     * @param jndiName JNDI名
     * @param schedule スケジュールカレンダ
     * @throws ParseException
     * @throws NamingException
     * @throws SchedulerException
     * @throws RemoteException 
     * @throws CreateException 
     * @since 2.0.0
     * 
     * @see #getCronString(Calendar)
     * @see com.clustercontrol.monitor.run.bean.QuartzConstant
     * @see com.clustercontrol.commons.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see com.clustercontrol.bean.ValidConstant
     */
    public void addSchedule(MonitorInfo info, String jndiName, Calendar schedule) throws ParseException, NamingException, SchedulerException {
    	
    	String monitorId = info.getMonitorId();
    	String monitorTypeId = info.getMonitorTypeId();
    	
        //JobDetail作成
	    JobDetail job = new JobDetail(
	    		monitorId, 
	    		monitorTypeId, 
	            EJBInvokerJob.class);
	    
	    //ジョブ完了時に削除されないようにする。
	    job.setDurability(true);
	    //ジョブ実行失敗時に再実行するようにする。
	    job.setRequestsRecovery(true);
	    
	    //JobDetailに呼び出すクラスとメソッドを設定
	    job.getJobDataMap().put(
	            EJBInvokerJob.EJB_JNDI_NAME_KEY, 
	            jndiName);
	    job.getJobDataMap().put(
	            EJBInvokerJob.EJB_METHOD_KEY, 
	            QuartzConstant.METHOD_NAME);
	    
	    String serverName = System.getProperty("jboss.server.name");
	    if(serverName.equals("all")){
	        job.getJobDataMap().put(
	                EJBInvokerJob.PROVIDER_URL, 
	                "jnp://localhost:1100");
	    }

	    //JobDetailに呼び出すメソッドの引数を設定
	    // 監視対象IDを設定
	    Object[] jdArgs = new Object[3];
	    Class[] jdArgsType = new Class[3];
	    jdArgs[0] = monitorTypeId;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[0] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    // 監視項目IDを設定
	    jdArgs[1] = monitorId;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[1] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    // 監視判定タイプを設定
	    jdArgs[2] = Integer.valueOf(info.getMonitorType());
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[2] = Integer.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    
	    //JobDetailに有効/無効を設定
	    if(info.getValid() == ValidConstant.TYPE_VALID){
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_VALID));
	    }
	    else{
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_INVALID));
	    }
	    
	    Trigger trigger = null;
	    if(schedule == null){
	    	int interval = info.getRunInterval();
	    	trigger = new SimpleTrigger(monitorId, monitorTypeId, new Date(), new Date(Long.MAX_VALUE) ,SimpleTrigger.REPEAT_INDEFINITELY, interval*1000);
	    }
	    else{
		    //CronTriggerを作成
	    	trigger = new CronTrigger(monitorId, monitorTypeId);
		    //スケジュールを設定
	    	((CronTrigger)trigger).setCronExpression(getCronString(schedule));
	    }
        
        //QuartzのSchedulerをルックアップ
	    QuartzManager manager = QuartzUtil.getQuartzManager();

        //Schedulerにジョブを登録する
        try {
			manager.addSchedule(job, trigger);
		} catch (RemoteException e) {
		}
        
        //無効の場合、ジョブ登録後にポーズ
	    if(info.getValid() == ValidConstant.TYPE_INVALID){
	    	try {
				manager.pauseSchedule(job.getName(), monitorTypeId);
			} catch (RemoteException e) {
			}
	    }
    }
    
	/**
	 * 引数で指定された監視情報のQuartzのスケジュールと有効/無効を変更します。
	 * <p>
	 * <ol>
	 * <li>Quartzから監視情報を削除します。</li>
	 * <li>Quartzに監視情報を登録します。</li>
	 * </ol>
	 * 
     * @param info 監視情報
     * @param schedule スケジュールカレンダ
     * @throws NamingException
     * @throws SchedulerException
     * @throws ParseException
	 * @throws RemoteException 
	 * @throws CreateException 
     * @since 2.0.0
     * 
     * @see com.clustercontrol.monitor.run.bean.QuartzConstant
     * @see com.clustercontrol.commons.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see #deleteSchedule(java.lang.String, java.lang.String)
     */
    public void modifySchedule(MonitorInfo info, Calendar schedule) throws NamingException, SchedulerException, ParseException {
    	
    	String monitorId = info.getMonitorId();
    	String monitorTypeId = info.getMonitorTypeId();
        
        //QuartzのSchedulerをルックアップ
    	QuartzManager manager = QuartzUtil.getQuartzManager();
        
        //JobDetail取得
        JobDetail job = null;
		try {
			job = manager.getJobDetail(monitorId, monitorTypeId);
		} catch (RemoteException e) {
		}
        
	    //ジョブ完了時に削除されないようにする。
	    job.setDurability(true);
	    //ジョブ実行失敗時に再実行するようにする。
	    job.setRequestsRecovery(true);
	    
	    //JobDetailに呼び出すメソッドの引数を設定
	    // 監視対象IDを設定
	    Object[] jdArgs = new Object[3];
	    Class[] jdArgsType = new Class[3];
	    jdArgs[0] = monitorTypeId;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[0] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    // 監視項目IDを設定
	    jdArgs[1] = monitorId;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[1] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    // 監視判定タイプを設定
	    jdArgs[2] = Integer.valueOf(info.getMonitorType());
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[2] = Integer.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    
	    //JobDetailに有効/無効を設定
	    if(info.getValid() == ValidConstant.TYPE_VALID){
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_VALID));
	    }
	    else{
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_INVALID));
	    }
	    
	    Trigger trigger = null;
	    if(schedule == null){
	    	int interval = info.getRunInterval();
	    	trigger = new SimpleTrigger(monitorId, monitorTypeId, new Date(), new Date(Long.MAX_VALUE) ,SimpleTrigger.REPEAT_INDEFINITELY, interval*1000);
	    }
	    else{
		    //CronTriggerを作成
	    	trigger = new CronTrigger(monitorId, monitorTypeId);
		    //スケジュールを設定
	    	((CronTrigger)trigger).setCronExpression(getCronString(schedule));
	    }
        
        //スケジュールを削除
        deleteSchedule(monitorTypeId, monitorId);
        
        //Schedulerにジョブを登録する
        try {
			manager.addSchedule(job, trigger);
		} catch (RemoteException e) {
		}
        
        //無効の場合、ジョブ登録後にポーズ
	    if(info.getValid() == ValidConstant.TYPE_INVALID){
	    	try {
				manager.pauseSchedule(job.getName(), job.getGroup());
			} catch (RemoteException e) {
			}
	    }
    }
    
    /**
     * 引数で指定された監視情報をQuartzから削除します。
     * 
     * @param monitorTypeId 監視対象ID
     * @param monitorId 監視項目ID
     * @throws NamingException
     * @throws SchedulerException
     * @throws RemoteException
     * @throws CreateException
     * 
     * @see com.clustercontrol.monitor.run.bean.QuartzConstant
     * @see com.clustercontrol.commons.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     */
    public void deleteSchedule(String monitorTypeId, String monitorId) throws NamingException, SchedulerException {
        m_log.debug("deleteSchedule() : type =" + monitorTypeId + ", id=" + monitorId);
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //ジョブ削除
		try {
			manager.deleteSchedule(monitorId, monitorTypeId);
		} catch (RemoteException e) {
		}
    }
    
    /**
     * 引数で指定されたスケージュールカレンダのCron形式のスケージュール定義を返します。
     * 
     * @param schedule スケジュールカレンダ
     * @return Cron形式スケジュール
     */
    protected String getCronString(Calendar schedule){
        m_log.debug("getCronString()");
        
        if(schedule != null){
        	
        	// cron形式でスケジュールを作成する
            StringBuffer cron = new StringBuffer();
            if(schedule.get(Calendar.HOUR_OF_DAY )> 0){
                cron.append(schedule.get(Calendar.SECOND));
                cron.append(" ");
                cron.append(schedule.get(Calendar.MINUTE));
                cron.append(" ");
                cron.append("*/" + schedule.get(Calendar.HOUR_OF_DAY));
                cron.append(" * * ? *");
            }
            else if(schedule.get(Calendar.MINUTE) > 0){
                cron.append(schedule.get(Calendar.SECOND));
                cron.append(" ");
                cron.append("*/" + schedule.get(Calendar.MINUTE));
                cron.append(" * * * ? *");
            }
            else{
                cron.append("*/" + schedule.get(Calendar.SECOND));
                cron.append(" * * * * ? *");
            }
            return cron.toString();
        }
        return null;
	}
}
