/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を作成するクラスです。
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class AddNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( AddNotify.class );
	
	/**
	 * 通知情報を作成します。
	 * <p>
	 * <ol>
	 *  <li>通知情報を作成します。</li>
	 *  <li>通知イベント情報を作成し、通知情報に設定します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュに追加されます。</li>
	 * </ol>
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext#NotifyInfoContext(int, java.lang.String, com.clustercontrol.notify.bean.NotifyInfo)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	@SuppressWarnings("unchecked")
	public boolean add(NotifyInfo info) throws CreateException, NamingException, JMSException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.NOTIFY);
    	
		SendQueue queue = null;
		try 
		{
			Timestamp now = new Timestamp(new Date().getTime());
			
			// 通知情報を挿入
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().create(
					info.getNotifyId(),
					info.getDescription(),
					info.getStatusFlg(),
					info.getStatusValidPeriod(),
					info.getStatusInvalidFlg(),
					info.getStatusUpdatePriority(),
					info.getEventFlg(),
					info.getInhibitionFlg(),
					info.getInhibitionFrequency(),
					info.getInhibitionPeriod(),
					now,
					now
			);
			Collection ct = notify.getNotifyEventInfo();
			
			ArrayList<NotifyEventInfo> eventList = info.getNotifyEventInfo();
			ArrayList<NotifyEventInfoLocal> eventLocalList = new ArrayList<NotifyEventInfoLocal>();
			
			// 通知イベント情報を挿入
			NotifyEventInfo event = null;
			for(int index=0; index<eventList.size(); index++){
				event = (NotifyEventInfo)eventList.get(index);
				if(event != null){
					NotifyEventInfoLocal local = NotifyEventInfoUtil.getLocalHome().create(
							info.getNotifyId(),
							event.getPriority(),
							event.getEventNormalFlg(),
							event.getEventNormalState(),
							event.getEventInhibitionFlg(),
							event.getEventInhibitionState(),
							event.getMailFlg(),
							event.getMailAddress());
					eventLocalList.add(local);
				}
			}
			if(eventLocalList != null){
				ct.addAll(eventLocalList);
				notify.setNotifyEventInfo(ct);
			}
			
			NotifyInfoContext context = new NotifyInfoContext(NotifyInfoContext.TYPE_ADD, info.getNotifyId(), info);
			
			// 通知情報キャッシュを更新
			queue = new SendQueue(QueueConstant.QUEUE_NAME_LOG);
			queue.put(context);
			queue.terminate();
			
		} catch (CreateException e) {
            AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		} catch (JMSException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		}
		finally{
			if(queue != null){
				queue.terminate();	
			}
		}
		
		return true;
	}
	
}
