/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performanceMGR.util.code;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.TreeSet;

import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.bean.DeviceData;
import com.clustercontrol.performanceMGR.dao.DeviceDAO;
import com.clustercontrol.performanceMGR.util.Messages;
import com.clustercontrol.performanceMGR.util.Repository;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class CollectorItemCodeTable {
	private static final ArrayList<CollectorItemCategoryInfo> m_categoryList; // カテゴリのリスト
	private static final ArrayList<CollectorItemTreeItem> m_itemTree; // 収集項目のツリー情報
	private static final HashMap<String, CollectorItemTreeItem> m_codeTable; // 収集項目コードがキー
	
	public CollectorItemCodeTable() {
	}
	
	static {
		m_categoryList = new ArrayList<CollectorItemCategoryInfo>();
		m_itemTree = new ArrayList<CollectorItemTreeItem>();
		m_codeTable = new HashMap<String, CollectorItemTreeItem>();
		
		// カテゴリコードからカテゴリ情報を参照するためのテーブル
		HashMap<String, CollectorItemTreeItem> categoryTable = 
			new HashMap<String, CollectorItemTreeItem>();
		
		// 設定ファイルを読み込む
		Properties properties = new Properties();
		
		try {
			// デフォルトロケールを取得
			Locale locale = Locale.getDefault();
			
			String homeDir = System.getProperty("jboss.server.home.dir");
			String confDir = homeDir + File.separator + "conf" + File.separator;
			String propertyFile;
			// 言語が日本語の場合
			if(locale.getLanguage().equals("ja")){
				propertyFile = confDir + "performance_ja.properties";
			} else { // 日本語以外の場合
				propertyFile = confDir + "performance.properties";
			}
			
			// プロパティファイルからキーと値のリストを読み込みます
			properties.load(new FileInputStream(propertyFile));
			
			// カテゴリ情報の読み込み
			int num = Integer.parseInt(properties.getProperty("collector.category.num"));
			for(int i=0; i < num; i++){
				String categoryCode = properties.getProperty("collector.category.code." + i);
				String categoryName = properties.getProperty("collector.category.name." + i);
				if(categoryCode != null && categoryName != null){
					CollectorItemCategoryInfo category = new CollectorItemCategoryInfo(categoryCode, categoryName);
					m_categoryList.add(category);
					
					CollectorItemTreeItem categoryItem = new CollectorItemTreeItem(null, category, null);  // 親の要素はないためnull
					m_itemTree.add(categoryItem);
					
					// カテゴリコードでカテゴリ情報を登録
					categoryTable.put(categoryCode, categoryItem);
				}
			}
			
			// 収集項目コードの読み込み
			num = Integer.parseInt(properties.getProperty("collector.item.num"));
			
			for(int i=0; i < num; i++){
				String itemCode = properties.getProperty("collector.item.code." + i);
				String categoryCode = properties.getProperty("collector.item.category.code." + i);
				String parentCode = properties.getProperty("collector.item.parent.code." + i);
				String itemName = properties.getProperty("collector.item.name." + i);
				String measure = properties.getProperty("collector.item.measure." + i);
				String deviceFlagString = properties.getProperty("collector.device.support." + i);
				String deviceType = properties.getProperty("collector.device.type." + i);
				String rangeFixedString = properties.getProperty("collector.graph.range." + i);
				
				if(itemCode != null && categoryCode != null
						&& deviceFlagString != null && rangeFixedString != null){
					boolean deviceFlag = Boolean.parseBoolean(deviceFlagString);
					
					boolean rangeFixed;
					if(rangeFixedString.equals("fix")){
						rangeFixed = true;
					} else {
						rangeFixed = false;
					}
					
					// カテゴリ名を調べます
					CollectorItemTreeItem categoryTreeItem = categoryTable.get(categoryCode);
					String category = null;
					if(categoryTreeItem != null){
						category = categoryTreeItem.getCategory().getCategoryName();
					}
					
					CollectorItemCodeInfo codeInfo = new CollectorItemCodeInfo(
							itemCode,
							categoryCode,
							parentCode,
							category,
							itemName,
							measure,
							deviceFlag,
							deviceType,
							rangeFixed
					);
					
					// 親のオブジェクトを取得（存在しない場合はnull）
					CollectorItemTreeItem parentItem = 
						(CollectorItemTreeItem) m_codeTable.get(parentCode);
					
					// 親のコードが存在しない場合はカテゴリの直下の要素とする
					if(parentItem == null){
						parentItem = categoryTreeItem;
					}
					
					CollectorItemTreeItem ctItem = 
						new CollectorItemTreeItem(parentItem, codeInfo, null);
					
					// 収集項目コードをキーとして登録
					m_codeTable.put(itemCode, ctItem);
				}
			}
		} catch (Exception e) {
			m_codeTable.clear();
			m_itemTree.clear();
			m_codeTable.clear();
			// エラー処理
			e.printStackTrace();
		}
	}
	
	/**
	 * 指定収集項目コードの項目名を返します。
	 * 
	 * @param itemCode
	 * @return 指定収集項目コードの項目名
	 */
	public static String getItemName(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode))
		.getItemName();
	}
	
	/**
	 * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
	 * 
	 * @param itemCode 収集項目コード
	 * @param deviceName デバイス情報
	 * @return 指定収集項目コードの項目名
	 */
	public static String getFullItemName(String itemCode, String deviceName) {
		String name;
		if (deviceName == null || deviceName.equals(Messages.getString("none"))) {
			name = ((CollectorItemTreeItem) m_codeTable.get(itemCode))
			.getItemName();
		} else {
			name = ((CollectorItemTreeItem) m_codeTable.get(itemCode))
			.getItemName()
			+ "(" + deviceName + ")";
		}
		
		return name;
	}
	
	/**
	 * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
	 * 
	 * @param collectorItemInfo 収集項目情報
	 * @return 指定収集項目コードの項目名
	 */
	public static String getFullItemName(CollectorItemInfo collectorItemInfo) {
		String itemCode = collectorItemInfo.getCollectorItemCode();
		
		// デバイス別の収集を行う項目か否かで出力を変更
		if(isDeviceSupport(itemCode)){
			// デバイス名も含めた項目名を返す
			return getFullItemName(
					collectorItemInfo.getCollectorItemCode(), 
					collectorItemInfo.getDeviceName()
			);
		} else {   	
			return getItemName(itemCode);
		}
	}
	
	/**
	 * 指定収集項目コードの性能値の単位を返します。
	 * 
	 * @return 性能値の単位
	 */
	public static String getMeasure(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode)).getMeasure();
	}
	
	/**
	 * 性能値の値が百分率か否かを返します。
	 * 
	 * @return true 百分率 false 百分率ではない
	 */
	public static boolean isRangeFixed(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode))
		.isRangeFixed();
	}
	
	/**
	 * デバイス別の性能値か否かを返します。
	 * 
	 * @return true デバイス別の性能値、false デバイス別の性能値ではない
	 */
	public static boolean isDeviceSupport(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode)).isDeviceFlag();
	}
	
	/**
	 * デバイス種別を返します。
	 * 
	 * @return デバイス種別
	 */
	public static String getDeviceType(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode)).getDeviceType();
	}
	
	/**
	 * カテゴリ数を返します。
	 * 
	 * @return カテゴリ数
	 */
	public static int getCategorySize() {
		return m_categoryList.size();
	}
	
	/**
	 * 指定インデックスのカテゴリ名を返します。
	 * 
	 * @return カテゴリ名
	 */
	public static String getCategoryName(int index) {
		return ((CollectorItemCategoryInfo) m_categoryList.get(index))
		.getCategoryName();
	}
	
	/**
	 * 指定収集項目コードのカテゴリ名を返します。
	 * 
	 * @return カテゴリ名
	 */
	public static String getCategoryName(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode))
		.getCategory().getCategoryName();
	}
	
	/**
	 * 指定インデックスのカテゴリコードを返します。
	 * 
	 * @return カテゴリコード
	 */
	public static String getCategoryCode(int index) {
		return ((CollectorItemCategoryInfo) m_categoryList.get(index))
		.getCategoryCode();
	}
	
	/**
	 * 指定収集項目コードのカテゴリコードを返します。
	 * 
	 * @return カテゴリコード
	 */
	public static String getCategoryCode(String itemCode) {
		return ((CollectorItemTreeItem) m_codeTable.get(itemCode))
		.getCategory().getCategoryCode();
	}
	
	/**
	 * 全てのコードを返します。
	 * 
	 * @return 全てのコード
	 */
	public static String[] getCodeAll() {
		TreeSet<String> set = new TreeSet<String>(m_codeTable.keySet());
		return (String[])set.toArray(new String[set.size()]);
	}
	
	/**
	 * 指定の収集IDの収集のデバイス情報をDBから取得し収集項目ツリーを生成して返します。
	 * 指定の収集IDがnullの場合はリポジトリからデバイス情報を取得する。
	 * 
	 * @param facilityId ファシリティID
	 * @param collectorId 収集ID
	 * @return
	 */
	public static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId){
		List<DeviceData> deviceList = new ArrayList<DeviceData>();
		
		if(collectorId == null){
			if(facilityId != null){
				// リポジトリからデバイス情報を取得
				Repository repository = new Repository();
				
				// ファシリティがノードの場合だけリポジトリからデバイス情報を取得する
				if(repository.isNode(facilityId)){
					deviceList = repository.getDeviceList(facilityId);
				}
			}
		} else {
			// DBからデバイス情報を取得
			// デバイス情報用のDAOを生成
			DeviceDAO dao = new DeviceDAO();
			
			// デバイス情報をDBから取得する
			Collection devicList = dao.selectDeviceByCollectorID(collectorId);
			
			// デバイス情報を設定
			Iterator itr = devicList.iterator();
			while(itr.hasNext()){
				DeviceData device = (DeviceData)itr.next();
				
				if(device.getFacilityId().equals(facilityId)){
					deviceList.add(device);
				}
			}
		}
		
		return getCollectorTree(facilityId, deviceList);
	}
	
	/**
	 * 指定のファシリティのデバイス情報を元に、収集項目ツリーを生成して返します。
	 *  
	 */
	public static CollectorItemTreeItem[] getCollectorTree(String facilityId, List deviceList){
		CollectorItemTreeItem[] returnTrees = new CollectorItemTreeItem[m_categoryList.size()];
		
		for(int i=0; i<returnTrees.length; i++){
			final CollectorItemTreeItem masterTree = (CollectorItemTreeItem)m_itemTree.get(i);
			
			// カテゴリ用(最上位)の要素を生成する
			returnTrees[i] = new CollectorItemTreeItem(
					null,  // 親要素の指定なし
					masterTree.getData(),
					null   // デバイス情報なし
			);
			
			// マスタの収集項目ツリー(プロパティから生成されたもので読み込み専用)のカテゴリ要素(最上位)の
			// 子要素を取得する。
			List children = masterTree.getChildrens();
			Iterator itr = children.iterator();
			
			// マスタの子要素をコピーしたものを生成する。
			// デバイス情報を指定されている場合は、デバイス情報に基づき
			// デバイス別収集項目のツリーを生成する。
			// デバイス情報がない場合は、デバイス別収集項目は作成しない
			while(itr.hasNext()){
				CollectorItemTreeItem masterTreeChild = (CollectorItemTreeItem)itr.next();
				CollectorItemCodeInfo masterCodeInfo = (CollectorItemCodeInfo)masterTreeChild.getData();
				
				if(masterCodeInfo.isDeviceFlag()){ // デバイス対応の場合					
					if(deviceList != null){
						String deviceType = masterCodeInfo.getDeviceType();

						Iterator idev = deviceList.iterator();
						while(idev.hasNext()){
							DeviceData device = (DeviceData)idev.next();
							if(device.getDeviceType().equals(deviceType)){
								makeTree(returnTrees[i], masterTreeChild, device);
							}
						}
					}
				} else {
					// デバイス情報なしでツリーの要素を作成する
					makeTree(returnTrees[i], masterTreeChild, null);					
				}
			}
		}
		
		return returnTrees;
	}
	
	/**
	 * 指定のファシリティのデバイス情報を元に、収集項目ツリーを生成して返します。
	 * (収集IDにはnullが設定されます)
	 * 
	 * @param facilityID ファシリティID
	 * @param dataSet デバイス情報
	 * @return 指定のファシリティで設定可能な収集項目の一覧(収集IDはnull)
	 */
	public static List getAvailableCollectorItemList(String facilityId){
		List<DeviceData> deviceList = new ArrayList<DeviceData>();
		ArrayList<CollectorItemInfo> retItemList = new ArrayList<CollectorItemInfo>();
		int collectorItemId = 0;  // 返却するリストの収集項目ID
		
		if(facilityId != null){
			// リポジトリからデバイス情報を取得
			Repository repository = new Repository();
			
			// ファシリティがノードの場合だけリポジトリからデバイス情報を取得する
			if(repository.isNode(facilityId)){
				deviceList = repository.getDeviceList(facilityId);
			}
		}
		
		String[] codes = getCodeAll();
		
		for(int i=0; i<codes.length; i++){
			String itemCode = codes[i];
			
			CollectorItemInfo itemInfo = null;
			
			int deviceIndex = 0;
			String deviceName = Messages.getString("none");
			// デバイス別の収集をサポートする収集か否かを判定
			if(!isDeviceSupport(itemCode)){
				// デバイス別の収集をサポートしない場合
				itemInfo = new CollectorItemInfo(
						null,  // 収集ID
						collectorItemId,  // 収集項目ID
						itemCode,  // 収集項目コード
						deviceIndex,   // デバイスのインデックス
						deviceName,  // デバイス名
						false  // 閾値フラグ
				);
				retItemList.add(collectorItemId, itemInfo);
				collectorItemId++;   // 収集項目IDをインクリメント
			} else {
				// デバイス別の収集をサポートする場合
				if(deviceList != null){  // デバイス情報が登録されている場合
					// デバイスの数だけループ
					Iterator itr = deviceList.iterator();
					while(itr.hasNext()){
						DeviceData deviceData = (DeviceData)itr.next();
						
						// 収集コードで扱うデバイス種別を取得する
						String deviceType = getDeviceType(itemCode);
						
						// 収集項目コードで設定されたデバイス種別と一致する場合
						if(deviceData.getDeviceType().equals(deviceType)){							
							deviceIndex = deviceData.getDeviceIndex();   // デバイスのインデックス
							deviceName = deviceData.getDeviceName();  // デバイス名
							
							itemInfo = new CollectorItemInfo(
									null,  // 収集ID
									collectorItemId,  // 収集項目ID
									itemCode,  // 収集項目コード
									deviceIndex,   // デバイスのインデックス
									deviceName,  // デバイス名
									false  // 閾値フラグ
							);
							retItemList.add(collectorItemId, itemInfo);
							collectorItemId++;   // 収集項目IDをインクリメント
						}
					}
				}
			}
		}
		return retItemList;
	}
	
	private static void makeTree(CollectorItemTreeItem parent, CollectorItemTreeItem masterItem, DeviceData deviceData){
		CollectorItemTreeItem treeItem = new CollectorItemTreeItem(parent, masterItem.getData(), deviceData);
		
		List children = masterItem.getChildrens();
		Iterator itr = children.iterator();
		
		while(itr.hasNext()){
			CollectorItemTreeItem masterTreeChild = (CollectorItemTreeItem)itr.next();
			makeTree(treeItem, masterTreeChild, deviceData);
		}
	}
	
	public static HashMap getItemCodeMap(){
		return m_codeTable;
	}
}