/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.dao;

import java.io.UnsupportedEncodingException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.CommunicationException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.UserAttributeConstant;
import com.clustercontrol.accesscontrol.ejb.entity.RoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.RoleUtil;
import com.clustercontrol.accesscontrol.ejb.entity.UserBean;
import com.clustercontrol.accesscontrol.ejb.entity.UserPK;
import com.clustercontrol.accesscontrol.util.LdapConnectionManager;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * UserのDAOインターフェースを実装するクラスです。
 *
 * @version 2.2.0
 * @since 2.0.0
 * 
 * @see com.clustercontrol.accesscontrol.ejb.entity.UserBean
 * @see com.clustercontrol.accesscontrol.dao.UserDAO
 */
public class UserDAOImpl implements UserDAO {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( UserDAOImpl.class );
	/** オブジェクトクラス属性名 */
	protected static final String OBJECTCLASS = "objectClass";
	/** ベースDN */
	protected static final String BASE_DN = "ou=hinemos,ou=people";
	/** Userのオブジェクトクラス属性値 */
	protected static final String OBJECT_CLASS_USER = "ccUser";
	/** LDAP用コンテキスト取得時のリトライ上限 */
	private static int MAX_RETRY = 3;
	
	public void init() {
		
	}

	/** 
	 * 引数で指定されたプライマリキーでLDAPを検索し、取得した内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#load(com.clustercontrol.accesscontrol.ejb.entity.UserPK, com.clustercontrol.accesscontrol.ejb.entity.UserBean)
	 */
	public void load(UserPK pk, UserBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			
			try {
				clearBean(ejb);
				
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
				
				//dn取得
				ejb.setDn(pk.getDn());
				//uid取得
				ejb.setUid((String)attrs.get(UserAttributeConstant.UID).get(0));
				//cn取得
				if(attrs.get(UserAttributeConstant.CN) != null){
					ejb.setCn((String)attrs.get(UserAttributeConstant.CN).get(0));
				}
				//sn取得
				if(attrs.get(UserAttributeConstant.SN) != null){
					ejb.setSn((String)attrs.get(UserAttributeConstant.SN).get(0));
				}
				//userPassword取得
				if(attrs.get(UserAttributeConstant.USERPASSWORD) != null){
					byte[] passwordByte = (byte[])attrs.get(UserAttributeConstant.USERPASSWORD).get(0);
					String password = null;
					try {
						password = new String(passwordByte, "UTF-8");
					} catch (UnsupportedEncodingException e) {
					}
					ejb.setPassword(password);
				}
				//description取得
				if(attrs.get(UserAttributeConstant.DESCRIPTION) != null){
					ejb.setDescription((String)attrs.get(UserAttributeConstant.DESCRIPTION).get(0));
				}
				//ccCreateTimestamp取得
				if(attrs.get(UserAttributeConstant.CREATETIMESTAMP) != null){
					ejb.setCreateTimestamp(stringToDate((String)attrs.get(UserAttributeConstant.CREATETIMESTAMP).get(0)));
				}
				//ccCreatorsName取得
				if(attrs.get(UserAttributeConstant.CREATORSNAME) != null){
					ejb.setCreatorsName((String)attrs.get(UserAttributeConstant.CREATORSNAME).get(0));
				}
				//ccModifiersName取得
				if(attrs.get(UserAttributeConstant.MODIFIERSNAME) != null){
					ejb.setModifiersName((String)attrs.get(UserAttributeConstant.MODIFIERSNAME).get(0));
				}
				//ccModifyTimestamp取得
				if(attrs.get(UserAttributeConstant.MODIFYTIMESTAMP) != null){
					ejb.setModifyTimestamp(stringToDate((String)attrs.get(UserAttributeConstant.MODIFYTIMESTAMP).get(0)));
				}
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
	
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("load(): catch NamingException" + ex.getMessage());
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
				throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("load(): LDAP query result is null");
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("load() context close failed : " + e.getMessage());
				}
			}
			return;
		}
	}

	/**
	 * 引数で指定されたEntity Beanの内容でLDAPを更新します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#store(com.clustercontrol.accesscontrol.ejb.entity.UserBean)
	 */
	public void store(UserBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(ejb.getDn());
				
				javax.naming.directory.Attributes addAttrs = 
					new javax.naming.directory.BasicAttributes();
				javax.naming.directory.Attributes replaceAttrs = 
					new javax.naming.directory.BasicAttributes();
				javax.naming.directory.Attributes removeAttrs = 
					new javax.naming.directory.BasicAttributes();
				
				//cn設定
				if(attrs.get(UserAttributeConstant.CN) == null && ejb.getCn() != null){
					addAttrs.put(UserAttributeConstant.CN, ejb.getCn());
				}
				else if(attrs.get(UserAttributeConstant.CN) != null && ejb.getCn() == null){
					removeAttrs.put(UserAttributeConstant.CN, null);
				}
				else if(attrs.get(UserAttributeConstant.CN) != null && ejb.getCn() != null){
					replaceAttrs.put(UserAttributeConstant.CN, ejb.getCn());
				}
				//sn設定
				if(attrs.get(UserAttributeConstant.SN) == null && ejb.getSn() != null){
					addAttrs.put(UserAttributeConstant.SN, ejb.getSn());
				}
				else if(attrs.get(UserAttributeConstant.SN) != null && ejb.getSn() == null){
					removeAttrs.put(UserAttributeConstant.SN, null);
				}
				else if(attrs.get(UserAttributeConstant.SN) != null && ejb.getSn() != null){
					replaceAttrs.put(UserAttributeConstant.SN, ejb.getSn());
				}
				//password設定
				if(attrs.get(UserAttributeConstant.USERPASSWORD) == null && ejb.getPassword() != null){
					addAttrs.put(UserAttributeConstant.USERPASSWORD, ejb.getPassword().getBytes());
				}
				else if(attrs.get(UserAttributeConstant.USERPASSWORD) != null && ejb.getPassword() == null){
					removeAttrs.put(UserAttributeConstant.USERPASSWORD, null);
				}
				else if(attrs.get(UserAttributeConstant.USERPASSWORD) != null && ejb.getPassword() != null){
					replaceAttrs.put(UserAttributeConstant.USERPASSWORD, ejb.getPassword().getBytes());
				}
				//description設定
				if(attrs.get(UserAttributeConstant.DESCRIPTION) == null && ejb.getDescription() != null){
					addAttrs.put(UserAttributeConstant.DESCRIPTION, ejb.getDescription());
				}
				else if(attrs.get(UserAttributeConstant.DESCRIPTION) != null && ejb.getDescription() == null){
					removeAttrs.put(UserAttributeConstant.DESCRIPTION, null);
				}
				else if(attrs.get(UserAttributeConstant.DESCRIPTION) != null && ejb.getDescription() != null){
					replaceAttrs.put(UserAttributeConstant.DESCRIPTION, ejb.getDescription());
				}
				//ccCreateTimestamp設定
				if(attrs.get(UserAttributeConstant.CREATETIMESTAMP) == null && ejb.getCreateTimestamp() != null){
					addAttrs.put(UserAttributeConstant.CREATETIMESTAMP, dateToString(ejb.getCreateTimestamp()));
				}
				else if(attrs.get(UserAttributeConstant.CREATETIMESTAMP) != null && ejb.getCreateTimestamp() == null){
					removeAttrs.put(UserAttributeConstant.CREATETIMESTAMP, null);
				}
				else if(attrs.get(UserAttributeConstant.CREATETIMESTAMP) != null && ejb.getCreateTimestamp() != null){
					replaceAttrs.put(UserAttributeConstant.CREATETIMESTAMP, dateToString(ejb.getCreateTimestamp()));
				}
				//ccCreatorsName設定
				if(attrs.get(UserAttributeConstant.CREATORSNAME) == null && ejb.getCreatorsName() != null){
					addAttrs.put(UserAttributeConstant.CREATORSNAME, ejb.getCreatorsName());
				}
				else if(attrs.get(UserAttributeConstant.CREATORSNAME) != null && ejb.getCreatorsName() == null){
					removeAttrs.put(UserAttributeConstant.CREATORSNAME, null);
				}
				else if(attrs.get(UserAttributeConstant.CREATORSNAME) != null && ejb.getCreatorsName() != null){
					replaceAttrs.put(UserAttributeConstant.CREATORSNAME, ejb.getCreatorsName());
				}
				//ccModifiersName設定
				if(attrs.get(UserAttributeConstant.MODIFIERSNAME) == null && ejb.getModifiersName() != null){
					addAttrs.put(UserAttributeConstant.MODIFIERSNAME, ejb.getModifiersName());
				}
				else if(attrs.get(UserAttributeConstant.MODIFIERSNAME) != null && ejb.getModifiersName() == null){
					removeAttrs.put(UserAttributeConstant.MODIFIERSNAME, null);
				}
				else if(attrs.get(UserAttributeConstant.MODIFIERSNAME) != null && ejb.getModifiersName() != null){
					replaceAttrs.put(UserAttributeConstant.MODIFIERSNAME, ejb.getModifiersName());
				}
				//ccModifyTimestamp設定
				if(attrs.get(UserAttributeConstant.MODIFYTIMESTAMP) == null && ejb.getModifyTimestamp() != null){
					addAttrs.put(UserAttributeConstant.MODIFYTIMESTAMP, dateToString(ejb.getModifyTimestamp()));
				}
				else if(attrs.get(UserAttributeConstant.MODIFYTIMESTAMP) != null && ejb.getModifyTimestamp() == null){
					removeAttrs.put(UserAttributeConstant.MODIFYTIMESTAMP, null);
				}
				else if(attrs.get(UserAttributeConstant.MODIFYTIMESTAMP) != null && ejb.getModifyTimestamp() != null){
					replaceAttrs.put(UserAttributeConstant.MODIFYTIMESTAMP, dateToString(ejb.getModifyTimestamp()));
				}
				//属性を変更する（追加）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.ADD_ATTRIBUTE, 
						addAttrs);
				//属性を変更する（置換）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.REPLACE_ATTRIBUTE, 
						replaceAttrs);
				//属性を変更する（削除）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.REMOVE_ATTRIBUTE, 
						removeAttrs);
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
	
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("store(): catch NamingException" + ex.getMessage());
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("store() : LDAP query result is null");
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("store() context close failed : " + e.getMessage());
				}
			}
			return;
		}
	}

	/**
	 * 引数で指定されたプライマリキーでLDAPから削除します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#remove(com.clustercontrol.accesscontrol.ejb.entity.UserPK)
	 */
	public void remove(UserPK pk) throws RemoveException, EJBException {
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//エントリを削除
				ctx.destroySubcontext(pk.getDn());
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
					
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("remove(): catch NamingException" + ex.getMessage());
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
		            AplLogger apllog = new AplLogger("ACC", "acc");
		            String[] args = {pk.getDn()};
		            apllog.put("SYS", "003", args);
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("remove(): LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("remove() context close failed : " + e.getMessage());
				}
			}
			return;
		}
	}

	/**
	 * 引数で指定されたEntity Beanの内容をLDAPに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#create(com.clustercontrol.accesscontrol.ejb.entity.UserBean)
	 */
	public UserPK create(UserBean ejb) throws CreateException, EJBException {
		UserPK pk = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// Attributes for new entry
				javax.naming.directory.Attributes attrs =
					new javax.naming.directory.BasicAttributes();
				
				//objectClassを設定
				javax.naming.directory.Attribute attr =
					new javax.naming.directory.BasicAttribute(OBJECTCLASS);
				attr.add(0, OBJECT_CLASS_USER);
				attrs.put(attr);
				
				//uidを設定
				attrs.put(UserAttributeConstant.UID, ejb.getUid());
				// dn文字列作成
				StringBuffer dn = new StringBuffer();
				dn.append(UserAttributeConstant.UID);
				dn.append("=");
				dn.append(ejb.getUid());
				dn.append(",");
				dn.append(BASE_DN);
				
				//dnを設定
				ejb.setDn(dn.toString());
	
				//エントリを作成
				ctx.createSubcontext(ejb.getDn(), attrs);
	
				pk = new UserPK(ejb.getDn());
				
			} catch ( javax.naming.NamingException ex ) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				
				// NamingExceptionが発生した場合は、catchしリトライする
				m_log.debug("create(): catch NamingException" + ex.getMessage());
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
		            AplLogger apllog = new AplLogger("ACC", "acc");
		            String[] args = {ejb.getDn()};
		            apllog.put("SYS", "002", args);
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
	
//				2.1.1改修分マージ時にコメントアウト 7/5 kozakai
	//		    if(ex instanceof CommunicationException){
	//		        LdapConnectionManager.getConnectionManager().setDirContext(null);
	//		    }
	//		    else if(ex instanceof NameAlreadyBoundException){
	//		    	throw new DuplicateKeyException(ex.getMessage());
	//			}
	//			throw new CreateException(ex.getMessage());
	
				
				
				//ここではjavax.naming.NameAlreadyBoundExceptionはキャッチできない?
				//catch(javax.naming.NameAlreadyBoundException e){
				//	m_log.info("create(): user " + ejb.cn + " is already exist");
				//}
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("create(): LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("create() context close failed : " + e.getMessage());
				}
			}
			return pk;
		}
		return pk;
	}

	/**
	 * 全件取得する検索を行います。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.UserBean#ejbFindAll()
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		ArrayList<UserPK> ret = new ArrayList<UserPK>();

		m_log.debug("findAll()");
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_USER);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
	
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null) {
					m_log.debug("findAll() : No Search Result");
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					UserPK pk = new UserPK(aResult.getName() + "," + BASE_DN);
					ret.add(pk);
				}
				
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				m_log.debug("findAll() : " + ex.getMessage());
	
				//中身を空にします。
				ret.clear();
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
//				2.1.1改修分マージ時にコメントアウト 7/5 kozakai
//				m_log.debug("findAll() : " + e.getMessage());
//				throw new FinderException(e.getMessage());
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findAll() : LDAP query result is null");
				
				//中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findAll() context close failed : " + e.getMessage());
				}
			}
	
			return ret;
		}
		return ret;
	}
	

	/**
	 * 1件取得する検索を行います。
	 * 引数で指定されたプライマリキーで、LDAPを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.UserBean#ejbFindByPrimaryKey(UserPK)
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#findByPrimaryKey(com.clustercontrol.accesscontrol.ejb.entity.UserPK)
	 */
	public UserPK findByPrimaryKey(UserPK pk) throws FinderException {
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			try {
				m_log.debug("findByPrimaryKey() : dn = " + pk.getDn());
				
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
				attrs.hashCode();
				
				return pk;
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findByPrimaryKey() : " + ex.getMessage());
	
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new FinderException("dn=" + pk.getDn() + " not found.");
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
//				2.1.1改修分マージ時にコメントアウト 7/5 kozakai
//				m_log.debug("findByPrimaryKey() : " + ex.getMessage());
//				String msg = "RoleDAOImpl.findByCnAndMenber() find error : dn=" + pk.getDn() + " not found.";
//				throw new FinderException(msg);
			
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByPrimaryKey(): LDAP query result is null");
	
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("findByPrimaryKey() context close failed : " + e.getMessage());
				}
			}
		}
		return pk;
	}
	
	/**
	 * 1件取得する検索を行います。
	 * 引数で指定されたユーザIDで、LDAPを検索します。
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.UserBean#ejbFindByUid(String)
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#findByUid(java.lang.String)
	 */
	public UserPK findByUid(String uid) throws FinderException {
		m_log.debug("findByUid() : uid = " + uid);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// 名前付きオブジェクトを検索
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_USER);
		search.append(")(");
		search.append(UserAttributeConstant.UID);
		search.append("=");
		search.append(uid);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		UserPK pk = null;
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null) {
					String msg = "UserDAOImpl.findByUid() find error : uid=" + uid + " not found.";
					throw new FinderException(msg);
				}
				
				javax.naming.directory.SearchResult aResult = null;
				javax.naming.directory.Attributes attrs = null;
				while ( results.hasMoreElements() ) {
					aResult = (javax.naming.directory.SearchResult)results.nextElement();
					attrs = aResult.getAttributes();
					break;
				}
				
				if (attrs == null) {
					String msg = "UserDAOImpl.findByUid() find error : uid=" + uid + " not found.";
					throw new FinderException(msg);
				}
				
				pk = new UserPK(aResult.getName() + "," + BASE_DN);
				
				} catch (NamingException ex) {
					if(ex instanceof CommunicationException){
						LdapConnectionManager.getConnectionManager().setDirContext(null);
					}
			    
					m_log.debug("findByUid() : " + ex.getMessage());
	
					// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
					// ここでカウントアップする。
					retryCount++;
					if(retryCount >= MAX_RETRY){
						throw new FinderException("uid=" + uid + " not found.");
					} else {
						//リトライ上限以下の場合には再度LDAPに問い合わせをします。
						//whlie loopに戻ります。
						continue;
	//		    m_log.debug("findByUid() : " + e.getMessage());
	//			String msg = "UserDAOImpl.findByUid() find error : uid=" + uid + " not found.";
	//			throw new FinderException(msg);
					}
					
				} catch (NullPointerException e){
					//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
					//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
					m_log.debug("findByUid() : LDAP query result is null");
	
					// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
					// ここでカウントアップする。
					retryCount++;
					if(retryCount >= MAX_RETRY){
						throw new EJBException(e);
					} else {
						//リトライ上限以下の場合には再度LDAPに問い合わせをします。
						//whlie loopに戻ります。
						continue;
					}
					
				} finally {
					try {
						if(results != null)
							results.close();
						if(ctx != null)
							ctx.close();
					} catch (NamingException e) {
						m_log.error("findByUid() context close failed : " + e.getMessage());
					}
			}
			return pk;
		}
		return pk;
	}
	
	
	/**
	 * 複数件取得する検索を行います。
	 * 引数で指定された値で、LDAPを検索します。<BR>
	 * 引数：ユーザID、ユーザ名、説明、アクセス権限リスト
	 * 
	 * @see com.clustercontrol.accesscontrol.ejb.entity.UserBean#ejbFindUser(String, String, String, List)
	 * @see com.clustercontrol.accesscontrol.dao.UserDAO#findUser(java.lang.String, java.lang.String, java.lang.String, java.util.List)
	 */
	public Collection findUser(
			String uid, 
			String name, 
			String description, 
			List accessList) throws FinderException {
		ArrayList<UserPK> ret = new ArrayList<UserPK>();

		m_log.debug("findUser()");
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_USER);
		// UIDを検索文字列に設定
		if(uid != null && uid.compareTo("") != 0){
			search.append(")(");
			search.append(UserAttributeConstant.UID);
			search.append("=");
			search.append(uid);
		}
		// 名前を検索文字列に設定
		if(name != null && name.compareTo("") != 0){
			search.append(")(");
			search.append(UserAttributeConstant.CN);
			search.append("=");
			search.append(name);
		}
		// 説明を検索文字列に設定
		if(description != null && description.compareTo("") != 0){
			search.append(")(");
			search.append(UserAttributeConstant.DESCRIPTION);
			search.append("=");
			search.append(description);
		}
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null) {
					m_log.debug("findUser() : No Search Result");
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = 
						(javax.naming.directory.SearchResult)results.nextElement();
					UserPK pk = new UserPK(aResult.getName() + "," + BASE_DN);
					
					boolean find = true;
					for(int i = 0; i < accessList.size(); i++){
						try {
							RoleLocal role = 
								RoleUtil.getLocalHome().findByCnAndMenber(
										(String)accessList.get(i), 
										pk.getDn());
							role.hashCode();
							
							find = true;
						} catch (FinderException e) {
							find = false;
							break;
						} catch (NamingException e) {
							find = false;
							break;
						}
					}
					if(find){
						ret.add(pk);
					}
				}
				
			} catch (NamingException ex) {
				if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findUser() : " + ex.getMessage());
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				
				//中身を空にします。
				ret.clear();
				
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findUser() : LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				
				//中身を空にします。
				ret.clear();
				
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findUser() context close failed : " + e.getMessage());
				}
			}
	
			return ret;
		}
			return ret;
	}
	
	/**
	 * StringをDateに変換する
	 * 
	 * @param value String
	 * 
	 * @return Date
	 */
	public Date stringToDate(String value) {
		Date ret = null;
		
		SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss.SSS");

		try {
			ret = formatter.parse(value);
		} catch (ParseException e) {
		}

		return ret;
	}
	
	/**
	 * DateをStringに変換する
	 * 
	 * @param value Date
	 * @return String
	 */
	public String dateToString(Date value) {
		String ret = null;
		
		SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss.SSSZ");

		ret = formatter.format(value);


		return ret;
	}
	
	/**
	 * 引数で渡されたBeanの値をクリアします。
	 * 
	 * @param ejb User{@link com.clustercontrol.accesscontrol.ejb.entity.UserBean}
	 */
	protected void clearBean(UserBean ejb){
	    ejb.setUid(null);
	    ejb.setDn(null);
	    ejb.setCn(null);
	    ejb.setSn(null);
	    ejb.setPassword(null);
	    ejb.setDescription(null);
	    ejb.setCreateTimestamp(null);
	    ejb.setCreatorsName(null);
	    ejb.setModifiersName(null);
	    ejb.setModifyTimestamp(null);
	}
}
