/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logtransfer.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.logtransfer.action.DeleteLogTransfer;
import com.clustercontrol.logtransfer.action.GetLogTransfer;
import com.clustercontrol.logtransfer.action.ModifyLogTransfer;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.bean.LogTransferTabelDefine;
import com.clustercontrol.logtransfer.composite.LogTransferListComposite;
import com.clustercontrol.util.Messages;

/**
 * ログ転送一覧ダイアログクラス<BR>
 * 
 * @version 2.2.0
 * @since 2.1.0
 */
public class LogTransferListDialog extends CommonDialog {
	
	// ----- instance フィールド ----- //
	
	/** ログ転送一覧 */
	private LogTransferListComposite logTransferList = null;
	
	/** 追加ボタン */
	private Button buttonAdd = null;
	
	/** 変更ボタン */
	private Button buttonModify = null;
	
	/** 削除ボタン */
	private Button buttonDelete = null;
	
	/** 有効ボタン */
    private Button buttonValid = null;
    
    /** 無効ボタン */
    private Button buttonInvalid = null;
	
	
	// ----- コンストラクタ ----- //
	
	/**
	 * ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public LogTransferListDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}
	
	// ----- instance メソッド ----- //
	
	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	protected Point getInitialSize() {
		return new Point(800, 350);
	}
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のインスタンス
	 */
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		
		// タイトル
		shell.setText(Messages.getString("dialog.logtransfer.list"));
		
		// レイアウト
		GridLayout layout = new GridLayout(8, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 8;
		parent.setLayout(layout);
		
		/*
		 * ログ転送一覧
		 */
		this.logTransferList = new LogTransferListComposite(parent, SWT.BORDER);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 7;
		this.logTransferList.setLayoutData(gridData);
		
		/*
		 * 操作ボタン
		 */
		Composite composite = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		composite.setLayoutData(gridData);
		
		// 追加ボタン
		this.buttonAdd = this
		.createButton(composite, Messages.getString("add"));
		this.buttonAdd.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				LogTransferCreateDialog dialog = new LogTransferCreateDialog(getParentShell());
				if (dialog.open() == IDialogConstants.OK_ID) {
					logTransferList.update();
				}
			}
		});
		
		// 変更ボタン
		@SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
		this.buttonModify = this.createButton(composite, Messages.getString("modify"));
		this.buttonModify.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				
				String transferId = null;
				
				Table table = logTransferList.getTableViewer().getTable();
				TableItem[] item = table.getSelection();
				
				if(item != null && item.length>0){
					try{
						transferId = (String)((ArrayList)item[0].getData()).get(LogTransferTabelDefine.TRANSFER_ID);
					}
					catch(Exception ex){
					}
				}
				
				if (transferId != null) {
					LogTransferCreateDialog dialog = new LogTransferCreateDialog(getParentShell(), transferId);
					if (dialog.open() == IDialogConstants.OK_ID) {
						int selectIndex = table.getSelectionIndex();
						logTransferList.update();
						table.setSelection(selectIndex);
					}
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.logtransfer.8"));
				}
			}
		});
		
		// 削除ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonDelete = this.createButton(composite, Messages.getString("delete"));
		this.buttonDelete.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				
				String transferId = null;
				
				Table table = logTransferList.getTableViewer().getTable();
				TableItem[] item = table.getSelection();
				
				if(item != null && item.length>0){
					try{
						transferId = (String)((ArrayList)item[0].getData()).get(LogTransferTabelDefine.TRANSFER_ID);
					}
					catch(Exception ex){
					}
				}
				
				if (transferId != null) {
					
					String[] args = { transferId };
					if (MessageDialog.openConfirm(
		        			null, 
		        			Messages.getString("confirmed"), 
		        			Messages.getString("message.logtransfer.7", args))) {
						
						boolean result = new DeleteLogTransfer().delete(transferId);
						if(result){
		                	MessageDialog.openInformation(
		                			null, 
		                			Messages.getString("successful"), 
		                			Messages.getString("message.logtransfer.5", args));
		                	
							logTransferList.update();
						}
						else{
		                	MessageDialog.openError(
		                			null, 
		                			Messages.getString("failed"), 
		                			Messages.getString("message.logtransfer.6", args));
						}
					}
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.logtransfer.8"));
				}
			}
		});
		
		// 有効ボタン
		dummy = new Label(composite, SWT.NONE);
        dummy = new Label(composite, SWT.NONE);
        this.buttonValid = this.createButton(composite, Messages.getString("valid"));
        this.buttonValid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で有効に変更
            	setValid(YesNoConstant.TYPE_YES);
            }
        });
        
        // 無効ボタン
        this.buttonInvalid = this.createButton(composite, Messages.getString("invalid"));
        this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で無効に変更
            	setValid(YesNoConstant.TYPE_NO);
            }
        });
		
		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 8;
		line.setLayoutData(gridData);
		
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
	}
	
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	protected ValidateResult validate() {
		return super.validate();
	}
	
	/**
	 * 共通のボタンを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * @param label ボタンのラベル
	 * @return 生成されたボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);
		
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		
		button.setText(label);
		
		return button;
	}
	
	/**
	 * 閉じるボタンを作成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 */
	protected void createButtonsForButtonBar(Composite parent) {
		
		// 閉じるボタン
		this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
	}
	
	/**
     * 有効・無効変更処理
     * 
     * @param valid
     */
    public void setValid(int valid) {

		//選択された監視項目IDを取得
		ArrayList<String> list = logTransferList.getSelectionData();
		if (list != null && list.size() > 0) {
			
			StringBuilder transferIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					transferIds.append(", ");
				}
				transferIds.append(list.get(i));
			}

            String[] confirmArgs = { transferIds.toString() };
            String message;
            if(valid == YesNoConstant.TYPE_YES)
            	message = Messages.getString("message.logtransfer.16",confirmArgs);
            else
            	message = Messages.getString("message.logtransfer.17",confirmArgs);
            if (!MessageDialog.openConfirm(
        			null, 
        			Messages.getString("confirmed"), 
        			message)) {
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				String transferId = list.get(i);
				
				if(transferId != null && !transferId.equals("")){
			        //監視情報を取得
					LogTransferInfo  info = new GetLogTransfer().getLogTransfer(transferId);        	
			        
			        //有効・無効を設定
			        info.setValid(valid);
			        
			        //ログ転送情報を更新
			        boolean result = new ModifyLogTransfer().modify(info);
                    if(!result){
                        String[] args = { transferId };
                    	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.logtransfer.4", args));
                    }
				}
			}
    			
            int selectIndex = logTransferList.getTableViewer().getTable().getSelectionIndex();
            logTransferList.update();
            logTransferList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageDialog.openWarning(
        			null, 
        			Messages.getString("warning"), 
        			Messages.getString("message.logtransfer.8"));
        }
    }
}