/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.calendar.bean.ScheduleTableConstant;
import com.clustercontrol.calendar.factory.SelectCalendar;
import com.clustercontrol.jobmanagement.bean.QuartzConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartMasterLocal;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;

/**
 * 実行予定を検索するクラスです。
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class SelectRunSchedule {
	/** ログ出力 */
    protected static Log m_log = LogFactory.getLog( SelectRunSchedule.class );
    
	/**
	 * 実行予定一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>Quartzからスケジュール情報を取得します。</li>
	 * <li>取得したスケジュール情報の数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1スケジュール情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.RunScheduleListTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>実行予定一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ 実行予定情報1 {カラム1の値, カラム2の値, … }, 実行予定情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 * </ol>
	 * 
     * @return 実行予定一覧情報
     * @throws NamingException
     * @throws SchedulerException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getList() throws NamingException, SchedulerException {
        m_log.debug("getList()");
        
        ArrayList list = new ArrayList();
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //ジョブ名リスト取得
        String[] jobs = null;
		try {
			jobs = manager.getJobs(QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
			//TODO RemoteException未実装
		}
        for(int i = 0; i < jobs.length; i++){
            //ジョブ(JobDetail)取得
            JobDetail job = null;
			try {
				job = manager.getJobDetail(jobs[i], QuartzConstant.GROUP_NAME);
			} catch (RemoteException e) {
				//TODO RemoteException未実装
			}

            ArrayList info = new ArrayList();
            //スケジュールIDを取得
            info.add(job.getName());
            //スケジュール名を取得
            info.add(job.getDescription());
            //ジョブIDを取得
            Object[] jdArgs = (Object[])job.getJobDataMap().get(EJBInvokerJob.EJB_ARGS_KEY);
            info.add((String)jdArgs[0]);
            //ジョブ名を取得
            String jobName = (String)job.getJobDataMap().get(QuartzConstant.JOB_NAME_KEY);
            info.add(jobName);
            //カレンダIDを取得
            if(jdArgs.length > 1){
            	info.add((String)jdArgs[1]);
            }
            else{
            	info.add("");
            }
            info.add(new Boolean(true));
            
            list.add(info);
        }
	    return list;
    }
    
	/**
	 * 実行予定情報を取得します。
	 * <p>
	 * <ol>
	 * <li>Quartzからスケジュール情報を取得します。</li>
	 * <li>取得したスケジュール情報の数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1スケジュール情報を基に実行予定を、基準日から4週間分のカラム順（{@link com.clustercontrol.jobmanagement.bean.RunScheduleListTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>実行予定情報（Objectの2次元配列）</dt>
	 *   <dd>{ 実行予定1 {カラム1の値, カラム2の値, … }, 実行予定2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 * </ol>
	 * 
     * @param base 基準日
     * @return 実行予定情報
     * @throws NamingException
     * @throws SchedulerException
     * @throws FinderException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getSchedule(Date base) throws NamingException, SchedulerException, FinderException {
        m_log.debug("getSchedule()");
        
        ArrayList list = new ArrayList();
        SelectCalendar selectCalendar = new SelectCalendar();
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //ジョブ名リスト取得
        String[] jobs = null;
		try {
			jobs = manager.getJobs(QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
			//TODO RemoteException未実装
		}
        for(int i = 0; i < jobs.length; i++){
            //ジョブ(JobDetail)取得
            JobDetail job = null;
			try {
				job = manager.getJobDetail(jobs[i], QuartzConstant.GROUP_NAME);
			} catch (RemoteException e) {
				//TODO RemoteException未実装
			}

            ArrayList info = new ArrayList();
            //スケジュールIDを取得
            info.add(job.getName());
            //カレンダIDを取得
            Object[] jdArgs = (Object[])job.getJobDataMap().get(EJBInvokerJob.EJB_ARGS_KEY);
            String calendarId = null;
            if(jdArgs.length > 1){
            	calendarId = (String)jdArgs[1];
            }
            
            //4週間分の予定を取得
    		for(int j = 0; j < ScheduleTableConstant.DAYS; j++){
        		Calendar baseCalendar = Calendar.getInstance();
        		baseCalendar.setTime(base);
    			baseCalendar.add(Calendar.DAY_OF_MONTH, j);
    			Date work = baseCalendar.getTime();
    			if(calendarId != null && calendarId.length() > 0){
    				//カレンダIDの該当カレンダにて、実行日かチェック
	    			if(selectCalendar.isSchedule(calendarId, work)){
	    				info.add(work);
	    			}
	    			else{
	    				info.add(null);
	    			}
    			}
    			else{
    				//カレンダIDが設定されていない場合、実行日とする
    				info.add(work);
    			}
    		}
            list.add(info);
        }
	    return list;
    }
    
	/**
	 * 実行予定詳細一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>スケジュールIDが一致するスケジュール情報をQuartzから取得します。</li>
	 * <li>取得したスケジュール情報からジョブIDを取得し、ジョブIDでジョブマスタを取得します。</li>
	 * <li>テーブルツリー情報のルート(最上位)を作成します。</li>
	 * <li>ジョブマスタとテーブルツリー情報のルートを渡し、実行予定詳細一覧情報を作成します。</li>
	 * </ol>
	 * 
     * @return 実行予定一覧情報
     * @throws NamingException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#createDetailListTree(JobMasterLocal, CommonTableTreeItem)
     */
	public CommonTableTreeItem getDetailList(String scheduleId) throws FinderException, NamingException, SchedulerException {
        //QuartzのSchedulerをルックアップ
		QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //JobDetail取得
        JobDetail job = null;
		try {
			job = manager.getJobDetail(scheduleId, QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
			//TODO RemoteException未実装
		}
        
        //ジョブIDを取得
        Object[] jdArgs = (Object[])job.getJobDataMap().get(EJBInvokerJob.EJB_ARGS_KEY);
        String jobId = (String)jdArgs[0];
		
		//ジョブをジョブIDで検索し取得
		JobMasterLocal jobMaster = JobMasterUtil.getLocalHome().findByPrimaryKey(new JobMasterPK(jobId));

		//CommonTableTreeItemの最上位インスタンスを作成
	    CommonTableTreeItem tree = new CommonTableTreeItem(null, null);
	    //ジョブ詳細ツリーを作成
	    createDetailListTree(jobMaster, tree);
	    
	    return tree;
	}
	
	/**
	 * 実行予定詳細一覧情報を作成します。<BR>
	 * 再帰呼び出しを行います。
	 * <p>
	 * <ol>
	 * <li>ジョブマスタから実行予定詳細一覧情報の1行を作成します。</li>
	 * <li>テーブルツリー情報を作成します。</li>
	 * <li>ジョブIDが親ジョブIDとして一致するジョブリレーションマスタを取得します。</li>
	 * <li>ジョブリレーションマスタの数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>ジョブリレーションマスタからジョブマスタを取得します。</li>
	 *  <li>ジョブマスタとテーブルツリー情報を渡し、実行予定詳細一覧情報を作成します。</li>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionJob セッションジョブ
	 * @param parent 親テーブルツリー情報
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
     * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#createDetailListTreeData(JobMasterLocal)
     */
	protected void createDetailListTree(
			JobMasterLocal job, 
	        CommonTableTreeItem parent) throws FinderException, NamingException {
	    
	    //ジョブからジョブ詳細一覧の１行を作成
		ArrayList info = createDetailListTreeData(job);
		
		//CommonTableTreeItemを作成
		CommonTableTreeItem item = new CommonTableTreeItem(parent, info);
		
		Collection collection = null;
		try {
		    //ジョブリレーションを親ジョブIDで検索
		    collection = 
		    	JobRelationMasterUtil.getLocalHome().findByParentJobId(job.getJob_id());
		} catch (FinderException e) {
			//TODO FinderException 未実装
		} catch (NamingException e) {
			//TODO NamingException 未実装
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
			    //ジョブリレーションを取得
			    JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
			    //ジョブを取得
			    JobMasterLocal childJob = children.getJobMaster();
			    //ジョブ詳細ツリーを作成
			    createDetailListTree(childJob, item);
			}
		}
	}
	
	/**
	 * 実行予定詳細一覧情報の1行を作成します。
	 * <p>
	 * <ol>
	 * <li>1ジョブマスタをテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.RunScheduleDetailListTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *  <dl>
	 *  <dt>実行予定詳細一覧情報の1行（Objectの配列）</dt>
	 *  <dd>実行予定詳細一覧情報の1行 {カラム1の値, カラム2の値, … }</dd>
	 *  </dl>
	 * </ol>
	 * 
	 * @param sessionJob
	 * @return 実行予定詳細一覧情報の1行
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList createDetailListTreeData(JobMasterLocal job) {
	    //ジョブからジョブ詳細一覧の１行を作成
	    ArrayList info = new ArrayList();
	    info.add("");
	    info.add(job.getJob_id());
	    info.add(job.getJob_name());
	    info.add(job.getJob_type());
	    JobStartMasterLocal start = job.getJobStartMaster();
	    if(start != null && start.getCalendar().intValue() == YesNoConstant.TYPE_YES){
	    	info.add(start.getCalendar_id());
	    }
	    else{
	    	info.add(null);
	    }

	    return info;
	}
	
	/**
	 * 実行予定情報を取得します。
	 * <p>
	 * <ol>
	 * <li>Quartzからスケジュール情報を取得します。</li>
	 * <li>取得したスケジュール情報からジョブIDを取得し、ジョブIDでジョブマスタを取得します。</li>
	 * <li>テーブルツリー情報のルート(最上位)を作成します。</li>
	 * <li>ジョブマスタとテーブルツリー情報のルートを渡し、実行予定詳細情報を作成します。</li>
	 * </ol>
	 * 
	 * @param scheduleId スケジュールID
	 * @param base 基準日
	 * @return 実行予定詳細情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SchedulerException
	 * 
     * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#createDetailScheduleTree(JobMasterLocal, CommonTableTreeItem, String, Date)
     */
	public CommonTableTreeItem getDetailSchedule(String scheduleId, Date base) throws FinderException, NamingException, SchedulerException {
        //QuartzのSchedulerをルックアップ
		QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //JobDetail取得
        JobDetail job = null;
		try {
			job = manager.getJobDetail(scheduleId, QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
			//FIXME RemoteException未実装
		}
        
        //ジョブIDを取得
        Object[] jdArgs = (Object[])job.getJobDataMap().get(EJBInvokerJob.EJB_ARGS_KEY);
        String jobId = (String)jdArgs[0];
        //カレンダIDを取得
        String calendarId = null;
        if(jdArgs.length > 1){
        	calendarId = (String)jdArgs[1];
        }
		
		//ジョブをジョブIDで検索し取得
		JobMasterLocal jobMaster = JobMasterUtil.getLocalHome().findByPrimaryKey(new JobMasterPK(jobId));

		//CommonTableTreeItemの最上位インスタンスを作成
	    CommonTableTreeItem tree = new CommonTableTreeItem(null, null);
	    //ジョブ詳細ツリーを作成
	    createDetailScheduleTree(jobMaster, tree, calendarId, base);
	    
	    return tree;
	}
	
	/**
	 * 実行予定詳細情報の1行を作成します。<BR>
	 * 再帰呼び出しを行います。
	 * <p>
	 * <ol>
	 * <li>ジョブマスタから実行予定詳細情報の1行を作成します。</li>
	 * <li>テーブルツリー情報を作成します。</li>
	 * <li>ジョブIDが親ジョブIDとして一致するジョブリレーションマスタを取得します。</li>
	 * <li>ジョブリレーションマスタの数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>ジョブリレーションマスタからジョブマスタを取得します。</li>
	 *  <li>ジョブマスタとテーブルツリー情報を渡し、実行予定詳細情報を作成します。</li>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionJob
	 * @param parent
	 * @throws NamingException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#createDetailScheduleTreeData(JobMasterLocal, ArrayList, String, Date)
	 */
	protected void createDetailScheduleTree(
			JobMasterLocal job, 
	        CommonTableTreeItem parent, 
	        String parentCalendarId, 
	        Date base) throws FinderException, NamingException {
	    
	    //ジョブからジョブ詳細一覧の１行を作成
		ArrayList info = 
			createDetailScheduleTreeData(job, parent.getData(), parentCalendarId, base);
		
		//CommonTableTreeItemを作成
		CommonTableTreeItem item = new CommonTableTreeItem(parent, info);
		
		Collection collection = null;
		try {
		    //ジョブリレーションを親ジョブIDで検索
		    collection = 
		    	JobRelationMasterUtil.getLocalHome().findByParentJobId(job.getJob_id());
		} catch (FinderException e) {
			//TODO FinderException 未実装
		} catch (NamingException e) {
			//TODO NamingException 未実装
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
			    //ジョブリレーションを取得
			    JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
			    //ジョブを取得
			    JobMasterLocal childJob = children.getJobMaster();
			    //ジョブ詳細ツリーを作成
			    createDetailScheduleTree(childJob, item, null, base);
			}
		}
	}
	
	/**
	 * 実行予定詳細情報の1行を作成します。
	 * <p>
	 * <ol>
	 * <li>1ジョブマスタを基に実行予定を、基準日から4週間分のカラム順（{@link com.clustercontrol.jobmanagement.bean.RunScheduleDetailTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *  <dl>
	 *  <dt>実行予定詳細情報の1行（Objectの配列）</dt>
	 *  <dd>実行予定詳細情報の1行 {カラム1の値, カラム2の値, … }</dd>
	 *  </dl>
	 * </ol>
	 * 
	 * @param job ジョブマスタ
	 * @param parentInfo 親テーブルツリー情報の実行予定詳細情報の1行データ
	 * @param parentCalendarId 親ジョブのカレンダID
	 * @param base 基準日
	 * @return 実行予定詳細一覧情報の1行
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList createDetailScheduleTreeData(
			JobMasterLocal job, 
			ArrayList parentInfo, 
			String parentCalendarId, 
			Date base) throws FinderException, NamingException {
		//parentCalendarIdは最上位ジョブのデータ作成時のみ設定される
		
	    //ジョブからジョブ詳細一覧の１行を作成
	    ArrayList info = new ArrayList();
	    SelectCalendar selectCalendar = new SelectCalendar();
	    
	    info.add("");
	    info.add(job.getJob_id());
	    JobStartMasterLocal start = job.getJobStartMaster();
	    String calendarId = null;
	    if(start != null && start.getCalendar().intValue() == YesNoConstant.TYPE_YES){
	    	calendarId = start.getCalendar_id();
	    }
	    
	    //4週間分の予定を取得
		for(int j = 0; j < ScheduleTableConstant.DAYS; j++){
    		Calendar baseCalendar = Calendar.getInstance();
    		baseCalendar.setTime(base);
			baseCalendar.add(Calendar.DAY_OF_MONTH, j);
			Date work = baseCalendar.getTime();
			
			//上位ジョブが実行日かチェック
			boolean run = true;
			if(parentCalendarId != null && parentCalendarId.length() > 0){
				//上位ジョブに指定されたカレンダIDにてチェック
    			if(!selectCalendar.isSchedule(parentCalendarId, work)){
    				//上位ジョブが実行日でない
    				run = false;
    			}
			}
			else{
				//上位ジョブの実行日と比較
				if(parentInfo != null && parentInfo.get(j + 2) == null){
					//上位ジョブが実行日でない
					run = false;
				}
			}
			
			if(run){
				//上位ジョブが実行日の場合
				if(calendarId != null && calendarId.length() > 0){
					//カレンダIDの該当カレンダにて、実行日かチェック
	    			if(selectCalendar.isSchedule(calendarId, work)){
	    				info.add(work);
	    			}
	    			else{
	    				info.add(null);
	    			}
				}
				else{
					//カレンダIDが設定されていない場合、実行日とする
					info.add(work);
				}
			}
			else{
				//上位ジョブが実行日でないため、当該ジョブも実行日としない
				info.add(null);
			}
		}
	    return info;
	}
}
