/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.composite;

import java.rmi.AccessException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.monitor.MonitorPlugin;
import com.clustercontrol.monitor.action.GetEventListTableDefine;
import com.clustercontrol.monitor.action.GetFilterEvent;
import com.clustercontrol.monitor.bean.EventTableDefine;
import com.clustercontrol.monitor.dialog.EventInfoDialog;
import com.clustercontrol.monitor.preference.MonitorPreferencePage;
import com.clustercontrol.util.Messages;

/**
 * イベント情報一覧のコンポジットクラス<BR>
 * 
 * イベント情報一覧部分のテーブルのコンポジット
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class EventListComposite extends Composite {

    /** テーブルビューア */
    private CommonTableViewer tableViewer = null;

    /** 危険ラベル */
    private Label criticalLabel = null;

    /** 警告ラベル */
    private Label warningLabel = null;

    /** 通知ラベル */
    private Label infoLabel = null;

    /** 不明ラベル */
    private Label unknownLabel = null;

    /** 合計ラベル */
    private Label totalLabel = null;
    
    private Shell m_shell = null;
    
    private boolean openAccessInformation = false;
    private boolean openInformation = false;


    /**
     * インスタンスを返します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public EventListComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
        m_shell = this.getShell();
    }

    /**
     * コンポジットを配置します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(5, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 5;
        table.setLayoutData(gridData);

        // ステータス作成

        // 危険
        this.criticalLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.criticalLabel.setLayoutData(gridData);
        this.criticalLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_RED));

        // 警告
        this.warningLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.warningLabel.setLayoutData(gridData);
        this.warningLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_YELLOW));

        // 通知
        this.infoLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.infoLabel.setLayoutData(gridData);
        this.infoLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_GREEN));

        // 不明
        this.unknownLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.unknownLabel.setLayoutData(gridData);
        this.unknownLabel.setBackground(new Color(null, 128, 192, 255));

        // 合計
        this.totalLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.totalLabel.setLayoutData(gridData);

        // テーブルビューアの作成
        this.tableViewer = new CommonTableViewer(table);
        this.tableViewer
                .createTableColumn(GetEventListTableDefine.getEventListTableDefine(),
                        EventTableDefine.SORT_COLUMN_INDEX,
                        EventTableDefine.SORT_ORDER);
         for (int i = 0; i < table.getColumnCount(); i++){
        	table.getColumn(i).setMoveable(true);
        }       
        // ダブルクリックした場合、イベントログの詳細情報ダイアログを表示する
		this.tableViewer.addDoubleClickListener(
                new IDoubleClickListener() {
                    public void doubleClick(DoubleClickEvent event) {
                    	
                        // 選択アイテムを取得する
                        List list = (List) ((StructuredSelection)event.getSelection()).getFirstElement();
                    	
                        EventInfoDialog dialog = new EventInfoDialog(m_shell, list);
                        dialog.open();
                    }
                });
    }

    /**
     * このコンポジットが利用する共通テーブルビューアーを返します。
     * 
     * @return 共通テーブルビューアー
     */
    public CommonTableViewer getTableViewer() {
        return this.tableViewer;
    }
    
    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return this.tableViewer.getTable();
    }

    /**
     * ビューを更新します。<BR>
     * 引数で指定されたファシリティの配下全てのファシリティのイベント一覧情報を取得し、
     * 共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>監視管理のプレファレンスページより、監視[イベント]ビューの表示イベント数を取得します。</li>
	 * <li>引数で指定されたファシリティに属するイベント一覧情報を、表示イベント数分取得します。</li>
	 * <li>表示イベント数を超える場合、メッセージダイアログを表示します。</li>
	 * <li>共通テーブルビューアーにイベント情報一覧をセットします。</li>
	 * </ol>
     * 
     * @param facilityId 表示対象の親ファシリティID
     * 
     * @see com.clustercontrol.monitor.action.GetFilterEvent#getFilterInfo(String, int)
     * @see #updateStatus(ViewListInfo)
     */
    public void update(String facilityId) {
        super.update();

        ViewListInfo infoList = null;

        try {
        	int messages = MonitorPlugin.getDefault().getPreferenceStore().getInt(
        			MonitorPreferencePage.P_EVENT_MAX);
            infoList = new GetFilterEvent().getFilterInfo(facilityId, messages);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (infoList == null) {
            infoList = new ViewListInfo();
            infoList.setList(new ArrayList());
        }
        else if(infoList.getList() == null){
        	infoList.setList(new ArrayList());
        }
        
        if(MonitorPlugin.getDefault().getPreferenceStore().getBoolean(
        		MonitorPreferencePage.P_EVENT_MESSAGE_FLG)){
        	if(infoList.getTotal() > infoList.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.monitor.12"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					MonitorPlugin.getDefault().getPreferenceStore(), 
        					MonitorPreferencePage.P_EVENT_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }

        this.updateStatus(infoList);
        Object[] args = { this.totalLabel.getText() };
        this.totalLabel.setText(Messages.getString("records", args));

        tableViewer.setInput(infoList.getList());
        
    }

    /**
     * ビューを更新します。<BR>
     * 引数で指定された条件に一致するイベント一覧情報を取得し、共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>監視管理のプレファレンスページより、監視[イベント]ビューの表示イベント数を取得します。</li>
	 * <li>引数で指定された条件に一致するイベント一覧情報を、表示イベント数分取得します。</li>
	 * <li>表示イベント数を超える場合、メッセージダイアログを表示します。</li>
	 * <li>共通テーブルビューアーにイベント情報一覧をセットします。</li>
	 * </ol>
     * 
     * @param facilityId 表示対象の親ファシリティID
     * @param condition 検索条件
     * 
     * @see com.clustercontrol.monitor.action.GetFilterEvent#getFilterInfo(String, Property, int)
     * @see #updateStatus(ViewListInfo)
     */
    public void update(String facilityId, Property condition) {
        super.update();

        ViewListInfo infoList = null;

        try {
        	int messages = MonitorPlugin.getDefault().getPreferenceStore().getInt(
        			MonitorPreferencePage.P_EVENT_MAX);
            infoList = new GetFilterEvent()
                    .getFilterInfo(facilityId, condition, messages);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (infoList == null) {
            infoList = new ViewListInfo();
            infoList.setList(new ArrayList());
        }
        else if(infoList.getList() == null){
        	infoList.setList(new ArrayList());
        }
        
        if(MonitorPlugin.getDefault().getPreferenceStore().getBoolean(
        		MonitorPreferencePage.P_EVENT_MESSAGE_FLG)){
        	if(infoList.getTotal() > infoList.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.monitor.12"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					MonitorPlugin.getDefault().getPreferenceStore(), 
        					MonitorPreferencePage.P_EVENT_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }

        this.updateStatus(infoList);
        Object[] args = { this.totalLabel.getText() };
        this.totalLabel.setText(Messages.getString("filtered.records", args));

        tableViewer.setInput(infoList.getList());
    }

    /**
     * ステータスラベルを更新します。<BR>
     * 引数で指定されたビュー一覧情報より、重要度ごとの件数，全件数を取得し、
     * ステータスラベルを更新します。
     * 
     * @param list ビュー一覧情報
     */
    protected void updateStatus(ViewListInfo list) {
        // ラベル更新
        this.criticalLabel.setText(String.valueOf(list.getCritical()));
        this.warningLabel.setText(String.valueOf(list.getWarning()));
        this.infoLabel.setText(String.valueOf(list.getInfo()));
        this.unknownLabel.setText(String.valueOf(list.getUnKnown()));
        this.totalLabel.setText(String.valueOf(list.getTotal()));
    }
    
}