/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.composite;

import java.rmi.AccessException;
import java.util.ArrayList;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.monitor.action.GetFilterStatus;
import com.clustercontrol.monitor.action.GetStatusListTableDefine;
import com.clustercontrol.monitor.bean.StatusInfoData;
import com.clustercontrol.monitor.bean.StatusTableDefine;
import com.clustercontrol.util.Messages;

/**
 * ステータス情報一覧のコンポジットクラス<BR>
 * 
 *ステータス情報一覧部分のテーブルのコンポジット
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class StatusListComposite extends Composite {

    /** テーブルビューア */
    private CommonTableViewer tableViewer = null;

    /** 危険ラベル */
    private Label criticalLabel = null;

    /** 警告ラベル */
    private Label warningLabel = null;

    /** 通知ラベル */
    private Label infoLabel = null;

    /** 不明ラベル */
    private Label unknownLabel = null;

    /** 合計ラベル */
    private Label totalLabel = null;
    
    private boolean openAccessInformation = false;


    /**
     * インスタンスを返します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public StatusListComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを配置します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(5, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 5;
        table.setLayoutData(gridData);

        // ステータス作成

        // 危険
        this.criticalLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.criticalLabel.setLayoutData(gridData);
        this.criticalLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_RED));

        // 警告
        this.warningLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.warningLabel.setLayoutData(gridData);
        this.warningLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_YELLOW));

        // 通知
        this.infoLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.infoLabel.setLayoutData(gridData);
        this.infoLabel.setBackground(this.getDisplay().getSystemColor(
                SWT.COLOR_GREEN));

        // 不明
        this.unknownLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.unknownLabel.setLayoutData(gridData);
        this.unknownLabel.setBackground(new Color(null, 128, 192, 255));

        // 合計
        this.totalLabel = new Label(this, SWT.CENTER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.totalLabel.setLayoutData(gridData);

        // テーブルビューアの作成
        this.tableViewer = new CommonTableViewer(table);
        this.tableViewer.createTableColumn(GetStatusListTableDefine.getStatusListTableDefine(),
                StatusTableDefine.SORT_COLUMN_INDEX,
                StatusTableDefine.SORT_ORDER);

        for (int i = 0; i < table.getColumnCount(); i++){
        	table.getColumn(i).setMoveable(true);
        } 
 
    }

    /**
     * このコンポジットが利用する共通テーブルビューアーを返します。
     * 
     * @return 共通テーブルビューアー
     */
    public CommonTableViewer getTableViewer() {
        return this.tableViewer;
    }
    
    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return this.tableViewer.getTable();
    }

    /**
     * ビューを更新します。<BR>
     * 引数で指定されたファシリティの配下全てのファシリティのステータス一覧情報を取得し、
     * 共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティに属するステータス情報一覧を取得します。</li>
	 * <li>共通テーブルビューアーにステータス情報一覧をセットします。</li>
	 * </ol>
     * 
     * @param facilityId 表示対象の親ファシリティID
     * 
     * @see com.clustercontrol.monitor.action.GetFilterStatus#getFilterInfo(String)
     * @see #updateStatus(ArrayList)
     */
    public void update(String facilityId) {
        super.update();

        ArrayList infoList = null;
		try {
			infoList = new GetFilterStatus().getFilterInfo(facilityId);
		} catch (AccessException e) {
			if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
		}
		
        if (infoList == null) {
            infoList = new ArrayList();
        }
        
        this.updateStatus(infoList);
        Object[] args = { this.totalLabel.getText() };
        this.totalLabel.setText(Messages.getString("records", args));

        tableViewer.setInput(infoList);
    }

    /**
     * ビューを更新します。<BR>
     * 引数で指定された条件に一致するステータス情報一覧を取得し、共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数で指定された条件に一致するステータス情報一覧を取得します。</li>
	 * <li>共通テーブルビューアーにステータス情報一覧をセットします。</li>
	 * </ol>
     * 
     * @param facilityId 表示対象の親ファシリティID
     * @param condition 検索条件
     * 
     * @see com.clustercontrol.monitor.action.GetFilterEvent#getFilterInfo(String, Property, int)
     * @see #updateStatus(ArrayList)
     */
    public void update(String facilityId, Property condition) {
        super.update();

        ArrayList infoList = null;
		try {
			infoList = new GetFilterStatus().getFilterInfo(facilityId, condition);
		} catch (AccessException e) {
			if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
		}
		
		if (infoList == null) {
            infoList = new ArrayList();
        }

        this.updateStatus(infoList);
        Object[] args = { this.totalLabel.getText() };
        this.totalLabel.setText(Messages.getString("filtered.records", args));

        tableViewer.setInput(infoList);

    }

    /**
     * ステータスラベルを更新します。<BR>
     * 引数で指定されたステータス情報一覧より、重要度ごとの件数，全件数を取得し、
     * ステータスラベルを更新します。
     * 
     * @param list ステータス情報一覧
     */
    protected void updateStatus(ArrayList list) {
        int[] status = new int[4];

        if (list != null) {
            int count = list.size();

            for (int i = 0; i < count; i++) {
            	ArrayList data = (ArrayList) list.get(i);
            	int value = ((Integer) data.get(0)).intValue();

                switch (value) {
                case PriorityConstant.TYPE_CRITICAL:
                    status[0]++;
                    break;
                case PriorityConstant.TYPE_WARNING:
                    status[1]++;
                    break;
                case PriorityConstant.TYPE_INFO:
                    status[2]++;
                    break;
                case PriorityConstant.TYPE_UNKNOWN:
                    status[3]++;
                    break;
                }
            }
        }

        // ラベル更新
        this.criticalLabel.setText(String.valueOf(status[0]));
        this.warningLabel.setText(String.valueOf(status[1]));
        this.infoLabel.setText(String.valueOf(status[2]));
        this.unknownLabel.setText(String.valueOf(status[3]));
        int total = status[0] + status[1] + status[2] + status[3];
        this.totalLabel.setText(String.valueOf(total));
    }
      
}