/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.util;

import java.rmi.RemoteException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoData;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoUtil;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoLocal;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoUtil;

/**
 * ステータス情報を更新するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class OutputStatus extends InhibitAction{

	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OutputStatus.class );

	
	public synchronized void outputStatus(OutputNotifyGroupInfo outputInfo) throws FinderException, NamingException {
		NotifyInfoLocal notifyInfo = NotifyInfoUtil.getLocalHome().findByPrimaryKey(outputInfo.getNotifyId());
		
		NotifyStatusInfoLocal statusInfo = NotifyStatusInfoUtil.getLocalHome().findByPrimaryKey(
				new NotifyStatusInfoPK(outputInfo.getNotifyId(), outputInfo.getPriority()));
		
		if(statusInfo == null){
			m_log.debug("onMessage() : 出力する重要度が存在しません。");
			return;
		}
		
		// 対象の重要度に通知フラグたっていた場合
		if(statusInfo.getStatusFlg() == ValidConstant.TYPE_VALID) {
		
			/**
			 * 実行するかどうかの判定
			 */
			boolean isRun = false;
			
			// 抑制する場合（該当の重要度に抑制フラグがあり、抑制方法が「なし」以外の場合）
			if (statusInfo.getStatusInhibitionFlg().intValue() == ValidConstant.TYPE_VALID &&
					notifyInfo.getInhibitionFlg().intValue() != ExclusionConstant.TYPE_NO) {
				
				try {
					boolean isInhibit = super.isInhibit(outputInfo,	notifyInfo);
					
					// 抑制しない場合
					if(!isInhibit){
						isRun = true;
					}
					
				} catch (NumberFormatException e) {
					m_log.error("outputStatus() : " + e.getMessage(),e);
				} catch (RemoteException e) {
					m_log.error("outputStatus() : " + e.getMessage(),e);
				} catch (CreateException e) {
					m_log.error("outputStatus() : " + e.getMessage(),e);
				}	
			}
			// 抑制しない場合（上記以外）
			else {
				isRun = true;
			}
			
			
			/**
			 * 実行
			 */
			if(isRun){
				this.outputStatusInfo(outputInfo, statusInfo);
			}
			
		}
	}
	
	/**
	 * ステータス情報を出力します。<BR>
	 * 同じステータス情報が存在しない場合は、ステータス情報を作成します。
	 * 同じステータス情報がすでに存在する場合は、ステータス情報を更新します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @return 成功した場合は、 <code>true</code>
	 */
	public boolean outputStatusInfo(OutputNotifyGroupInfo output, NotifyStatusInfoLocal statusInfo) {
		
		boolean result = false;
		
		if(statusInfo.getStatusFlg().intValue() == ValidConstant.TYPE_VALID)
		{
			StatusInfoLocal outputStatus = null;
			boolean insertFlg = false;
			
			try
			{
				// エンティティ情報の検索
				StatusInfoPK pk = new StatusInfoPK(output.getMonitorId(), output.getPluginId(),output.getFacilityId());
				outputStatus = StatusInfoUtil.getLocalHome().findByPrimaryKey(pk);
			}
			catch (NamingException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(FinderException e)
			{
				// 検索条件に合致するCMPエンティティが存在しない
				insertFlg = true;
			}
			
			try
			{
				// エンティティ情報の生成
				if(insertFlg)
				{
					insertStatusInfo(output, statusInfo);
					result = true;
				}
				//　エンティティ情報の更新
				else
				{
					updateStatusInfo(outputStatus, output, statusInfo);
					result = true;
				}
			}
			catch(NamingException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(CreateException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * ステータス情報を作成します。
	 * 
	 * @param outputInfo 出力情報
	 * @param statusInfo 通知情報
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void  insertStatusInfo(OutputNotifyGroupInfo outputInfo, NotifyStatusInfoLocal statusInfo) throws NamingException, CreateException{
	
		try
		{
			// 有効期限制御フラグ
			Integer expirationFlg = getExpirationFlg(statusInfo);
			// 有効期限日時
			Timestamp expirationDate = null;
			
			// 有効期限日時
			if(expirationFlg != null){
		        expirationDate = getExpirationDate(outputInfo, statusInfo);
			}
			
			StatusInfoUtil.getLocalHome().create(
					outputInfo.getFacilityId(),
					outputInfo.getMonitorId(),
					outputInfo.getPluginId(),
					outputInfo.getApplication(),
					outputInfo.getMessageId(),
					outputInfo.getMessage(),
					new Integer(outputInfo.getPriority()),
					new java.sql.Timestamp(outputInfo.getGenerationDate().getTime()),
					outputInfo.getOutputDate(),
					expirationFlg,
					expirationDate);
		}
		catch(NamingException e)
		{
			throw e;
		}
		catch(CreateException e)
		{
			throw e;
		}
	}
	
	/**
	 * ステータス情報を更新します。
	 * 
	 * @param outputStatus ステータス情報のローカルコンポーネントインターフェース
	 * @param output 出力情報
	 * @param statusInfo 通知情報
	 * @throws EJBException
	 */
	public void updateStatusInfo(StatusInfoLocal outputStatus, OutputNotifyGroupInfo output, NotifyStatusInfoLocal statusInfo) throws EJBException{
	
		if(statusInfo != null)
		{
			try
			{
				// 有効期限制御フラグ
				Integer expirationFlg = getExpirationFlg(statusInfo);
				// 有効期限日時
				Timestamp expirationDate = null;
				// 有効期限日時
				if(expirationFlg != null){
			        expirationDate = getExpirationDate(output, statusInfo);
				}
				
				outputStatus.setApplication(output.getApplication());
				outputStatus.setMessageId(output.getMessageId());
				outputStatus.setMessage(output.getMessage());
			
				// 重要度が変更されていた場合、出力日時を更新する
				if (outputStatus.getPriority().intValue() != output.getPriority()) {
					outputStatus.setGenerationDate(new java.sql.Timestamp(output.getGenerationDate().getTime()));
				}
				
				outputStatus.setPriority(new Integer(output.getPriority()));				
				outputStatus.setOutputDate(output.getOutputDate());
				outputStatus.setExpirationFlg(expirationFlg);
				outputStatus.setExpirationDate(expirationDate);
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				throw e;
			}
		}

	}
	
	/**
	 * 存続期間経過後の処理の制御フラグを返します。
	 * 
	 * @param statusInfo
	 * @see com.clustercontrol.bean.StatusExpirationConstant
	 * @return 存続期間経過後の処理の制御フラグ
	 */
	private Integer getExpirationFlg(NotifyStatusInfoLocal statusInfo){

		// 有効期限制御フラグ
		Integer expirationFlg = null;

		if(statusInfo != null)
		{
			if(StatusExpirationConstant.TYPE_DELETE  == statusInfo.getStatusInvalidFlg()){
				expirationFlg = new Integer(statusInfo.getStatusInvalidFlg());
			}
			if(StatusExpirationConstant.TYPE_UPDATE  == statusInfo.getStatusInvalidFlg()){
				// 有効期間経過後の扱いが、更新されていない旨のメッセージに置き換える場合は、
				// 置換え後の重要度を設定する
				expirationFlg = new Integer(statusInfo.getStatusUpdatePriority());	
			}
		}
		return expirationFlg;
	}
	
	/**
	 * 存続期間日時を返します。
	 * 
	 * @param outputInfo 出力情報
	 * @param statusInfo 通知情報
	 * @return 存続期間日時
	 */
	private Timestamp getExpirationDate(OutputNotifyGroupInfo outputInfo, NotifyStatusInfoLocal statusInfo){

		// 存続期間日時
		Timestamp expirationDate = null;

		if(outputInfo != null && statusInfo != null)
		{
			if(statusInfo.getStatusValidPeriod() > 0){
				Calendar cal = null;
				cal = Calendar.getInstance();
				cal.setTime(outputInfo.getOutputDate());
		        cal.add(Calendar.MINUTE, statusInfo.getStatusValidPeriod());
		        expirationDate = new Timestamp(cal.getTimeInMillis());
			}
		}
		return expirationDate;
	}
}
