/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifySubstitutionParameterConstant;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoUtil;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.notify.mail.ejb.session.MailTemplateControllerUtil;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.util.Messages;

/**
 * メールを送信するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class SendMail extends InhibitAction{
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SendMail.class );
	
	/** 件名用日時フォーマット。 */
	private static final String SUBJECT_DATE_FORMAT = "yyyy/MM/dd HH:mm:ss";
	
	/** 差出人アドレス。 */
	static private String FROM_ADDRESS = null;
	/** 差出人個人名。 */
	static private String FROM_PERSONAL_NAME = null;
	/** 返信の送信先アドレス。 */
	static private String REPLY_TO_ADDRESS = null;
	/** 返信の送信先個人名。 */
	static private String REPLY_TO_PERSONAL_NAME = null;
	/** エラー送信先アドレス。 */
	static private String ERRORS_TO_ADDRESS = null;
	/** 再送回数。 */
	static private int RETRY_COUNT = 0;
	/** ロケール情報。 */
	private Locale m_local =Locale.getDefault();
	
	static{
		
		
		/**
		 * 接続先をプロパティファイルから取得
		 * 
		 */
		Properties properties = new Properties();
		
		//FIXME RuntimeのExceptionが発生する恐れがあるためtry-catchとする
		try {
			String etcdir = System.getProperty("hinemos.manager.etc.dir");
			String propertyFile = etcdir + File.separator + "hinemos.properties";
			
			// プロパティファイルからキーと値のリストを読み込みます
			properties.load(new FileInputStream(propertyFile));
			
			FROM_ADDRESS = properties.getProperty("common.mail.from.address");
			FROM_PERSONAL_NAME = properties.getProperty("common.mail.from.personal.name");
			REPLY_TO_ADDRESS = properties.getProperty("common.mail.reply.to.address");
			REPLY_TO_PERSONAL_NAME = properties.getProperty("common.mail.reply.personal.name");
			ERRORS_TO_ADDRESS = properties.getProperty("common.mail.errors.to.address");
			RETRY_COUNT = Integer.parseInt(properties.getProperty("common.mail.retry.count"));
			
			
		} catch (Exception e) {
		}
		
		
	}
	 
	/**
	 * メールの送信を行います。
	 * 
	 * @param outputInfo 出力・通知情報
	 * @throws UnsupportedEncodingException
	 * @throws RemoteException
	 * @throws NamingException
	 * @throws MessagingException
	 * @throws CreateException
	 * @throws FinderException
	 */

	public synchronized void sendMail(OutputNotifyGroupInfo outputInfo) throws UnsupportedEncodingException, RemoteException, NamingException, MessagingException, CreateException, FinderException {
		NotifyInfoLocal notifyInfo = NotifyInfoUtil.getLocalHome().findByPrimaryKey(outputInfo.getNotifyId());
		
		NotifyMailInfoLocal mailInfo = NotifyMailInfoUtil.getLocalHome().findByPrimaryKey(
				new NotifyMailInfoPK(outputInfo.getNotifyId(), outputInfo.getPriority()));
		
		if(mailInfo == null){
			m_log.debug("onMessage() : 出力する重要度が存在しません。");
			return;
		}
		
		// 対象の重要度に通知フラグたっていた場合
		if (mailInfo.getMailFlg() == ValidConstant.TYPE_VALID){
			
			/**
			 * 実行するかどうかの判定
			 */
			boolean isRun = false;

			// 抑制する場合（該当の重要度に抑制フラグがあり、抑制方法が「なし」以外の場合）
			if (mailInfo.getMailInhibitionFlg().intValue() == ValidConstant.TYPE_VALID &&
					mailInfo.getMailInhibitionFlg().intValue() != ExclusionConstant.TYPE_NO){
				try {
					boolean isInhibit = super.isInhibit(outputInfo,	notifyInfo);
					
					// 抑制しない場合
					if(!isInhibit){
						isRun = true;
					}
					
				} catch (NumberFormatException e) {
					m_log.error("sendMail() : " + e.getMessage(),e);
				} catch (RemoteException e) {
					m_log.error("sendMail() : " + e.getMessage(),e);
				} catch (CreateException e) {
					m_log.error("sendMail() : " + e.getMessage(),e);
				}	
			}
			// 抑制しない場合（上記以外）
			else {
				isRun = true;
			}
			
			/**
			 * 実行
			 */
			if(isRun){				
				String address = mailInfo.getMailAddress();
				StringTokenizer t = new StringTokenizer(address, ";");
				ArrayList<String> addressList = new ArrayList<String>();
				while (t.hasMoreTokens()) {
					addressList.add(t.nextToken());
				}
				String[] addresses = (String[]) addressList.toArray(new String[0]);
	
				if(addresses == null || addresses.length<=0) {
					m_log.debug("onMessage() : 通知するメールアドレスが存在しません。");
					return;
				}
				
				this.sendMail(addresses, outputInfo, mailInfo);
			}
		}
	}
	
	
	
	/**
	 * メールを送信します。
	 * 
	 * <p>下記の情報は、ファイルより取得します。
	 * <p>
	 * <ul>
	 *  <li>差出人アドレス</li>
	 *  <li>差出人個人名</li>
	 *  <li>返信の送信先アドレス</li>
	 *  <li>返信の送信先個人名</li>
	 *  <li>エラー送信先アドレス</li>
	 * </ul>
	 * 
	 * @param address 送信先アドレス
	 * @param source 出力内容
	 * @param mailInfo 通知内容
	 * @return 送信に成功した場合、<code> true </code>
	 * @throws NamingException
	 * @throws MessagingException
	 * @throws UnsupportedEncodingException
	 * @throws FinderException 
	 * @throws CreateException 
	 * @throws RemoteException 
	 */
	public boolean sendMail(String[] address, OutputNotifyGroupInfo source, NotifyMailInfoLocal mailInfo) throws NamingException, MessagingException, UnsupportedEncodingException, CreateException, FinderException, RemoteException{
		
		if(address == null || address.length<=0)
		{
			return false;
		}
		
		Context initialContext = null;
		
		try
		{
			initialContext = new InitialContext();

			// 再送信フラグ設定
			boolean retryFlg = true;
			
			// JavaMail Sessionリソース検索
			Session session=(Session)initialContext.lookup("java:/Mail");
			
			Message mes = new MimeMessage(session);
			
			
            // 送信元メールアドレスと送信者名を指定
            if(FROM_ADDRESS != null && FROM_PERSONAL_NAME != null){
                mes.setFrom(new InternetAddress(FROM_ADDRESS, FROM_PERSONAL_NAME,"iso-2022-jp"));
            }else if(FROM_ADDRESS != null && FROM_PERSONAL_NAME == null){
                mes.setFrom(new InternetAddress(FROM_ADDRESS));
            }
            // REPLY-TOを指定
            if(REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME != null){
		        InternetAddress reply[] = {new InternetAddress(REPLY_TO_ADDRESS,REPLY_TO_PERSONAL_NAME,"iso-2022-jp")};
		        mes.setReplyTo(reply);
		        mes.reply(true);
            }else if(REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME == null){
		        InternetAddress reply[] = {new InternetAddress(REPLY_TO_ADDRESS)};
		        mes.setReplyTo(reply);
		        mes.reply(true);
            }


            // ERRORS-TOを指定
            if(ERRORS_TO_ADDRESS != null ){
                mes.setHeader("Errors-To", ERRORS_TO_ADDRESS);
            }

			// 送信先メールアドレスを指定
            InternetAddress[] toAddress = this.getAddress(address);
            if(toAddress == null || toAddress.length<=0)
            {
            	return false;
            }
			mes.setRecipients(javax.mail.Message.RecipientType.TO, toAddress);
			
			// メールの件名を指定
			mes.setSubject(MimeUtility.encodeText(this.getSubject(source, mailInfo),"iso-2022-jp","B"));
			
			// メールの内容を指定
			mes.setContent(this.getText(source, mailInfo), "text/plain; charset=iso-2022-jp");
			
			// 送信日付を指定
			mes.setSentDate(this.getSentDate(source.getOutputDate()));
			
			// 再送信フラグがtrueかつ再送回数以内の場合			
			for (int i = 0; i < RETRY_COUNT+1 && retryFlg; i++) {
				try {
					// メール送信
					Transport.send(mes);
					retryFlg = false;
				}
				catch(MessagingException me) {
					// メール再送信フラグを挙げる
					retryFlg = true;
					if(i < RETRY_COUNT ) {
						m_log.debug("sendMail() : メール再送信処理");
					}
					else {
						throw me;
					}
					
				}
			}
			
			return true;
		}
		
		//FIXME NotifyMailBeanにて処理をするように例外をスローする
		catch (NamingException e) {
			throw e;
		}
		catch(MessagingException e) {
			throw e;
		}
		catch(UnsupportedEncodingException e) {
			throw e;
		}
		catch(RemoteException e){
			throw e;
		}
		//FIXME falseでハンドリングをしていない
		
	}
	
	
	/**
	 * 引数で指定された送信先アドレスの<code> InternetAddress </code>オブジェクトを返します。
	 * 
	 * @param address 送信先アドレスの文字列配列
	 * @return <code> InternetAddress </code>オブジェクトの配列
	 */
	private InternetAddress[] getAddress(String[] address) {
		
		InternetAddress toAddress[] = null;
		Vector<InternetAddress> list = new Vector<InternetAddress>();
		
		if(address != null)
		{
			
            for (int index=0; index<address.length; index++) 
            {
            	try
				{
            		list.add(new InternetAddress(address[index]));
				}
            	catch(AddressException ae)
				{
				}
            }
            
            if(list.size() > 0)
            {
            	toAddress = new InternetAddress[list.size()];
                list.copyInto(toAddress);	
            }
		}
		return toAddress;
	}
	
	/**
	 * メール件名を返します。
	 * 
	 * @param source 出力内容
	 * @param mailInfo 通知内容
	 * @return メール件名
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws RemoteException 
	 */
	private String getSubject(OutputNotifyGroupInfo source, NotifyMailInfoLocal mailInfo) throws CreateException, NamingException, FinderException, RemoteException {
		
		String subject = null;
		
		if(source instanceof OutputNotifyGroupInfo)
		{
			try {
				if (mailInfo.getMailTemplateId() != null) 
				{
					MailTemplateInfoData mailData = MailTemplateControllerUtil.getLocalHome().create().getMailTemplateInfo(mailInfo.getMailTemplateId());
					subject = substitution(mailData.getSubject(), source);
						
				}
				else 
				{					
					subject = Messages.getString("mail.subject", m_local) + "(" + PriorityConstant.typeToString(source.getPriority()) + ")";
				}
			}
			//FIXME NotifyMailBeanにて処理をするように例外をスローする
			catch (CreateException e) {
				throw(e);
			} catch (NamingException e) {
				throw(e);
			} catch (FinderException e) {
				throw(e);
			}
		}
		return subject;
	}
	
	/**
	 * メール本文を返します。
	 * 
	 * @param source 出力内容
	 * @param mailInfo 通知内容
	 * @return メール本文
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws RemoteException 
	 */
	private String getText(OutputNotifyGroupInfo source, NotifyMailInfoLocal mailInfo) throws CreateException, NamingException, FinderException, RemoteException {
		
		StringBuffer buf = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);
		
		if(source instanceof OutputNotifyGroupInfo)
		{
			try {
				if (mailInfo.getMailTemplateId() != null) 
				{
					MailTemplateInfoData mailData = MailTemplateControllerUtil.getLocalHome().create().getMailTemplateInfo(mailInfo.getMailTemplateId());
					buf.append(substitution(mailData.getBody(), source));
				}
				else
				{
					buf.append(Messages.getString("generation.time",m_local) + " : " +  sdf.format(source.getGenerationDate()) + "\n");
					buf.append(Messages.getString("application",m_local) + " : " +  source.getApplication() + "\n");
					buf.append(Messages.getString("priority",m_local) + " : " +  PriorityConstant.typeToString(source.getPriority()) + "\n");
					buf.append(Messages.getString("message",m_local) + " : " +  source.getMessage() + "\n");
					buf.append(Messages.getString("scope",m_local) + " : " +  source.getScopeText() + "\n");
				}
			}
			//FIXME NotifyMailBeanにて処理をするように例外をスローする
			catch (CreateException e) {
				throw(e);
			} catch (NamingException e) {
				throw(e);
			} catch (FinderException e) {
				throw(e);
			}
		}
		return buf.toString();
	}
	
	/**
	 * 引数で指定された文字列内の変数を置換します。
	 * 
	 * @param text 変換対象となるtext
	 * @param info 出力内容
	 * @return 変換後のtext
	 */
	public static synchronized String substitution(String text, OutputNotifyGroupInfo info) {
		
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);
		
		// 結果格納用
		String retText;
		
		retText = text.replaceAll(NotifySubstitutionParameterConstant.APPLICATION_REGEX, toLiteral(info.getApplication()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.FACILITY_ID_REGEX, toLiteral(info.getFacilityId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MONITOR_ID_REGEX, toLiteral(info.getMonitorId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.GENERATION_DATE_REGEX, toLiteral(sdf.format(info.getGenerationDate())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PLUGIN_ID_REGEX, toLiteral(info.getPluginId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_REGEX, toLiteral(PriorityConstant.typeToString(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_JP_REGEX, toLiteral(PriorityConstant.typeToStringJP(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_EN_REGEX, toLiteral(PriorityConstant.typeToStringEN(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_NUM_REGEX, toLiteral(String.valueOf(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.SCOPE_REGEX, toLiteral(info.getScopeText()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MESSAGE_ID_REGEX, toLiteral(info.getMessageId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MESSAGE_REGEX, toLiteral(info.getMessage()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.ORG_MESSAGE_REGEX, toLiteral(info.getMessageOrg()));
		
		// IPアドレスとホスト名の置換
		try {
			// 置換元となっている文章に、IPアドレスとノード名の置換用文字列が存在する場合
			if(retText.indexOf(NotifySubstitutionParameterConstant.IP_ADDRESS) != -1
					|| retText.indexOf(NotifySubstitutionParameterConstant.NODE) != -1){

				String nodeName = null;
				String ipNetworkNumber = null;
				
				String facilityId = info.getFacilityId();
				
				if(facilityId != null && !facilityId.equals("")){
				
					FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
	
					// 対象ファシリティIDがノードである場合
					if(facility != null){
	
						// ノード名の取得
						nodeName = facility.getNodeName();
						
						// IPアドレスの取得
						int version = 4;
						if(facility.getIpProtocolNumber() != null){
							version = facility.getIpProtocolNumber();
						}
						
						// 入力されているバージョンを比較し、対応するIPアドレスを取得する
						if(version == 4) {
							ipNetworkNumber = facility.getIpNetworkNumber();
						}
						else {
							ipNetworkNumber = facility.getIpNetworkNumberV6();
						}
					}
				}
				
				// ノード名の置換
				if(nodeName != null && !nodeName.equals("")) {
					retText = retText.replaceAll(NotifySubstitutionParameterConstant.NODE_REGEX, toLiteral(nodeName));
				}
				// IPアドレスの置換
				if(ipNetworkNumber != null && !ipNetworkNumber.equals("")) {
					retText = retText.replaceAll(NotifySubstitutionParameterConstant.IP_ADDRESS_REGEX, toLiteral(ipNetworkNumber));
				}

			} 

		}
		catch (NamingException e) {
			m_log.error("substitution() : "+ e.getMessage(), e);
		} 
		catch (FinderException e) {
			m_log.error("substitution() : "+ e.getMessage(), e);
		}
		
		return retText;
	}
	
	/**
	 * IllegalArgumentExceptionを回避するため禁止文字($,\)をエスケープします。
	 * 
	 * @param text エスケープチェック対象となる文字列
	 * @return チェック後の文字列
	 */
    private static String toLiteral(String text) {
    	
    	if(text == null) {
    		return "";
    	}
    	
        if (text.indexOf('\\') == -1 && text.indexOf('$') == -1) {
            return text;
        }

        StringBuffer buffer = new StringBuffer();
        for (int i=0; i < text.length(); i++) {
            char c = text.charAt(i);
            if (c == '\\') {
                buffer.append('\\');
                buffer.append('\\');
            } else if (c == '$') {
                buffer.append('\\');
                buffer.append('$');
            } else {
                buffer.append(c);
            }
        }
        
        return buffer.toString();
    }	
	
	
	/**
	 * 送信日付を返します。<BR>
	 * 引数の日付が<code> null </code>の場合、現在時刻を返します。
	 * 
	 * @param timeStamp 日付
	 * @return 送信日付
	 */
	private Date getSentDate(Date timeStamp) {
		
		if(timeStamp == null)
		{
			timeStamp = new Date();	
		}
		return timeStamp;
	}
	
	
}
