/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller.jmx;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;

import com.clustercontrol.poller.snmp.SnmpPollerManager;
import com.clustercontrol.poller.util.ReceiveRepositoryTopic;


/**
 * @jmx.mbean
 *     name="user:service=SnmpPollerService,name=SnmpPollerService"
 *     description="SnmpPollerService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="snmppoller"
 */
public class SnmpPollerService extends ServiceMBeanSupport implements SnmpPollerServiceMBean{
	private String jndiName = "SnmpPollerManager";  // JNDIに登録する際の名前
	private String quartzJndiName = null;  // Quartzをlookupする際のJNDI名
	
	private SnmpPollerManager manager;
	
	// リポジトリ変更時にメッセージを受け取りノード情報を変更するクラス
	private ReceiveRepositoryTopic m_receiveRepositoryTopic;
	
    /**
     * @jmx.managed-attribute
     * @return
     */
    public String getJndiName() {
        return jndiName;
    }
    
    /**
     * @jmx.managed-attribute
     * 
     * @param jndiName
     * @throws NamingException 
     * @throws NamingException
     */
    public void setJndiName(String jndiName) throws NamingException {
        log.info("setJndiName() SnmpService : " + jndiName);
    	
        String oldName = this.jndiName;
        this.jndiName = jndiName;
        
        if (super.getState() == STARTED) {
        	try {
        		unbind(oldName);
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to unbind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}

        	try{
        		rebind();
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to rebind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}
        }
    }
    
	/**
	 * QuartzのサービスのJNDI名を返します。
	 * @return QuartzサービスのJNDI名
	 *
     * @jmx.managed-attribute
	 */
	public String getQuartzJndiName() {
		return quartzJndiName;
	}

	/**
	 * QuartzのサービスのJNDI名を設定します。
	 * @param jndiName QuartzサービスのJNDI名
	 * 
     * @jmx.managed-attribute
   	 */
	public void setQuartzJndiName(String jndiName) {
		quartzJndiName = jndiName;
	}
	
    /**
     * @jmx.managed-attribute
     */
    public String getName() {
        return "SnmpPollerService(" + jndiName + ")";
    }
    
    /**
     * @jmx.managed-operation
     */
    public String printPollerInfo() {
        return manager.getPollerDebugInfo();
    }
        
    public void createService() {
        log.info("Create SnmpPollerService(" + jndiName + ")");
 
        manager = new SnmpPollerManager(jndiName);

        log.info("Created SnmpPollerService(" + jndiName + ")");
    }
    
    public void startService() throws NamingException {
        log.info("Start SnmpPollerService(" + jndiName + ")");
        
        // Topicを接続しリポジトリ変更にメッセージを受け取るようにする
        m_receiveRepositoryTopic = new ReceiveRepositoryTopic(manager);
        
    	rebind();
    	
        log.info("Started SnmpPollerService(" + jndiName + ")");
    }
    
    public void stopService() throws NamingException {
        log.info("Stop SnmpPollerService(" + jndiName + ")");

        // Topicの接続を切断
        m_receiveRepositoryTopic.terminate();
        
        unbind(jndiName);

        log.info("Stoped SnmpPollerService(" + jndiName + ")");
    }
    
    public void destroyService() throws Exception {
        log.info("Destroy SnmpPollerService(" + jndiName + ")");
        
        manager = null;

        log.info("Destroyed SnmpPollerService(" + jndiName + ")");
    }
    
    private static Context createContext(Context rootCtx, Name name) throws NamingException {
    	Context subctx = rootCtx;
    	
    	for (int n = 0; n < name.size(); n++) {
    		String atom = name.get(n);
    		
    		try {
    			Object obj = subctx.lookup(atom);
    			subctx = (Context) obj;
    		} catch (NamingException e) {
    			// 存在しない場合は、サブコンテキストを生成
    			subctx = subctx.createSubcontext(atom);
    		}
    	}
    	
    	return subctx;
    }
    
    private void rebind() throws NamingException {
    	InitialContext rootCtx = new InitialContext();
    	
    	Name fullName = rootCtx.getNameParser("").parse(jndiName);
    	Name parentName = fullName;
    	if(fullName.size() > 1){
    		parentName = fullName.getPrefix(fullName.size()-1);
    	} else {
    		parentName = new CompositeName();
    	}
    	
    	Context parentCtx = createContext(rootCtx, parentName);
    	Name atomName = fullName.getSuffix(fullName.size()-1);
    	String atomStirng = atomName.get(0);
    	
    	NonSerializableFactory.rebind(parentCtx, atomStirng, manager);
    }
    
    private void unbind(String jndiName) throws NamingException {
    	InitialContext rootCtx = null;
    	
    	try {
    		rootCtx = new InitialContext();
    		
    		Name fullName = rootCtx.getNameParser("").parse(jndiName);
    		Name atomName = fullName.getSuffix(fullName.size() - 1);
    		String atom = atomName.get(0);
    		
    		rootCtx.unbind(jndiName);
    		NonSerializableFactory.unbind(atom);
    	} finally {
    		if(rootCtx != null) { 
    			rootCtx.close(); 
    		}
    	}
    }
}
