/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.sharedtable;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

/**
 * キーとオブジェクトのマッピングを保持するクラス
 * 
 * @version 3.0.0
 * @since 2.0.0
 */
public class DataTable implements Cloneable {
	private AtomicReference<ConcurrentHashMap<String, TableEntry>> m_hm;   // キーと値オブジェクトのマッピングを保持するマップ
	
	private volatile long lastModify;
	
	/**
	 * 新しい空のテーブルを作成します。
	 */
	public DataTable(){
	}
	
	/**
	 * 指定された値と指定されたキーをこのテーブルに関連付けます。
	 * 
	 * @param key 指定される値が関連付けられるキー
	 * @param date 指定される値に関連付けられる時刻
	 * @param value 指定されるキーに関連付けられる値
	 */
	public void putValue(String key, long date, Object value) {
		if(m_hm == null){
			m_hm = new AtomicReference<ConcurrentHashMap<String, TableEntry>>
			(new ConcurrentHashMap<String, TableEntry>());
		}
		
		m_hm.get().put(key, new TableEntry(key, date, value));
		lastModify = System.currentTimeMillis();
	}

//	/**
//	 * テーブルで保持していた値を全て削除し、
//	 * 指定されたテーブルで全ての値を書き換えます。
//	 * 
//	 * @param table 格納する値のセットが格納されているテーブル
//	 */
//	public void setValueAll(DataTable table) {
//		m_hm.set(table.m_hm.get());
//		lastModify = System.currentTimeMillis();
//	}
	
	/**
	 * 指定されたキーにマップされている値を返します。
	 * 
	 * @param key 関連付けられた値が返されるキー
	 * @return 指定されたキーにマッピングしている値オブジェクト。
	 *          このキーに対する値オブジェクがテーブルにない場合は null
	 */
	public TableEntry getValue(String key){
		// 生成された後値が一度もセットされていない場合
		if(m_hm == null){
			return null;
		}
		
		TableEntry entry = m_hm.get().get(key);
		
		return entry;
	}

	/**
	 * キー（文字列）にマッピングされている値のうち、
	 * 指定された接頭辞で始まるキーにマッピングされている値のセットを返します。
	 * 
	 * @param prefix 接頭辞
	 * @return 指定された接頭辞で始まるキーで取得できる値のセット
	 * 値を保持しているが指定の接頭辞で始まるキーのものが存在しない場合は空のセットを返す
	 * 値をまったく保持していない場合は、nullを返す
	 */
	public Set<TableEntry> getValueSetStartWith(String prefix) {
		Set<TableEntry> set = new HashSet<TableEntry>();

		// 生成された後値が一度もセットされていない場合
		if(m_hm == null){
			return null;
		}
		
		// キーとされているフルのOIDを取得しそのフルOIDの文字列の先頭部分が、
		// 引数指定のOIDである場合は、その値を戻りのセットに格納する
		Iterator<String> itr = m_hm.get().keySet().iterator();
		while (itr.hasNext()) {
			String key = (String) itr.next();
			
			// もし先頭の部分が一致するならそのツリーの子孫と判定
			if (key.startsWith(prefix)) {
				set.add(m_hm.get().get(key));
			}
		}
		
		return set;
	}
	
	/**
	 * 指定されたキーにマップされている値を削除します。
	 * 
	 * @param key 関連付けられた値のキー
	 * @return 指定されたキーにマッピングしている値オブジェクト。
	 *          このキーに対する値オブジェクがテーブルにない場合は null
	 */
	public TableEntry removeValue(String key){
		// 生成された後値が一度もセットされていない場合
		if(m_hm == null){
			return null;
		}
		
		TableEntry entry = m_hm.get().remove(key);
		lastModify = System.currentTimeMillis();
		return entry;
	}
	
	/**
	 * キーのセットを返します。
	 * @return キーのセット
	 */
	public Set<String> keySet(){
		// 生成された後値が一度もセットされていない場合
		if(m_hm == null){
			return new HashSet<String>();
		}
		
		return this.m_hm.get().keySet();
	}

	/**
	 * 最終更新日時を返します。
	 * @return 最終更新日時
	 */
	public long getLastModify() {
		return lastModify;
	}
}
