/*
 
Copyright (C) 2006 - 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.snmptrap.bean.MonitorListTabelDefine;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.factory.AddMibMaster;
import com.clustercontrol.snmptrap.factory.AddMonitorInfo;
import com.clustercontrol.snmptrap.factory.AddOidMaster;
import com.clustercontrol.snmptrap.factory.DeleteMibMaster;
import com.clustercontrol.snmptrap.factory.DeleteMonitorInfo;
import com.clustercontrol.snmptrap.factory.DeleteOidMaster;
import com.clustercontrol.snmptrap.factory.ModifyMibMaster;
import com.clustercontrol.snmptrap.factory.ModifyMonitorInfo;
import com.clustercontrol.snmptrap.factory.ModifyOidMaster;
import com.clustercontrol.snmptrap.factory.SelectMibMaster;
import com.clustercontrol.snmptrap.factory.SelectMonitorInfo;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleUtil;
import com.clustercontrol.util.Messages;

/**
 * SNMPトラップ監視を制御するSession Bean <BR>
 * 
 * @ejb.bean name="MonitorSnmpTrapController"
 *           jndi-name="MonitorSnmpTrapController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *     
 * @version 2.4.0
 * @since 2.1.0
 * 
 */
public abstract class MonitorSnmpTrapControllerBean implements SessionBean, CheckFacility {
	
	protected static Log m_log = LogFactory.getLog( MonitorSnmpTrapControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SNMPTRAP監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException 
	 */
	public boolean addMonitor(MonitorSnmpTrapInfo info) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// SNMPTRAP監視情報を登録
		AddMonitorInfo add = new AddMonitorInfo();
		return  add.add(info, pri.getName());
	}
	
	/**
	 * マネージャのSNMP TRAP監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean modifyMonitor(MonitorSnmpTrapInfo info) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMPTRAP監視情報を更新
		ModifyMonitorInfo modify = new ModifyMonitorInfo();
		return  modify.modify(info, pri.getName());
	}
	
	/**
	 * SQL監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean deleteMonitor(String monitorId) throws InvalidTransactionException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {

		// SNMPTRAP監視情報を削除
		DeleteMonitorInfo delete = new DeleteMonitorInfo();

		return  delete.delete(monitorId);
	}
	
	/**
	 * MIB一覧を取得します。<BR>
	 * 
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<String> getMibList() throws FinderException, NamingException {

		// MIB一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<String> list = select.getMibList();

		return list;
	}
	
	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 
	 * 
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws FinderException
	 * @throws NamingException
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws FinderException, NamingException {

		// 監視マスタ情報一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<SnmpTrapMasterInfo> list = select.getMasterList(mib);

		return list;
	}
	
	/**
	 * SNMPTRAP監視設定情報を取得します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @see com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo
	 */
	public MonitorSnmpTrapInfo getMonitorInfo(String monitorId) throws CreateException, FinderException, NamingException {

		// SNMPTRAP監視情報を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		MonitorSnmpTrapInfo info = select.getMonitorInfo(monitorId);

		return info;
	}

	/**
	 * SNMPTRAP監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり、内部のArrayListは
	 * 下記の様に格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(oids.size());
	 *  info.add(facilityPath);
	 *  info.add(monitor.getValidFlg().intValue());
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getMonitorList() throws CreateException, FinderException, NamingException {

		// SNMPTRAP監視一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList list = select.getMonitorList();

		return list;
	}
	
	/**
	 * 監視一覧テーブル定義を取得します。<BR>
	 * 
	 * ロケールをマネージャで管理するために監視一覧のダイアログの
	 * テーブル部分の表示項目と表示順をマネージャで生成します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param local
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getMonitorListTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(MonitorListTabelDefine.MONITOR_ID,
				new TableColumnInfo(Messages.getString("monitor.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));	
		tableDefine.add(MonitorListTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.SNMPTRAP_NUMBER, 
				new TableColumnInfo(Messages.getString("snmptrap.records", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 200, SWT.LEFT));
//		tableDefine.add(MonitorListTabelDefine.CALENDAR_ID, 
//				new TableColumnInfo(Messages.getString("calendar", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.VALID_FLG, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		
		return tableDefine;
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="SnmptrapMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			// ファシリティIDが監視対象になっている監視情報を取得する。
			Collection ct = SnmpTrapInfoUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_SNMPTRAP_MONITOR);
			}
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
	
	
	/**
	 * MIBの内容を取得します。<BR>
	 * 
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws FinderException
	 * @throws NamingException
	 * @throws  
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws FinderException, NamingException {

		SelectMibMaster  selector = new SelectMibMaster(); 
		
		return selector.getMibMasterData(Mib);
	}
	
	/**
	 * MIB情報の登録
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 *
	 * @param MibData
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		AddMibMaster adder = new AddMibMaster();
		
		return adder.add(MibData , pri.getName());
	}
	
	/**
	 * MIB情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param MibData
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
	Principal pri = m_context.getCallerPrincipal();
		
		ModifyMibMaster modifier = new ModifyMibMaster();
		
		return modifier.modify(MibData , pri.getName());
	}
	
	/**
	 * MIB情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param mib
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean deleteMibMaster(String Mib) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
	Principal pri = m_context.getCallerPrincipal();
		
		DeleteMibMaster eraser= new DeleteMibMaster();
		
		return eraser.delete(Mib);
	}
	
	
	/**
	 * OID情報の登録
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 *
	 * @param MibData
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		AddOidMaster adder = new AddOidMaster();
		
		return adder.add(Oid , pri.getName());
	}
	
	/**
	 * OID情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param MibData
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
	Principal pri = m_context.getCallerPrincipal();
		
		ModifyOidMaster modifier = new ModifyOidMaster();
		
		return modifier.modify(Oid , pri.getName());
	}
	
	/**
	 * OID情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param mib
	 * @return
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		DeleteOidMaster eraser= new DeleteOidMaster();
		
		return eraser.delete(oid);
	}
	
	
	
}
