/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.composite;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.syslogng.action.GetLogFilterListTableDefine;
import com.clustercontrol.syslogng.action.LogManager;
import com.clustercontrol.syslogng.bean.LogFilterInfo;

/**
 * フィルタ情報一覧コンポジットクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class LogListComposite extends Composite {
	
	/** テーブルビューアー。 */
	private LogFilterInfoTableViewer tableViewer = null;
	
	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public LogListComposite(Composite parent, int style) {
		super(parent, style);
		
		this.initialize();
	}
	
	/**
	 * コンポジットを配置します。
	 * 
	 * @see com.clustercontrol.syslogng.action.GetLogFilterListTableDefine#get()
	 * @see com.clustercontrol.syslogng.action.LogManager#initialize()
	 * @see #update()
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		
		Table table = new Table(this, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);
		
		// テーブルビューアの作成
		this.tableViewer = new LogFilterInfoTableViewer(table);
		this.tableViewer.createTableColumn(GetLogFilterListTableDefine.get());
		
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		
		// syslog-ng一覧情報取得
		LogManager.getInstance().initialize();
		this.update();
	}
	
	/**
	 * このコンポジットが利用するテーブルビューアーを返します。
	 * 
	 * @return テーブルビューアー
	 */
	public LogFilterInfoTableViewer getTableViewer() {
		return this.tableViewer;
	}
	
	/**
	 * 現在選択されているアイテムを返します。
	 * <p>
	 * 選択されていない場合は、<code>null</code>を返します。
	 * 
	 * @return 選択アイテム
	 */
	public LogFilterInfo getFilterItem() {
		StructuredSelection selection = (StructuredSelection) this.tableViewer
		.getSelection();
		
		if (selection == null) {
			return null;
		} else {
			return (LogFilterInfo) selection.getFirstElement();
		}
	}
	
	/**
	 * コンポジットを更新します。<BR>
	 * フィルタ情報一覧を取得し、テーブルビューアーにセットします。
	 * 
	 * @see com.clustercontrol.syslogng.action.LogManager#get()
	 * @see com.clustercontrol.syslogng.composite.LogFilterInfoTableViewer
	 */
	public void update() {
		// データ取得
		Object[] list = LogManager.getInstance().get();
		
		// テーブル更新
		this.tableViewer.setInput(list);
	}
	
	/**
	 * 現在選択されているアイテム一覧を返します。
	 * 
	 * @return 選択アイテム （{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 */
	public ArrayList<LogFilterInfo> getSelectionData() {
		
		ArrayList<LogFilterInfo> data = new ArrayList<LogFilterInfo>();
		
		//選択されたアイテムを取得
		StructuredSelection selection = 
			(StructuredSelection)tableViewer.getSelection();
		List list = (List)selection.toList();
		
		if (list != null) {
			for(int index = 0; index < list.size(); index++){
				
				LogFilterInfo info = (LogFilterInfo)list.get(index);
				if (info instanceof LogFilterInfo) {
					data.add(info);
				}
			}
		}
		
		return data;
	}
}