// 
//    jSNMP - SNMPv1 & v2 Compliant Libraries for Java
//    Copyright (C) 2000  PlatformWorks, Inc.
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//   
// For more information contact: 
//  Brian Weaver      <weave@opennms.org>
//  http://www.opennms.org/
//
//  joeSNMP Licensing <joesnmp-license@lists.sourceforge.net>
//  http://sourceforge.net/projects/joesnmp/
//
// Modifications:
//
// $Id: SnmpAgentHandler.java,v 1.2 2004/11/10 23:48:01 andd Exp $
//
// 2004 Nov 10: Dimitris.Andreadis@jboss.org
//      Fixed javadoc comments
//
// 2000 Sep 08: Bob Snider <bsnider@seekone.com>
//      Adapted from SnmpTrapHandler
//

package org.opennms.protocols.snmp;

import java.net.InetAddress;

/**
 * <P>The SnmpAgentHandler interface is implemented by an object that
 * wishs to receive callbacks when an SNMP protocol data unit
 * is received from a manager.</P>
 *
 * @author  <a href="http://www.opennms.org/">OpenNMS</a>
 * @author  <a href="mailto:bsnider@seekone.com">Bob Snider</a>
 *
 */
public interface SnmpAgentHandler
{
  /**
   * <P>This method is defined to handle SNMP requests
   * that are received by the session. The parameters
   * allow the handler to determine the host, port, and
   * community string of the received PDU</P>
   *
   * @param session The SNMP session
   * @param manager The remote sender
   * @param port    The remote senders port
   * @param community  The community string
   * @param pdu     The SNMP pdu
   *
   */
  void snmpReceivedPdu(SnmpAgentSession session,
                       InetAddress      manager,
                       int              port,
                       SnmpOctetString  community,
                       SnmpPduPacket    pdu);
  
  /**
   * <P>This method is defined to handle SNMP Get requests
   * that are received by the session. The request has already
   * been validated by the system.  This routine will build a
   * response and pass it back to the caller.</P>
   *
   * @param pdu     The SNMP pdu
   * @param getNext The agent is requesting the lexically NEXT item after each
   *                    item in the pdu.
   *
   * @return SnmpPduRequest filled in with the proper response, or null if cannot process
   * NOTE: this might be changed to throw an exception.
   */
  SnmpPduRequest snmpReceivedGet(SnmpPduPacket pdu, boolean getNext);
  
  /**
   * <P>This method is defined to handle SNMP Set requests
   * that are received by the session. The request has already
   * been validated by the system.  This routine will build a
   * response and pass it back to the caller.</P>
   *
   * @param pdu     The SNMP pdu
   *
   * @return SnmpPduRequest filled in with the proper response, or null if cannot process
   * NOTE: this might be changed to throw an exception.
   */
  SnmpPduRequest snmpReceivedSet(SnmpPduPacket pdu);

  /**
   * <P>This method is invoked if an error occurs in 
   * the session. The error code that represents
   * the failure will be passed in the second parameter,
   * 'error'. The error codes can be found in the class
   * SnmpAgentSession class.</P>
   *
   * <P>If a particular PDU is part of the error condition
   * it will be passed in the third parameter, 'pdu'. The
   * pdu will be of the type SnmpPduRequest or SnmpPduTrap
   * object. The handler should use the "instanceof" operator
   * to determine which type the object is. Also, the object
   * may be null if the error condition is not associated
   * with a particular PDU.</P>
   *
   * @param session The SNMP Session
   * @param error   The error condition value.
   * @param ref     The PDU reference, or potentially null.
   *                It may also be an exception.
   */
  void SnmpAgentSessionError(SnmpAgentSession session, 
                             int              error,
                             Object           ref);
}