/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.composite.NumberVerifyListener;
import com.clustercontrol.composite.PositiveNumberVerifyListener;
import com.clustercontrol.composite.PositiveRealNumberVerifyListener;
import com.clustercontrol.composite.RealNumberVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.util.Messages;

/**
 * 数値監視の判定情報（重要度）コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>値取得の成功時（重要度毎に配置）</dd>
 *  <dd>　判定項目１ テキストボックス</dd>
 *  <dd>　判定項目１ 判定基準文字列ラベル（例：以上）</dd>
 *  <dd>　判定項目２ テキストボックス</dd>
 *  <dd>　判定項目２ 判定基準文字列ラベル（例：以下）</dd>
 *  <dd>値取得の失敗時</dd>
 *  <dd>　「重要度」 コンボボックス</dd>
 * </dl>
 *  
 * @version 2.1.0
 * @since 2.0.0
 */
public class NumericValueInfoComposite extends Composite {
	
	/** カラム数（タイトル）。 */
	public static final int WIDTH_TITLE = 5;
	
	/** カラム数（値）。 */
	public static final int WIDTH_VALUE = 2;
	
	/** 入力値の検証タイプ（整数）。 */
	public static final int INPUT_VERIFICATION_INTEGER_NUMBER = 0;
	/** 入力値の検証タイプ（実数）。 */
	public static final int INPUT_VERIFICATION_REAL_NUMBER = 1;
	/** 入力値の検証タイプ（正の整数）。 */
	public static final int INPUT_VERIFICATION_POSITIVE_INTEGER = 2;
	/** 入力値の検証タイプ（正の実数）。 */
	public static final int INPUT_VERIFICATION_POSITIVE_REAL = 3;
	
	/**
	 * 入力値の範囲検証フラグ。<BR>
	 * 判定項目１と２の入力値に対して範囲の検証を行う場合、<code> true </code>。
	 * <code> true </code>の場合、判定項目１の入力値 ＞ 判定項目２の入力値の場合、エラーダイアログが表示されます。
	 */
	private boolean m_rangeFlg = false;
	
	/** 項目名（判定項目１）。 */
	private String m_itemName1 = null;
	
	/** 項目名（判定項目２）。 */
	private String m_itemName2 = null;
	
	/** 判定基準文字列（判定項目１）。 */
	private String m_criterion1 = Messages.getString("greater");
	
	/** 判定基準文字列（判定項目２）。 */
	private String m_criterion2 = Messages.getString("less");
	
	/** 入力値の検証タイプ（判定項目１）。 */
	private int m_inputVerifyType1 = INPUT_VERIFICATION_INTEGER_NUMBER;
	
	/** 入力値の検証タイプ（判定項目２）。 */
	private int m_inputVerifyType2 = INPUT_VERIFICATION_INTEGER_NUMBER;
	
	/** 入力値（判定項目１：通知）。 */
	private Double m_value1Info = null;
	
	/** 入力値（判定項目２：通知）。 */
	private Double m_value2Info = null;
	
	/** 入力値（判定項目１：警告）。 */
	private Double m_value1Warn = null;
	
	/** 入力値（判定項目２：警告）。 */
	private Double m_value2Warn = null;
	
	/** 判定値（判定項目１：通知） テキストボックス。 */
	private Text m_textValue1Info = null;
	
	/** 判定値（判定項目２：通知） テキストボックス。 */
	private Text m_textValue2Info = null;
	
	/** 判定値（判定項目１：警告） テキストボックス。 */
	private Text m_textValue1Warn = null;
	
	/** 判定値（判定項目２：警告） テキストボックス。 */
	private Text m_textValue2Warn = null;
	
	/** 値取得の失敗時の重要度 コンボボックス。 */
	private Combo m_comboFailurePriority = null;
	
	/** 判定項目１用VerifyListener */
	private VerifyListener m_verifyListener1 = null;
	
	/** 判定項目２用VerifyListener */
	private VerifyListener m_verifyListener2 = null;
	
	public Combo getComboFailurePriority(){
		return this.m_comboFailurePriority;
	}
	
	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 判定基準文字列，入力値の検証タイプ 及び 入力値の初期値はデフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, int style, boolean rangeFlg, String item1, String item2) {
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 引数で指定された入力値の検証タイプで、全ての入力値テキストボックスの入力検証を行います。
	 * 判定基準文字列 及び 入力値の初期値はデフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param inputVerifyType 入力値の検証タイプ（判定項目１,２）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2, 
			int inputVerifyType) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_inputVerifyType1 = inputVerifyType;
		m_inputVerifyType2 = inputVerifyType;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 各判定項目ごとに引数で指定された入力検証タイプで、各入力値テキストボックスの入力検証を行います。
	 * 判定基準文字列 及び 入力値の初期値はデフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param inputVerifyType1 入力値の検証タイプ（判定項目１）
	 * @param inputVerifyType2 入力値の検証タイプ（判定項目２）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2, 
			int inputVerifyType1,
			int inputVerifyType2) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_inputVerifyType1 = inputVerifyType1;
		m_inputVerifyType2 = inputVerifyType2;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 各判定項目ごとに引数で指定された入力検証タイプで、各入力値テキストボックスの入力検証を行います。
	 * 入力値の初期値は、デフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param criterion1 判定基準文字列（判定項目１）
	 * @param criterion2 判定基準文字列（判定項目２）
	 * @param inputVerifyType1 入力値の検証タイプ（判定項目１）
	 * @param inputVerifyType2 入力値の検証タイプ（判定項目２）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2,
			String criterion1,
			String criterion2,
			int inputVerifyType1,
			int inputVerifyType2) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_criterion1 = criterion1;
		m_criterion2 = criterion2;
		m_inputVerifyType1 = inputVerifyType1;
		m_inputVerifyType2 = inputVerifyType2;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 引数で指定された入力値の検証タイプで、全ての入力値テキストボックスの入力検証を行います。
	 * 判定基準文字列は、デフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param inputVerifyType 入力値の検証タイプ（判定項目１,２）
	 * @param inputValue1 入力値の初期値（判定項目１）
	 * @param inputValue2 入力値の初期値（判定項目２）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2, 
			int inputVerifyType, 
			Double inputValue1, 
			Double inputValue2) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_inputVerifyType1 = inputVerifyType;
		m_inputVerifyType2 = inputVerifyType;
		m_value1Info = inputValue1;
		m_value1Warn = inputValue1;
		m_value2Info = inputValue2;
		m_value2Warn = inputValue2;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 各判定項目ごとに引数で指定された入力検証タイプで、各入力値テキストボックスの入力検証を行います。
	 * 判定基準文字列は、デフォルト値が使用されます。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param inputVerifyType1 入力値の検証タイプ（判定項目１）
	 * @param inputVerifyType2 入力値の検証タイプ（判定項目２）
	 * @param inputValue1Info 入力値の初期値（判定項目１：通知）
	 * @param inputValue2Info 入力値の初期値（判定項目２：通知）
	 * @param inputValue1Warn 入力値の初期値（判定項目１：警告）
	 * @param inputValue2Warn 入力値の初期値（判定項目２：警告）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2, 
			int inputVerifyType1, 
			int inputVerifyType2,
			Double inputValue1Info,
			Double inputValue2Info,
			Double inputValue1Warn,
			Double inputValue2Warn) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_inputVerifyType1 = inputVerifyType1;
		m_inputVerifyType2 = inputVerifyType2;
		m_value1Info = inputValue1Info;
		m_value1Warn = inputValue1Warn;
		m_value2Info = inputValue2Info;
		m_value2Warn = inputValue2Warn;
		
		this.initialize();
	}
	
	/**
	 * インスタンスを返します。
	 * <p> 
	 * 初期処理を呼び出し、コンポジットを配置します。<BR>
	 * 各判定項目ごとに引数で指定された入力検証タイプで、各入力値テキストボックスの入力検証を行います。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param rangeFlg 範囲検証フラグ
	 * @param item1 項目名（判定項目１）
	 * @param item2 項目名（判定項目２）
	 * @param criterion1 判定基準文字列（判定項目１）
	 * @param criterion2 判定基準文字列（判定項目２）
	 * @param inputVerifyType1 入力値の検証タイプ（判定項目１）
	 * @param inputVerifyType2 入力値の検証タイプ（判定項目２）
	 * @param inputValue1Info 入力値の初期値（判定項目１：通知）
	 * @param inputValue2Info 入力値の初期値（判定項目２：通知）
	 * @param inputValue1Warn 入力値の初期値（判定項目１：警告）
	 * @param inputValue2Warn 入力値の初期値（判定項目２：警告）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see #INPUT_VERIFICATION_INTEGER_NUMBER
	 * @see #INPUT_VERIFICATION_POSITIVE_INTEGER
	 * @see #INPUT_VERIFICATION_POSITIVE_REAL
	 * @see #INPUT_VERIFICATION_REAL_NUMBER
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NumericValueInfoComposite(Composite parent, 
			int style, 
			boolean rangeFlg, 
			String item1, 
			String item2,
			String criterion1,
			String criterion2,
			int inputVerifyType1, 
			int inputVerifyType2,
			Double inputValue1Info,
			Double inputValue2Info,
			Double inputValue1Warn,
			Double inputValue2Warn) {
		
		super(parent, style);
		
		m_rangeFlg = rangeFlg;
		m_itemName1 = item1;
		m_itemName2 = item2;
		m_criterion1 = criterion1;
		m_criterion2 = criterion2;
		m_inputVerifyType1 = inputVerifyType1;
		m_inputVerifyType2 = inputVerifyType2;
		m_value1Info = inputValue1Info;
		m_value1Warn = inputValue1Warn;
		m_value2Info = inputValue2Info;
		m_value2Warn = inputValue2Warn;
		
		this.initialize();
	}
	
	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);
		
		// ラベル
		// 値取得の成功時
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("successful.select.value") + " : ");
		
		// 重要度
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority"));
		
		// 閾値の下限
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE*2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_itemName1);
		
		// 閾値の上限
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE*2;
		gridData.horizontalAlignment = GridData.BEGINNING;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_itemName2);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// テキストボックス入力検証
		if(m_inputVerifyType1 == INPUT_VERIFICATION_REAL_NUMBER){
			m_verifyListener1 = new RealNumberVerifyListener();
		}
		else if(m_inputVerifyType1 == INPUT_VERIFICATION_POSITIVE_INTEGER){
			m_verifyListener1 = new PositiveNumberVerifyListener();
		}
		else if(m_inputVerifyType1 == INPUT_VERIFICATION_POSITIVE_REAL){
			m_verifyListener1 = new PositiveRealNumberVerifyListener();
		}
		else{
			m_verifyListener1 = new NumberVerifyListener();
		}
		
		// テキストボックス入力検証
		if(m_inputVerifyType2 == INPUT_VERIFICATION_REAL_NUMBER){
			m_verifyListener2 = new RealNumberVerifyListener();
		}
		else if(m_inputVerifyType2 == INPUT_VERIFICATION_POSITIVE_INTEGER){
			m_verifyListener2 = new PositiveNumberVerifyListener();
		}
		else if(m_inputVerifyType2 == INPUT_VERIFICATION_POSITIVE_REAL){
			m_verifyListener2 = new PositiveRealNumberVerifyListener();
		}
		else{
			m_verifyListener2 = new NumberVerifyListener();
		}
		
		// 重要度：通知
		label = this.getLabelPriority(this, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 閾値の下限
		this.m_textValue1Info = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textValue1Info.addVerifyListener(m_verifyListener1);
		this.m_textValue1Info.setLayoutData(gridData);
		
		// 以上
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_criterion1);
		
		// 閾値の上限
		this.m_textValue2Info = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textValue2Info.addVerifyListener(m_verifyListener2);
		this.m_textValue2Info.setLayoutData(gridData);
		
		// 以下
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_criterion2);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 重要度：警告
		label = this.getLabelPriority(this, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 閾値の下限
		this.m_textValue1Warn = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textValue1Warn.addVerifyListener(m_verifyListener1);
		this.m_textValue1Warn.setLayoutData(gridData);
		
		// 以上
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_criterion1);
		
		// 閾値の上限
		this.m_textValue2Warn = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textValue2Warn.addVerifyListener(m_verifyListener2);
		this.m_textValue2Warn.setLayoutData(gridData);
		
		// 以下
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(m_criterion2);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 重要度：異常
		label = this.getLabelPriority(this, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// （通知・警告以外）
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE*2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("other.priority"));
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 6;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 値取得の失敗時
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("failure.select.value") + " : ");
		
		// 重要度
		this.m_comboFailurePriority = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboFailurePriority.setLayoutData(gridData);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_CRITICAL);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_WARNING);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_INFO);
		this.m_comboFailurePriority.add(PriorityConstant.STRING_UNKNOWN);
		this.m_comboFailurePriority.setText(PriorityConstant.STRING_WARNING);
		
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
	}
	
	/**
	 * 引数で指定された監視情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 */
	public void setInputData(MonitorInfo info) {
		
		if(info != null){
			
			ArrayList list = info.getJudgementInfo();
			if(list != null){
				for(int index=0; index<list.size(); index++){
					MonitorNumericValueInfo numericValueInfo = (MonitorNumericValueInfo)list.get(index);
					if(numericValueInfo != null){
						if(PriorityConstant.TYPE_INFO ==  numericValueInfo.getPriority()){
							String lower = "";
							String upper = "";
							if(m_inputVerifyType1  == INPUT_VERIFICATION_INTEGER_NUMBER || 
									m_inputVerifyType1  == INPUT_VERIFICATION_POSITIVE_INTEGER){
								lower = Long.toString(Double.valueOf(numericValueInfo.getThresholdLowerLimit()).longValue());
							}
							else{
								lower = Double.toString(numericValueInfo.getThresholdLowerLimit());
							}
							if(m_inputVerifyType2  == INPUT_VERIFICATION_INTEGER_NUMBER || 
									m_inputVerifyType2  == INPUT_VERIFICATION_POSITIVE_INTEGER){
								upper = Long.toString(Double.valueOf(numericValueInfo.getThresholdUpperLimit()).longValue());
							}
							else{
								upper = Double.toString(numericValueInfo.getThresholdUpperLimit());
							}
							this.m_textValue1Info.setText(lower);
							this.m_textValue2Info.setText(upper);
						}
						else if(PriorityConstant.TYPE_WARNING ==  numericValueInfo.getPriority()){
							String lower = "";
							String upper = "";
							if(m_inputVerifyType1  == INPUT_VERIFICATION_INTEGER_NUMBER || 
									m_inputVerifyType1  == INPUT_VERIFICATION_POSITIVE_INTEGER){
								lower = Long.toString(Double.valueOf(numericValueInfo.getThresholdLowerLimit()).longValue());
							}
							else{
								lower = Double.toString(numericValueInfo.getThresholdLowerLimit());
							}
							if(m_inputVerifyType2  == INPUT_VERIFICATION_INTEGER_NUMBER || 
									m_inputVerifyType2  == INPUT_VERIFICATION_POSITIVE_INTEGER){
								upper = Long.toString(Double.valueOf(numericValueInfo.getThresholdUpperLimit()).longValue());
							}
							else{
								upper = Double.toString(numericValueInfo.getThresholdUpperLimit());
							}
							this.m_textValue1Warn.setText(lower);
							this.m_textValue2Warn.setText(upper);
						}
					}
				}
			}
			this.m_comboFailurePriority.setText(PriorityConstant.typeToString(info.getFailurePriority()));
		}
	}
	
	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see #setValidateResult(String, String)
	 */
	public ValidateResult createInputData(MonitorInfo info) {
		
		ArrayList<MonitorNumericValueInfo> valueList = new ArrayList<MonitorNumericValueInfo>();
		
		String lowerText = null;
		String upperText = null;
		Double lower = null;
		Double upper = null;
		
		// 重要度：通知
		MonitorNumericValueInfo valueInfo = getDefaultValueInfo(info, PriorityConstant.TYPE_INFO);
		
		lowerText = this.m_textValue1Info.getText();
		upperText = this.m_textValue2Info.getText();
		
		if (lowerText != null && !"".equals(lowerText.trim())) {
			try{
				lower = Double.valueOf(lowerText);
				if(m_value1Info != null && m_value2Info != null){
					if(lower < m_value1Info || lower > m_value2Info){
						String[] args = { m_itemName1, m_value1Info.toString(), m_value2Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.19", args));
					}
				}
				else if(m_value1Info != null && m_value2Info == null){
					if(lower < m_value1Info){
						String[] args = { m_itemName1, m_value1Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.20", args));
					}
				}
				else if(m_value1Info == null && m_value2Info != null){
					if(lower > m_value2Info){
						String[] args = { m_itemName1, m_value2Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.21", args));
					}
				}
				valueInfo.setThresholdLowerLimit(lower);
			}
			catch(NumberFormatException e){
				String[] args = { m_itemName1 };
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.3", args));
			}
		}
		else{
			String[] args = { m_itemName1 };
			return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.3", args));
		}
		if (upperText != null && !"".equals(upperText.trim())) {
			try{
				upper = Double.valueOf(upperText);
				if(m_value1Warn != null && m_value2Warn != null){
					if(upper < m_value1Warn || upper > m_value2Warn){
						String[] args = { m_itemName2, m_value1Warn.toString(), m_value2Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.19", args));
					}
				}
				else if(m_value1Warn != null && m_value2Warn == null){
					if(upper < m_value1Warn){
						String[] args = { m_itemName2, m_value1Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.20", args));
					}
				}
				else if(m_value1Warn == null && m_value2Warn != null){
					if(upper > m_value2Warn){
						String[] args = { m_itemName2, m_value2Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.21", args));
					}
				}
				valueInfo.setThresholdUpperLimit(upper);
			}
			catch(NumberFormatException e){
				String[] args = { m_itemName2 };
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.4", args));
			}
		}
		else{
			String[] args = { m_itemName2 };
			return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.4", args));
		}
		if(m_rangeFlg){
			if(upper.doubleValue() < lower.doubleValue()){
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.7"));
			}
		}
		valueList.add(valueInfo);
		
		// 重要度：警告
		MonitorNumericValueInfo valueWarn = getDefaultValueInfo(info, PriorityConstant.TYPE_WARNING);
		
		lowerText = this.m_textValue1Warn.getText();
		upperText = this.m_textValue2Warn.getText();
		
		if (lowerText != null
				&& !"".equals(lowerText.trim())) {
			try{
				lower = Double.valueOf(lowerText);
				if(m_value1Info != null && m_value2Info != null){
					if(lower < m_value1Info || lower > m_value2Info){
						String[] args = { m_itemName1, m_value1Info.toString(), m_value2Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.22", args));
					}
				}
				else if(m_value1Info != null && m_value2Info == null){
					if(lower < m_value1Info){
						String[] args = { m_itemName1, m_value1Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.23", args));
					}
				}
				else if(m_value1Info == null && m_value2Info != null){
					if(lower > m_value2Info){
						String[] args = { m_itemName1, m_value2Info.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.24", args));
					}
				}
				valueWarn.setThresholdLowerLimit(lower);	
			}
			catch(NumberFormatException e){
				String[] args = { m_itemName1 };
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.5", args));
			}
		}
		else{
			String[] args = { m_itemName1 };
			return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.5", args));
		}
		if (this.m_textValue2Warn.getText() != null
				&& !"".equals((this.m_textValue2Warn.getText()).trim())) {
			try{
				upper = Double.valueOf(upperText);
				if(m_value1Warn != null && m_value2Warn != null){
					if(upper < m_value1Warn || upper > m_value2Warn){
						String[] args = { m_itemName2, m_value1Warn.toString(), m_value2Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.22", args));
					}
				}
				else if(m_value1Warn != null && m_value2Warn == null){
					if(upper < m_value1Warn){
						String[] args = { m_itemName2, m_value1Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.23", args));
					}
				}
				else if(m_value1Warn == null && m_value2Warn != null){
					if(upper > m_value2Warn){
						String[] args = { m_itemName2, m_value2Warn.toString() };
						return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.24", args));
					}
				}
				valueWarn.setThresholdUpperLimit(upper);
			}
			catch(NumberFormatException e){
				String[] args = { m_itemName2 };
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.6", args));
			}
		}
		else{
			String[] args = { m_itemName2 };
			return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.6", args));
		}
		if(m_rangeFlg){
			if(upper.doubleValue() < lower.doubleValue()){
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.8"));
			}
		}
		valueList.add(valueWarn);
		
		// 重要度：危険
		MonitorNumericValueInfo valueCritical = getDefaultValueInfo(info, PriorityConstant.TYPE_CRITICAL);
		valueCritical.setThresholdLowerLimit(Double.valueOf(0));
		valueCritical.setThresholdUpperLimit(Double.valueOf(0));
		valueList.add(valueCritical);
		
		// 重要度：不明
		MonitorNumericValueInfo valueUnknown = getDefaultValueInfo(info, PriorityConstant.TYPE_UNKNOWN);
		valueUnknown.setThresholdLowerLimit(Double.valueOf(0));
		valueUnknown.setThresholdUpperLimit(Double.valueOf(0));
		valueList.add(valueUnknown);
		
		info.setJudgementInfo(valueList);
		
		// 値取得の失敗時
		info.setFailurePriority(PriorityConstant.stringToType(this.m_comboFailurePriority.getText()));
		
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
		this.m_textValue1Info.setEnabled(enabled);
		this.m_textValue2Info.setEnabled(enabled);
		this.m_textValue1Warn.setEnabled(enabled);
		this.m_textValue2Warn.setEnabled(enabled);
		this.m_comboFailurePriority.setEnabled(enabled);
	}
	
    /**
     * 重要度のラベルを返します。
     * 
     * @param parent 親のコンポジット
     * @param text ラベルに表示するテキスト
     * @param background ラベルの背景色
     * @return 生成されたラベル
     */
	protected Label getLabelPriority(Composite parent,
			String text,
			Color background
	) {
		
		// ラベル（重要度）
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(text + " : ");
		label.setBackground(background);
		
		return label;
	}
	
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
	
	/**
	 * 初期値を設定した数値監視の判定情報を返します。<BR>
	 * 監視対象ID，監視項目ID 及び 重要度をセットした数値監視の判定情報を返します。
	 * 
	 * @param info 監視情報
	 * @param priority 重要度
	 * @return 数値監視の判定情報
	 */
	protected MonitorNumericValueInfo getDefaultValueInfo(MonitorInfo info, int priority) {
		
		MonitorNumericValueInfo value = new MonitorNumericValueInfo();
		value.setMonitorId(info.getMonitorId());
		value.setMonitorTypeId(info.getMonitorTypeId());
		value.setPriority(Integer.valueOf(priority));
		
		return value;
	}
	
	/**
	 * 判定項目１の上下限を設定します。
	 * 
	 * @param low 下限（IntegerかDoubleで指定）
	 * @param high 上限（IntegerかDoubleで指定）
	 */
	public void setInputRange1(int low, int high) {
		//インスタンス別に上下限を設定
		if(this.m_verifyListener1 instanceof PositiveNumberVerifyListener){
			((PositiveNumberVerifyListener)this.m_verifyListener1).setLow(Integer.valueOf(low));
			((PositiveNumberVerifyListener)this.m_verifyListener1).setHigh(Integer.valueOf(high));
		}
		else if(this.m_verifyListener1 instanceof NumberVerifyListener){
			((NumberVerifyListener)this.m_verifyListener1).setLow(Integer.valueOf(low));
			((NumberVerifyListener)this.m_verifyListener1).setHigh(Integer.valueOf(high));
		}
		else if(this.m_verifyListener1 instanceof PositiveRealNumberVerifyListener){
			((PositiveRealNumberVerifyListener)this.m_verifyListener1).setLow(Double.valueOf(low));
			((PositiveRealNumberVerifyListener)this.m_verifyListener1).setHigh(Double.valueOf(high));
		}
		else if(this.m_verifyListener1 instanceof RealNumberVerifyListener){
			((RealNumberVerifyListener)this.m_verifyListener1).setLow(Double.valueOf(low));
			((RealNumberVerifyListener)this.m_verifyListener1).setHigh(Double.valueOf(high));
		}
	}
	
	/**
	 * 判定項目１の上下限を設定します。
	 * 
	 * @param low 下限（IntegerかDoubleで指定）
	 * @param high 上限（IntegerかDoubleで指定）
	 */
	public void setInputRange1(double low, double high) {
		//インスタンス別に上下限を設定
		if(this.m_verifyListener1 instanceof PositiveRealNumberVerifyListener){
			((PositiveRealNumberVerifyListener)this.m_verifyListener1).setLow(Double.valueOf(low));
			((PositiveRealNumberVerifyListener)this.m_verifyListener1).setHigh(Double.valueOf(high));
		}
		else if(this.m_verifyListener1 instanceof RealNumberVerifyListener){
			((RealNumberVerifyListener)this.m_verifyListener1).setLow(Double.valueOf(low));
			((RealNumberVerifyListener)this.m_verifyListener1).setHigh(Double.valueOf(high));
		}
	}
	
	/**
	 * 判定項目２の上下限を設定します。
	 * 
	 * @param low 下限（IntegerかDoubleで指定）
	 * @param high 上限（IntegerかDoubleで指定）
	 */
	public void setInputRange2(int low, int high) {
		//インスタンス別に上下限を設定
		if(this.m_verifyListener2 instanceof PositiveNumberVerifyListener){
			((PositiveNumberVerifyListener)this.m_verifyListener2).setLow(Integer.valueOf(low));
			((PositiveNumberVerifyListener)this.m_verifyListener2).setHigh(Integer.valueOf(high));
		}
		else if(this.m_verifyListener2 instanceof NumberVerifyListener){
			((NumberVerifyListener)this.m_verifyListener2).setLow(Integer.valueOf(low));
			((NumberVerifyListener)this.m_verifyListener2).setHigh(Integer.valueOf(high));
		}
		else if(this.m_verifyListener2 instanceof PositiveRealNumberVerifyListener){
			((PositiveRealNumberVerifyListener)this.m_verifyListener2).setLow(Double.valueOf(low));
			((PositiveRealNumberVerifyListener)this.m_verifyListener2).setHigh(Double.valueOf(high));
		}
		else if(this.m_verifyListener2 instanceof RealNumberVerifyListener){
			((RealNumberVerifyListener)this.m_verifyListener2).setLow(Double.valueOf(low));
			((RealNumberVerifyListener)this.m_verifyListener2).setHigh(Double.valueOf(high));
		}
	}
	
	/**
	 * 判定項目２の上下限を設定します。
	 * 
	 * @param low 下限（IntegerかDoubleで指定）
	 * @param high 上限（IntegerかDoubleで指定）
	 */
	public void setInputRange2(double low, double high) {
		//インスタンス別に上下限を設定
		if(this.m_verifyListener2 instanceof PositiveRealNumberVerifyListener){
			((PositiveRealNumberVerifyListener)this.m_verifyListener2).setLow(Double.valueOf(low));
			((PositiveRealNumberVerifyListener)this.m_verifyListener2).setHigh(Double.valueOf(high));
		}
		else if(this.m_verifyListener2 instanceof RealNumberVerifyListener){
			((RealNumberVerifyListener)this.m_verifyListener2).setLow(Double.valueOf(low));
			((RealNumberVerifyListener)this.m_verifyListener2).setHigh(Double.valueOf(high));
		}
	}
}